import { parser, trackVariables } from 'lezer-feel';
import { LRLanguage, indentNodeProp, delimitedIndent, continuedIndent, foldNodeProp, foldInside, LanguageSupport } from '@codemirror/language';
import { snippetCompletion, ifNotIn, completeFromList } from '@codemirror/autocomplete';

// / A collection of FEEL-related
// / [snippets](#autocomplete.snippet).
const snippets = [
    /*@__PURE__*/snippetCompletion('function(${params}) ${body}', {
        label: 'function',
        detail: 'definition',
        type: 'keyword'
    }),
    /*@__PURE__*/snippetCompletion('for ${var} in ${collection} return ${value}', {
        label: 'for',
        detail: 'expression',
        type: 'keyword'
    }),
    /*@__PURE__*/snippetCompletion('every ${var} in ${collection} satisfies ${condition}', {
        label: 'every',
        detail: 'quantified expression',
        type: 'keyword'
    }),
    /*@__PURE__*/snippetCompletion('some ${var} in ${collection} satisfies ${condition}', {
        label: 'some',
        detail: 'quantified expression',
        type: 'keyword'
    }),
    /*@__PURE__*/snippetCompletion('if ${condition} then ${value} else ${other value}', {
        label: 'if',
        detail: 'block',
        type: 'keyword'
    })
];

// / A language provider based on the [Lezer FEEL
// / parser](https://github.com/nikku/lezer-feel), extended with
// / highlighting and indentation information.
const feelLanguage = /*@__PURE__*/LRLanguage.define({
    parser: /*@__PURE__*/parser.configure({
        props: [
            /*@__PURE__*/indentNodeProp.add({
                'Context': /*@__PURE__*/delimitedIndent({
                    closing: '}'
                }),
                'List FilterExpression': /*@__PURE__*/delimitedIndent({
                    closing: ']'
                }),
                'ParenthesizedExpression FunctionInvocation': /*@__PURE__*/continuedIndent({
                    except: /^\s*\)/
                }),
                'ForExpression QuantifiedExpression IfExpression': /*@__PURE__*/continuedIndent({
                    except: /^\s*(then|else|return|satisfies)\b/
                }),
                'FunctionDefinition': /*@__PURE__*/continuedIndent({
                    except: /^\s*(\(|\))/
                })
            }),
            /*@__PURE__*/foldNodeProp.add({
                Context: foldInside,
                List: foldInside,
                ParenthesizedExpression: foldInside,
                FunctionDefinition(node) {
                    const last = node.getChild(')');
                    if (!last)
                        return null;
                    return {
                        from: last.to,
                        to: node.to
                    };
                }
            })
        ]
    }),
    languageData: {
        indentOnInput: /^\s*(\)|\}|\]|then|else|return|satisfies)$/,
        commentTokens: {
            line: '//',
            block: {
                open: '/*',
                close: '*/'
            }
        }
    }
});
// / A language provider for TypeScript.
const unaryTestsLanguage = /*@__PURE__*/feelLanguage.configure({ top: 'UnaryTests' });
// / Language provider for JSX.
const expressionLanguage = /*@__PURE__*/feelLanguage.configure({ top: 'Expression' });
const keywords = /*@__PURE__*/'return satisfies then in'.split(' ').map(kw => ({ label: kw, type: 'keyword' }));
const dontComplete = [
    'StringLiteral', 'Name',
    'LineComment', 'BlockComment'
];
// / FEEL support. Includes [snippet](#lang-feel.snippets)
// / completion.
function feel(config = {}) {
    const lang = config.dialect === 'unaryTests' ? unaryTestsLanguage : expressionLanguage;
    const contextualLang = lang.configure({
        contextTracker: trackVariables(config.context)
    });
    return new LanguageSupport(contextualLang, [
        feelLanguage.data.of({
            autocomplete: ifNotIn(dontComplete, completeFromList(snippets.concat(keywords)))
        })
    ]);
}

export { expressionLanguage, feel, feelLanguage, snippets, unaryTestsLanguage };
