var lezerFeel = require('lezer-feel');
var language = require('@codemirror/language');
var autocomplete = require('@codemirror/autocomplete');

/**
 * A collection of FEEL-related [snippets](#autocomplete.snippet).
 */
const snippets = [autocomplete.snippetCompletion('function(${params}) ${body}', {
  label: 'function',
  detail: 'definition',
  type: 'keyword'
}), autocomplete.snippetCompletion('for ${var} in ${collection} return ${value}', {
  label: 'for',
  detail: 'expression',
  type: 'keyword'
}), autocomplete.snippetCompletion('every ${var} in ${collection} satisfies ${condition}', {
  label: 'every',
  detail: 'quantified expression',
  type: 'keyword'
}), autocomplete.snippetCompletion('some ${var} in ${collection} satisfies ${condition}', {
  label: 'some',
  detail: 'quantified expression',
  type: 'keyword'
}), autocomplete.snippetCompletion('if ${condition} then ${value} else ${other value}', {
  label: 'if',
  detail: 'block',
  type: 'keyword'
}), autocomplete.snippetCompletion('{ ${key}: ${value} }', {
  label: 'context',
  detail: 'block',
  type: 'keyword'
}), autocomplete.snippetCompletion('null', {
  label: 'null',
  detail: 'literal',
  type: 'keyword'
}), autocomplete.snippetCompletion('true', {
  label: 'true',
  detail: 'literal',
  type: 'keyword'
}), autocomplete.snippetCompletion('false', {
  label: 'false',
  detail: 'literal',
  type: 'keyword'
})];

function contextualKeyword(options) {
  const {
    context: nodes,
    after,
    before,
    keyword
  } = options;
  return ifInside({
    nodes,
    before,
    after,
    keyword
  }, autocomplete.completeFromList([{
    label: keyword,
    type: 'keyword',
    boost: 10
  }]));
}
const keywordCompletions = [contextualKeyword({
  context: 'InExpression',
  keyword: 'in'
}), contextualKeyword({
  context: 'IfExpression',
  keyword: 'then',
  after: 'if',
  before: 'else'
}), contextualKeyword({
  context: 'IfExpression',
  keyword: 'else',
  after: 'then'
}), contextualKeyword({
  context: 'QuantifiedExpression',
  keyword: 'satisfies'
}), contextualKeyword({
  context: 'ForExpression',
  after: 'InExpressions',
  keyword: 'return'
})];
const dontComplete = ['StringLiteral', 'Identifier', 'LineComment', 'BlockComment', 'PathExpression', 'Context', 'Key', 'ParameterName'];
const doComplete = ['Expr', 'ContextEntry'];
function ifExpression(completionSource) {
  const allNodes = [...dontComplete, ...doComplete];
  return context => {
    const {
      state,
      pos
    } = context;
    const match = matchUp(language.syntaxTree(state).resolveInner(pos, -1), allNodes);
    if (match) {
      const [_, name] = match;
      if (dontComplete.includes(name)) {
        return null;
      }
    }
    return completionSource(context);
  };
}
function snippetCompletion(snippets) {
  return ifExpression(autocomplete.completeFromList(snippets.map(s => ({
    ...s,
    type: 'text'
  }))));
}
function matchLeft(node, position, nodes) {
  return matchChildren(node, position, nodes, -1);
}
function matchRight(node, position, nodes) {
  return matchChildren(node, position, nodes, 1);
}
function matchChildren(node, position, nodes, direction) {
  let child = node[direction > 0 ? 'childAfter' : 'childBefore'](position);
  while (child) {
    if (nodes.includes(child.name)) {
      return child;
    }
    if (child.type.isError && child.firstChild) {
      if (nodes.includes(child.firstChild.name)) {
        return child.firstChild;
      }
    }
    child = child[direction > 0 ? 'nextSibling' : 'prevSibling'];
  }
  return null;
}
function matchUp(node, nodeNames) {
  if (!Array.isArray(nodeNames)) {
    nodeNames = [nodeNames];
  }
  for (; node; node = node.parent) {
    const nodeType = node.type;
    const matchedName = nodeNames.find(name => name && nodeType.is(name));
    if (matchedName) {
      return [node, matchedName];
    }
    if (nodeType.isTop) {
      break;
    }
  }
  return null;
}
function ifInside(options, source) {
  const {
    nodes,
    before,
    after,
    keyword
  } = options;
  return context => {
    const {
      state,
      pos
    } = context;
    const match = matchUp(language.syntaxTree(state).resolveInner(pos, -1), nodes);
    if (!match) {
      return null;
    }
    const [node] = match;
    if (matchLeft(node, pos, [keyword, before])) {
      return null;
    }
    if (matchRight(node, pos, [keyword, after])) {
      return null;
    }
    if (after && !matchLeft(node, pos, [after])) {
      return null;
    }
    return source(context);
  };
}

/**
 * A FEEL language provider based on the
 * [Lezer FEEL parser](https://github.com/nikku/lezer-feel),
 * extended with highlighting and indentation information.
 */
const feelLanguage = language.LRLanguage.define({
  parser: lezerFeel.parser.configure({
    props: [language.indentNodeProp.add({
      'Context': language.delimitedIndent({
        closing: '}'
      }),
      'List FilterExpression': language.delimitedIndent({
        closing: ']'
      }),
      'ParenthesizedExpression FunctionInvocation': language.continuedIndent({
        except: /^\s*\)/
      }),
      'ForExpression QuantifiedExpression IfExpression': language.continuedIndent({
        except: /^\s*(then|else|return|satisfies)\b/
      }),
      'FunctionDefinition': language.continuedIndent({
        except: /^\s*(\(|\))/
      })
    }), language.foldNodeProp.add({
      Context: language.foldInside,
      List: language.foldInside,
      ParenthesizedExpression: language.foldInside,
      FunctionDefinition(node) {
        const last = node.getChild(')');
        if (!last) return null;
        return {
          from: last.to,
          to: node.to
        };
      }
    })]
  }),
  languageData: {
    indentOnInput: /^\s*(\)|\}|\]|then|else|return|satisfies)$/,
    commentTokens: {
      line: '//',
      block: {
        open: '/*',
        close: '*/'
      }
    }
  }
});
/**
 * A language provider for FEEL Unary Tests
 */
const unaryTestsLanguage = feelLanguage.configure({
  top: 'UnaryTests'
}, 'FEEL unary tests');
/**
 * Language provider for FEEL Expression
 */
const expressionLanguage = feelLanguage.configure({
  top: 'Expression'
}, 'FEEL expression');
/**
 * Feel language support for CodeMirror.
 *
 * Includes [snippet](#lang-feel.snippets)
 */
function feel(config = {}) {
  const language$1 = config.dialect === 'unaryTests' ? unaryTestsLanguage : expressionLanguage;
  const dialect = config.parserDialect;
  const contextTracker = lezerFeel.trackVariables(config.context);
  const contextualLang = language$1.configure({
    contextTracker,
    dialect
  });
  const completions = config.completions || [snippetCompletion(snippets), keywordCompletions].flat();
  return new language.LanguageSupport(contextualLang, [...completions.map(autocomplete => contextualLang.data.of({
    autocomplete
  }))]);
}

exports.contextualKeyword = contextualKeyword;
exports.doComplete = doComplete;
exports.dontComplete = dontComplete;
exports.expressionLanguage = expressionLanguage;
exports.feel = feel;
exports.feelLanguage = feelLanguage;
exports.ifExpression = ifExpression;
exports.ifInside = ifInside;
exports.keywordCompletions = keywordCompletions;
exports.matchChildren = matchChildren;
exports.matchLeft = matchLeft;
exports.matchRight = matchRight;
exports.snippetCompletion = snippetCompletion;
exports.snippets = snippets;
exports.unaryTestsLanguage = unaryTestsLanguage;
//# sourceMappingURL=index.cjs.map
