import { parser, trackVariables } from 'lezer-feel';
import { syntaxTree, LRLanguage, indentNodeProp, delimitedIndent, continuedIndent, foldNodeProp, foldInside, LanguageSupport } from '@codemirror/language';
import { snippetCompletion as snippetCompletion$1, completeFromList } from '@codemirror/autocomplete';

/**
 * A collection of FEEL-related [snippets](#autocomplete.snippet).
 */
const snippets = [snippetCompletion$1('function(${params}) ${body}', {
  label: 'function',
  detail: 'definition',
  type: 'keyword'
}), snippetCompletion$1('for ${var} in ${collection} return ${value}', {
  label: 'for',
  detail: 'expression',
  type: 'keyword'
}), snippetCompletion$1('every ${var} in ${collection} satisfies ${condition}', {
  label: 'every',
  detail: 'quantified expression',
  type: 'keyword'
}), snippetCompletion$1('some ${var} in ${collection} satisfies ${condition}', {
  label: 'some',
  detail: 'quantified expression',
  type: 'keyword'
}), snippetCompletion$1('if ${condition} then ${value} else ${other value}', {
  label: 'if',
  detail: 'block',
  type: 'keyword'
}), snippetCompletion$1('{ ${key}: ${value} }', {
  label: 'context',
  detail: 'block',
  type: 'keyword'
}), snippetCompletion$1('null', {
  label: 'null',
  detail: 'literal',
  type: 'keyword'
}), snippetCompletion$1('true', {
  label: 'true',
  detail: 'literal',
  type: 'keyword'
}), snippetCompletion$1('false', {
  label: 'false',
  detail: 'literal',
  type: 'keyword'
})];

function _extends() {
  _extends = Object.assign ? Object.assign.bind() : function (target) {
    for (var i = 1; i < arguments.length; i++) {
      var source = arguments[i];
      for (var key in source) {
        if (Object.prototype.hasOwnProperty.call(source, key)) {
          target[key] = source[key];
        }
      }
    }
    return target;
  };
  return _extends.apply(this, arguments);
}

function contextualKeyword(options) {
  const {
    context: nodes,
    after,
    before,
    keyword
  } = options;
  return ifInside({
    nodes,
    before,
    after,
    keyword
  }, completeFromList([{
    label: keyword,
    type: 'keyword',
    boost: 10
  }]));
}
const keywordCompletions = [contextualKeyword({
  context: 'InExpression',
  keyword: 'in'
}), contextualKeyword({
  context: 'IfExpression',
  keyword: 'then',
  after: 'if',
  before: 'else'
}), contextualKeyword({
  context: 'IfExpression',
  keyword: 'else',
  after: 'then'
}), contextualKeyword({
  context: 'QuantifiedExpression',
  keyword: 'satisfies'
}), contextualKeyword({
  context: 'ForExpression',
  after: 'InExpressions',
  keyword: 'return'
})];
const dontComplete = ['StringLiteral', 'Identifier', 'LineComment', 'BlockComment', 'PathExpression', 'Context', 'Key', 'ParameterName'];
const doComplete = ['Expr', 'ContextEntry'];
function ifExpression(completionSource) {
  const allNodes = [...dontComplete, ...doComplete];
  return context => {
    const {
      state,
      pos
    } = context;
    const match = matchUp(syntaxTree(state).resolveInner(pos, -1), allNodes);
    if (match) {
      const [_, name] = match;
      if (dontComplete.includes(name)) {
        return null;
      }
    }
    return completionSource(context);
  };
}
function snippetCompletion(snippets) {
  return ifExpression(completeFromList(snippets.map(s => _extends({}, s, {
    type: 'text'
  }))));
}
function matchLeft(node, position, nodes) {
  return matchChildren(node, position, nodes, -1);
}
function matchRight(node, position, nodes) {
  return matchChildren(node, position, nodes, 1);
}
function matchChildren(node, position, nodes, direction) {
  let child = node[direction > 0 ? 'childAfter' : 'childBefore'](position);
  while (child) {
    if (nodes.includes(child.name)) {
      return child;
    }
    if (child.type.isError && child.firstChild) {
      if (nodes.includes(child.firstChild.name)) {
        return child.firstChild;
      }
    }
    child = child[direction > 0 ? 'nextSibling' : 'prevSibling'];
  }
  return null;
}
function matchUp(node, nodeNames) {
  if (!Array.isArray(nodeNames)) {
    nodeNames = [nodeNames];
  }
  for (; node; node = node.parent) {
    const nodeType = node.type;
    const matchedName = nodeNames.find(name => name && nodeType.is(name));
    if (matchedName) {
      return [node, matchedName];
    }
    if (nodeType.isTop) {
      break;
    }
  }
  return null;
}
function ifInside(options, source) {
  const {
    nodes,
    before,
    after,
    keyword
  } = options;
  return context => {
    const {
      state,
      pos
    } = context;
    const match = matchUp(syntaxTree(state).resolveInner(pos, -1), nodes);
    if (!match) {
      return null;
    }
    const [node] = match;
    if (matchLeft(node, pos, [keyword, before])) {
      return null;
    }
    if (matchRight(node, pos, [keyword, after])) {
      return null;
    }
    if (after && !matchLeft(node, pos, [after])) {
      return null;
    }
    return source(context);
  };
}

/**
 * A FEEL language provider based on the
 * [Lezer FEEL parser](https://github.com/nikku/lezer-feel),
 * extended with highlighting and indentation information.
 */
const feelLanguage = LRLanguage.define({
  parser: parser.configure({
    props: [indentNodeProp.add({
      'Context': delimitedIndent({
        closing: '}'
      }),
      'List FilterExpression': delimitedIndent({
        closing: ']'
      }),
      'ParenthesizedExpression FunctionInvocation': continuedIndent({
        except: /^\s*\)/
      }),
      'ForExpression QuantifiedExpression IfExpression': continuedIndent({
        except: /^\s*(then|else|return|satisfies)\b/
      }),
      'FunctionDefinition': continuedIndent({
        except: /^\s*(\(|\))/
      })
    }), foldNodeProp.add({
      Context: foldInside,
      List: foldInside,
      ParenthesizedExpression: foldInside,
      FunctionDefinition(node) {
        const last = node.getChild(')');
        if (!last) return null;
        return {
          from: last.to,
          to: node.to
        };
      }
    })]
  }),
  languageData: {
    indentOnInput: /^\s*(\)|\}|\]|then|else|return|satisfies)$/,
    commentTokens: {
      line: '//',
      block: {
        open: '/*',
        close: '*/'
      }
    }
  }
});
/**
 * A language provider for FEEL Unary Tests
 */
const unaryTestsLanguage = feelLanguage.configure({
  top: 'UnaryTests'
}, 'FEEL unary tests');
/**
 * Language provider for FEEL Expression
 */
const expressionLanguage = feelLanguage.configure({
  top: 'Expression'
}, 'FEEL expression');
/**
 * Feel language support for CodeMirror.
 *
 * Includes [snippet](#lang-feel.snippets)
 */
function feel(config = {}) {
  const language = config.dialect === 'unaryTests' ? unaryTestsLanguage : expressionLanguage;
  const dialect = config.parserDialect;
  const contextTracker = trackVariables(config.context);
  const contextualLang = language.configure({
    contextTracker,
    dialect
  });
  const completions = config.completions || [snippetCompletion(snippets), keywordCompletions].flat();
  return new LanguageSupport(contextualLang, [...completions.map(autocomplete => contextualLang.data.of({
    autocomplete
  }))]);
}

export { contextualKeyword, doComplete, dontComplete, expressionLanguage, feel, feelLanguage, ifExpression, ifInside, keywordCompletions, matchChildren, matchLeft, matchRight, snippetCompletion, snippets, unaryTestsLanguage };
//# sourceMappingURL=index.js.map
