'use strict';

/**
 * Cache results of the first function call to ensure only calling once.
 *
 * ```js
 * var utils = require('lazy-cache')(require);
 * // cache the call to `require('ansi-yellow')`
 * utils('ansi-yellow', 'yellow');
 * // use `ansi-yellow`
 * console.log(utils.yellow('this is yellow'));
 * ```
 *
 * @param  {Function} `fn` Function that will be called only once.
 * @return {Function} Function that can be called to get the cached function
 * @api public
 */

function lazyCache(requireFn) {
  var cache = {};

  return function proxy(name, alias) {
    var key = alias;

    // camel-case the module `name` if `alias` is not defined
    if (typeof key !== 'string') {
      key = camelcase(name);
    }

    // create a getter to lazily invoke the module the first time it's called
    function getter() {
      return cache[key] || (cache[key] = requireFn(name));
    }

    // trip the getter if `process.env.UNLAZY` is defined
    if (unlazy(process.env)) {
      getter();
    }

    defineGetter(proxy, key, getter);
    return getter;
  };
}

/**
 * Return true if `process.env.LAZY` is true, or travis is running.
 */

function unlazy(env) {
  return env.UNLAZY === 'true' || env.UNLAZY === true || env.TRAVIS;
}

/**
 * Define a `getter` function for `prop` on the `cache` object. Dot notation is
 * supported.
 *
 * @param {Object} `cache`
 * @param {String} `prop`
 * @param {Function} `getter` Getter function
 * @return {Object}
 */

function defineGetter(cache, prop, getter) {
  if (!~prop.indexOf('.')) {
    defineProperty(cache, prop, getter);
    return cache;
  }
  var keys = prop.split('.');
  var last = keys.pop();
  var proxy = cache;
  var key;

  while ((key = keys.shift())) {
    while (key.slice(-1) === '\\') {
      key = key.slice(0, -1) + '.' + keys.shift();
    }
    proxy = proxy[key] || (proxy[key] = {});
  }

  defineProperty(proxy, last, getter);
  return cache;
}

/**
 * Define a `getter` function for `prop` on the `obj`.
 *
 * @param {Object} `obj`
 * @param {String} `prop`
 * @param {Function} `getter` Getter function
 */

function defineProperty(obj, prop, getter) {
  Object.defineProperty(obj, prop, {
    configurable: true,
    enumerable: true,
    get: getter
  });
}

/**
 * Camelcase the the given module `name`.
 */

function camelcase(str) {
  if (str.length === 1) {
    return str.toLowerCase();
  }
  str = str.replace(/^[\W_]+|[\W_]+$/g, '').toLowerCase();
  return str.replace(/[\W_]+(\w|$)/g, function(_, ch) {
    return ch.toUpperCase();
  });
}

/**
 * Expose `lazyCache`
 */

module.exports = lazyCache;
