"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _jsonldContextParser = require("jsonld-context-parser");

var _dataModel = require("@rdfjs/data-model");

var _promiseUtils = require("./promiseUtils");

/**
 * Resolves property names of a path
 * to their corresponding IRIs through a JSON-LD context.
 */
class JSONLDResolver {
  constructor(context) {
    this._context = new _jsonldContextParser.ContextParser().parse(context);
  }
  /**
   * The JSON-LD resolver supports all string properties.
   */


  supports(property) {
    return typeof property === 'string';
  }
  /**
   * Resolves the property by extending the query path with it.
   */


  resolve(property, pathData) {
    const predicate = {
      then: (0, _promiseUtils.getThen)(() => this.expandProperty(property))
    };
    return pathData.extendPath({
      property,
      predicate
    });
  }
  /**
   * Expands a JSON property key into a full IRI.
   */


  async expandProperty(property) {
    // JavaScript requires keys containing colons to be quoted,
    // so prefixed names would need to written as path['foaf:knows'].
    // We thus allow writing path.foaf_knows or path.foaf$knows instead.
    property = property.replace(/[_$]/, ':'); // Expand the property to a full IRI

    const expandedProperty = _jsonldContextParser.ContextParser.expandTerm(property, (await this._context), true);

    if (!_jsonldContextParser.ContextParser.isValidIri(expandedProperty)) throw new Error(`The JSON-LD context cannot expand the '${property}' property`);
    return (0, _dataModel.namedNode)(expandedProperty);
  }

}

exports.default = JSONLDResolver;