import PatternMatcher from './AsYouTypeFormatter.PatternMatcher'

describe('AsYouTypeFormatter.PatternMatcher', function() {
	it('should match a one-digit pattern', function() {
		const matcher = new PatternMatcher('4')

		expect(matcher.match('1')).to.be.undefined

		matcher.match('4').should.deep.equal({
			match: true
		})

		expect(matcher.match('44')).to.be.undefined

		matcher.match('44', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})
	})

	it('should match a two-digit pattern', function() {
		const matcher = new PatternMatcher('44')

		expect(matcher.match('1')).to.be.undefined

		matcher.match('4').should.deep.equal({
			partialMatch: true
		})

		matcher.match('44').should.deep.equal({
			match: true
		})

		expect(matcher.match('444')).to.be.undefined

		matcher.match('444', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})

		expect(matcher.match('55')).to.be.undefined
	})

	it('should match a one-digit one-of set (single digit)', function() {
		const matcher = new PatternMatcher('[4]')

		expect(matcher.match('1')).to.be.undefined

		matcher.match('4').should.deep.equal({
			match: true
		})

		expect(matcher.match('44')).to.be.undefined

		matcher.match('44', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})
	})

	it('should match a one-digit one-of set (multiple digits)', function() {
		const matcher = new PatternMatcher('[479]')

		expect(matcher.match('1')).to.be.undefined

		matcher.match('4').should.deep.equal({
			match: true
		})

		expect(matcher.match('44')).to.be.undefined

		matcher.match('44', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})
	})

	it('should match a one-digit one-of set using a dash notation (not inclusive)', function() {
		const matcher = new PatternMatcher('[2-5]')

		expect(matcher.match('1')).to.be.undefined

		matcher.match('4').should.deep.equal({
			match: true
		})

		expect(matcher.match('44')).to.be.undefined

		matcher.match('44', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})
	})

	it('should match a one-digit one-of set using a dash notation (inclusive)', function() {
		const matcher = new PatternMatcher('[3-4]')

		expect(matcher.match('1')).to.be.undefined

		matcher.match('4').should.deep.equal({
			match: true
		})

		expect(matcher.match('44')).to.be.undefined

		matcher.match('44', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})
	})

	it('should match a one-digit one-of set including a dash notation', function() {
		const matcher = new PatternMatcher('[124-68]')

		expect(matcher.match('0')).to.be.undefined

		matcher.match('1').should.deep.equal({
			match: true
		})

		matcher.match('2').should.deep.equal({
			match: true
		})

		expect(matcher.match('3')).to.be.undefined

		matcher.match('4').should.deep.equal({
			match: true
		})

		matcher.match('5').should.deep.equal({
			match: true
		})

		matcher.match('6').should.deep.equal({
			match: true
		})

		expect(matcher.match('7')).to.be.undefined

		matcher.match('8').should.deep.equal({
			match: true
		})

		expect(matcher.match('9')).to.be.undefined

		expect(matcher.match('88')).to.be.undefined

		matcher.match('88', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})
	})

	it('should match a two-digit one-of set', function() {
		const matcher = new PatternMatcher('[479][45]')

		expect(matcher.match('1')).to.be.undefined

		matcher.match('4').should.deep.equal({
			partialMatch: true
		})

		expect(matcher.match('5')).to.be.undefined
		expect(matcher.match('55')).to.be.undefined

		matcher.match('44').should.deep.equal({
			match: true
		})

		expect(matcher.match('444')).to.be.undefined

		matcher.match('444', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})
	})

	it('should match a two-digit one-of set (regular digit and a one-of set)', function() {
		const matcher = new PatternMatcher('1[45]')

		expect(matcher.match('0')).to.be.undefined

		matcher.match('1').should.deep.equal({
			partialMatch: true
		})

		matcher.match('15').should.deep.equal({
			match: true
		})

		expect(matcher.match('16')).to.be.undefined
	})

	it('should match a pattern with an or group', function() {
		const matcher = new PatternMatcher('7(?:1[0-68]|2[1-9])')

		expect(matcher.match('1')).to.be.undefined

		matcher.match('7').should.deep.equal({
			partialMatch: true
		})

		matcher.match('71').should.deep.equal({
			partialMatch: true
		})

		expect(matcher.match('73')).to.be.undefined

		matcher.match('711').should.deep.equal({
			match: true
		})

		expect(matcher.match('717')).to.be.undefined

		expect(matcher.match('720')).to.be.undefined

		matcher.match('722').should.deep.equal({
			match: true
		})

		expect(matcher.match('7222')).to.be.undefined

		matcher.match('7222', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})
	})

	it('should match an or pattern containing or groups', function() {
		const matcher = new PatternMatcher('2(?:2[024-9]|3[0-59]|47|6[245]|9[02-8])|3(?:3[28]|4[03-9]|5[2-46-8]|7[1-578]|8[2-9])')

		expect(matcher.match('1')).to.be.undefined

		matcher.match('2').should.deep.equal({
			partialMatch: true
		})

		matcher.match('3').should.deep.equal({
			partialMatch: true
		})

		expect(matcher.match('4')).to.be.undefined

		expect(matcher.match('21')).to.be.undefined

		matcher.match('22').should.deep.equal({
			partialMatch: true
		})

		expect(matcher.match('221')).to.be.undefined

		matcher.match('222').should.deep.equal({
			match: true
		})

		expect(matcher.match('2222')).to.be.undefined

		matcher.match('2222', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})

		matcher.match('3').should.deep.equal({
			partialMatch: true
		})

		matcher.match('33').should.deep.equal({
			partialMatch: true
		})

		matcher.match('332').should.deep.equal({
			match: true
		})

		expect(matcher.match('333')).to.be.undefined
	})

	it('should match an or pattern', function() {
		const matcher = new PatternMatcher('6|8')

		expect(matcher.match('5')).to.be.undefined

		matcher.match('6').should.deep.equal({
			match: true
		})

		expect(matcher.match('7')).to.be.undefined

		matcher.match('8').should.deep.equal({
			match: true
		})
	})

	it('should match an or pattern (one-of sets)', function() {
		const matcher = new PatternMatcher('[123]|[5-8]')

		expect(matcher.match('0')).to.be.undefined

		matcher.match('1').should.deep.equal({
			match: true
		})

		matcher.match('2').should.deep.equal({
			match: true
		})

		matcher.match('3').should.deep.equal({
			match: true
		})

		expect(matcher.match('4')).to.be.undefined

		matcher.match('5').should.deep.equal({
			match: true
		})

		matcher.match('6').should.deep.equal({
			match: true
		})

		matcher.match('7').should.deep.equal({
			match: true
		})

		matcher.match('8').should.deep.equal({
			match: true
		})

		expect(matcher.match('9')).to.be.undefined

		expect(matcher.match('18')).to.be.undefined

		matcher.match('18', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})
	})

	it('should match an or pattern (different lengths)', function() {
		const matcher = new PatternMatcher('60|8')

		expect(matcher.match('5')).to.be.undefined

		matcher.match('6').should.deep.equal({
			partialMatch: true
		})

		matcher.match('60').should.deep.equal({
			match: true
		})

		expect(matcher.match('61')).to.be.undefined

		expect(matcher.match('7')).to.be.undefined

		matcher.match('8').should.deep.equal({
			match: true
		})

		expect(matcher.match('68')).to.be.undefined
	})

	it('should match an or pattern (one-of sets) (different lengths)', function() {
		const matcher = new PatternMatcher('[123]|[5-8][2-8]')

		expect(matcher.match('0')).to.be.undefined
	})

	it('should match an or pattern (one-of sets and regular digits) (different lengths)', function() {
		const matcher = new PatternMatcher('[2358][2-5]|4')

		expect(matcher.match('0')).to.be.undefined

		matcher.match('2').should.deep.equal({
			partialMatch: true
		})

		expect(matcher.match('21')).to.be.undefined

		matcher.match('22').should.deep.equal({
			match: true
		})

		matcher.match('25').should.deep.equal({
			match: true
		})

		expect(matcher.match('26')).to.be.undefined

		expect(matcher.match('222')).to.be.undefined

		matcher.match('222', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})

		matcher.match('3').should.deep.equal({
			partialMatch: true
		})

		matcher.match('4').should.deep.equal({
			match: true
		})

		expect(matcher.match('6')).to.be.undefined
	})

	it('should match an or pattern (one-of sets and regular digits mixed) (different lengths)', function() {
		const matcher = new PatternMatcher('[2358]2|4')

		expect(matcher.match('0')).to.be.undefined

		matcher.match('2').should.deep.equal({
			partialMatch: true
		})

		expect(matcher.match('21')).to.be.undefined

		matcher.match('22').should.deep.equal({
			match: true
		})

		expect(matcher.match('222')).to.be.undefined

		matcher.match('222', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})

		matcher.match('3').should.deep.equal({
			partialMatch: true
		})

		matcher.match('4').should.deep.equal({
			match: true
		})

		expect(matcher.match('6')).to.be.undefined
	})

	it('should match an or pattern (one-of sets groups and regular digits mixed) (different lengths)', function() {
		const matcher = new PatternMatcher('1(?:11|[2-9])')

		matcher.match('1').should.deep.equal({
			partialMatch: true
		})

		expect(matcher.match('10')).to.be.undefined

		matcher.match('11').should.deep.equal({
			partialMatch: true
		})

		matcher.match('111').should.deep.equal({
			match: true
		})

		expect(matcher.match('1111')).to.be.undefined

		matcher.match('1111', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})

		matcher.match('12').should.deep.equal({
			match: true
		})

		expect(matcher.match('122')).to.be.undefined

		matcher.match('19').should.deep.equal({
			match: true
		})

		expect(matcher.match('5')).to.be.undefined
	})

	it('should match nested or groups', function() {
		const matcher = new PatternMatcher('1(?:2(?:3(?:4|5)|6)|7(?:8|9))0')

		matcher.match('1').should.deep.equal({
			partialMatch: true
		})

		expect(matcher.match('2')).to.be.undefined

		expect(matcher.match('11')).to.be.undefined

		matcher.match('12').should.deep.equal({
			partialMatch: true
		})

		expect(matcher.match('121')).to.be.undefined

		matcher.match('123').should.deep.equal({
			partialMatch: true
		})

		expect(matcher.match('1231')).to.be.undefined

		matcher.match('1234').should.deep.equal({
			partialMatch: true
		})

		matcher.match('12340').should.deep.equal({
			match: true
		})

		expect(matcher.match('123401')).to.be.undefined

		matcher.match('123401', { allowOverflow: true }).should.deep.equal({
			overflow: true
		})

		matcher.match('12350').should.deep.equal({
			match: true
		})

		expect(matcher.match('12360')).to.be.undefined

		matcher.match('1260').should.deep.equal({
			match: true
		})

		expect(matcher.match('1270')).to.be.undefined

		expect(matcher.match('1770')).to.be.undefined

		matcher.match('1780').should.deep.equal({
			match: true
		})

		matcher.match('1790').should.deep.equal({
			match: true
		})

		expect(matcher.match('18')).to.be.undefined
	})

	it('should match complex patterns', function() {
		const matcher = new PatternMatcher('(?:31|4)6|51|6(?:5[0-3579]|[6-9])|7(?:20|32|8)|[89]')

		expect(matcher.match('0')).to.be.undefined

		matcher.match('3').should.deep.equal({
			partialMatch: true
		})

		matcher.match('31').should.deep.equal({
			partialMatch: true
		})

		expect(matcher.match('32')).to.be.undefined

		matcher.match('316').should.deep.equal({
			match: true
		})

		expect(matcher.match('315')).to.be.undefined

		matcher.match('4').should.deep.equal({
			partialMatch: true
		})

		matcher.match('46').should.deep.equal({
			match: true
		})

		expect(matcher.match('47')).to.be.undefined

		matcher.match('5').should.deep.equal({
			partialMatch: true
		})

		expect(matcher.match('50')).to.be.undefined

		matcher.match('51').should.deep.equal({
			match: true
		})

		matcher.match('6').should.deep.equal({
			partialMatch: true
		})

		expect(matcher.match('64')).to.be.undefined

		matcher.match('65').should.deep.equal({
			partialMatch: true
		})

		matcher.match('650').should.deep.equal({
			match: true
		})

		expect(matcher.match('654')).to.be.undefined

		matcher.match('69').should.deep.equal({
			match: true
		})

		matcher.match('8').should.deep.equal({
			match: true
		})

		matcher.match('9').should.deep.equal({
			match: true
		})
	})
})