/**
 * @license
 * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
 * This code may only be used under the BSD style license found at
 * http://polymer.github.io/LICENSE.txt
 * The complete set of authors may be found at
 * http://polymer.github.io/AUTHORS.txt
 * The complete set of contributors may be found at
 * http://polymer.github.io/CONTRIBUTORS.txt
 * Code distributed by Google as part of the polymer project is also
 * subject to an additional IP rights grant found at
 * http://polymer.github.io/PATENTS.txt
 */
/**
 * The main LitElement module, which defines the [[`LitElement`]] base class and
 * related APIs.
 *
 *  LitElement components can define a template and a set of observed
 * properties. Changing an observed property triggers a re-render of the
 * element.
 *
 *  Import [[`LitElement`]] and [[`html`]] from this module to create a
 * component:
 *
 *  ```js
 * import {LitElement, html} from 'lit-element';
 *
 * class MyElement extends LitElement {
 *
 *   // Declare observed properties
 *   static get properties() {
 *     return {
 *       adjective: {}
 *     }
 *   }
 *
 *   constructor() {
 *     this.adjective = 'awesome';
 *   }
 *
 *   // Define the element's template
 *   render() {
 *     return html`<p>your ${adjective} template here</p>`;
 *   }
 * }
 *
 * customElements.define('my-element', MyElement);
 * ```
 *
 * `LitElement` extends [[`UpdatingElement`]] and adds lit-html templating.
 * The `UpdatingElement` class is provided for users that want to build
 * their own custom element base classes that don't use lit-html.
 *
 * @packageDocumentation
 */
import { PropertyValues, UpdatingElement } from './lib/updating-element.js';
import { RenderOptions } from 'lit-html';
import { CSSResultGroup, CSSResultOrNative } from './lib/css-tag.js';
export * from './lib/updating-element.js';
export { html, svg, TemplateResult } from 'lit-html';
export * from './lib/css-tag.js';
declare global {
    interface Window {
        litElementVersions: string[];
    }
}
declare type CSSResultFlatArray = CSSResultOrNative[];
export interface CSSResultArray extends Array<CSSResultOrNative | CSSResultArray> {
}
/**
 * Base element class that manages element properties and attributes, and
 * renders a lit-html template.
 *
 * To define a component, subclass `LitElement` and implement a
 * `render` method to provide the component's template. Define properties
 * using the [[`properties`]] property or the [[`property`]] decorator.
 */
export declare class LitElement extends UpdatingElement {
    /**
     * Ensure this class is marked as `finalized` as an optimization ensuring
     * it will not needlessly try to `finalize`.
     *
     * Note this property name is a string to prevent breaking Closure JS Compiler
     * optimizations. See updating-element.ts for more information.
     */
    protected static ['finalized']: boolean;
    /**
     * Reference to the underlying library method used to render the element's
     * DOM. By default, points to the `render` method from lit-html's render
     * module.
     *
     * This  property should not be confused with the `render` instance method,
     * which should be overridden to define a template for the element.
     *
     * @nocollapse
     */
    static render: (result: unknown, container: HTMLElement | DocumentFragment, options: RenderOptions) => void;
    /**
     * Array of styles to apply to the element. The styles should be defined
     * using the [[`css`]] tag function or via constructible stylesheets.
     */
    static styles?: CSSResultGroup;
    private static _elementStyles?;
    /**
     * Takes the styles the user supplied via the `static styles` property and
     * returns the array of styles to apply to the element.
     * Override this method to integrate into a style management system.
     *
     * Styles are deduplicated preserving the _last_ instance in the list. This
     * is a performance optimization to avoid duplicated styles that can occur
     * especially when composing via subclassing. The last item is kept to try
     * to preserve the cascade order with the assumption that it's most important
     * that last added styles override previous styles.
     *
     * @nocollapse
     */
    protected static finalizeStyles(styles?: CSSResultGroup): CSSResultFlatArray;
    protected static finalize(): boolean;
    /**
     * Node or ShadowRoot into which element DOM should be rendered. Defaults
     * to an open shadowRoot.
     */
    readonly renderRoot: HTMLElement | DocumentFragment;
    /**
     * Node before which to render content. This is used when shimming
     * `adoptedStyleSheets` and a style element may need to exist in the
     * shadowRoot after Lit rendered content.
     */
    private _renderBeforeNode?;
    /**
     * Performs element initialization. By default this calls
     * [[`createRenderRoot`]] to create the element [[`renderRoot`]] node and
     * captures any pre-set values for registered properties.
     */
    protected initialize(): void;
    /**
     * Returns the node into which the element should render and by default
     * creates and returns an open shadowRoot. Implement to customize where the
     * element's DOM is rendered. For example, to render into the element's
     * childNodes, return `this`.
     * @returns {Element|DocumentFragment} Returns a node into which to render.
     */
    protected createRenderRoot(): Element | ShadowRoot;
    /**
     * Applies the given styles to the element. Styling is applied to the element
     * only if the `renderRoot` is a `shadowRoot`. If the `rendeRoot` is not
     * a `shadowRoot`, this method may be overridden to apply styling in another
     * way. Styling will apply using `shadowRoot.adoptedStyleSheets` where
     * available and will fallback otherwise. When Shadow DOM is available but
     * `adoptedStyleSheets` is not, styles are appended to the the `shadowRoot`
     * to [mimic spec behavior](https://wicg.github.io/construct-stylesheets/#using-constructed-stylesheets).
     */
    protected adoptStyles(styles: CSSResultFlatArray): void;
    /**
     * Updates the element. This method reflects property values to attributes
     * and calls `render` to render DOM via lit-html. Setting properties inside
     * this method will *not* trigger another update.
     * @param changedProperties Map of changed properties with old values
     */
    protected update(changedProperties: PropertyValues): void;
    /**
     * Invoked on each update to perform rendering tasks. This method may return
     * any value renderable by lit-html's `NodePart` - typically a
     * `TemplateResult`. Setting properties inside this method will *not* trigger
     * the element to update.
     */
    protected render(): unknown;
}
//# sourceMappingURL=lit-element.d.ts.map