/**
@license
Copyright (c) 2019 The Polymer Project Authors. All rights reserved.
This code may only be used under the BSD style license found at
http://polymer.github.io/LICENSE.txt The complete set of authors may be found at
http://polymer.github.io/AUTHORS.txt The complete set of contributors may be
found at http://polymer.github.io/CONTRIBUTORS.txt Code distributed by Google as
part of the polymer project is also subject to an additional IP rights grant
found at http://polymer.github.io/PATENTS.txt
*/

/**
 * Whether the current browser supports `adoptedStyleSheets`.
 */
export const supportsAdoptingStyleSheets =
  window.ShadowRoot &&
  (window.ShadyCSS === undefined || window.ShadyCSS.nativeShadow) &&
  'adoptedStyleSheets' in Document.prototype &&
  'replace' in CSSStyleSheet.prototype;

export type CSSResultOrNative = CSSResult | CSSStyleSheet;

export interface CSSResultArray
  extends Array<CSSResultOrNative | CSSResultArray> {}

export type CSSResultGroup = CSSResultOrNative | CSSResultArray;

const constructionToken = Symbol();

export class CSSResult {
  readonly cssText: string;
  private _styleSheet?: CSSStyleSheet;

  constructor(cssText: string, safeToken: symbol) {
    if (safeToken !== constructionToken) {
      throw new Error(
        'CSSResult is not constructable. Use `unsafeCSS` or `css` instead.'
      );
    }
    this.cssText = cssText;
  }

  // Note, this is a getter so that it's lazy. In practice, this means
  // stylesheets are not created until the first element instance is made.
  get styleSheet(): CSSStyleSheet | undefined {
    // Note, if `supportsAdoptingStyleSheets` is true then we assume
    // CSSStyleSheet is constructable.
    if (supportsAdoptingStyleSheets && this._styleSheet === undefined) {
      this._styleSheet = new CSSStyleSheet();
      this._styleSheet.replaceSync(this.cssText);
    }
    return this._styleSheet;
  }

  toString(): string {
    return this.cssText;
  }
}

/**
 * Wrap a value for interpolation in a [[`css`]] tagged template literal.
 *
 * This is unsafe because untrusted CSS text can be used to phone home
 * or exfiltrate data to an attacker controlled site. Take care to only use
 * this with trusted input.
 */
export const unsafeCSS = (value: unknown) => {
  return new CSSResult(String(value), constructionToken);
};

const textFromCSSResult = (value: CSSResultGroup | number) => {
  if (value instanceof CSSResult) {
    return value.cssText;
  } else if (typeof value === 'number') {
    return value;
  } else {
    throw new Error(
      `Value passed to 'css' function must be a 'css' function result: ${value}. Use 'unsafeCSS' to pass non-literal values, but
            take care to ensure page security.`
    );
  }
};

const cssResultCache = new Map<string, CSSResult>();

/**
 * Template tag which which can be used with LitElement's [[LitElement.styles |
 * `styles`]] property to set element styles. For security reasons, only literal
 * string values may be used. To incorporate non-literal values [[`unsafeCSS`]]
 * may be used inside a template string part.
 */
export const css = (
  strings: TemplateStringsArray,
  ...values: (CSSResultGroup | number)[]
): CSSResultGroup => {
  const cssText = values.reduce(
    (acc, v, idx) => acc + textFromCSSResult(v) + strings[idx + 1],
    strings[0]
  );
  let result = cssResultCache.get(cssText);
  if (result === undefined) {
    cssResultCache.set(
      cssText,
      (result = new CSSResult(cssText, constructionToken))
    );
  }
  return result;
};
