/**
 * @license
 * Copyright 2021 Google LLC
 * SPDX-License-Identifier: BSD-3-Clause
 */
const microtask = Promise.resolve();
/**
 * An implementation of ReactiveControllerHost that is driven by React hooks
 * and `useController()`.
 */
class ReactControllerHost {
    constructor(kickCount, kick) {
        this._controllers = [];
        /* @internal */
        this._updatePending = true;
        this._kickCount = kickCount;
        this._kick = kick;
        this._updateCompletePromise = new Promise((res, _rej) => {
            this._resolveUpdate = res;
        });
    }
    addController(controller) {
        this._controllers.push(controller);
    }
    removeController(controller) {
        var _a;
        // Note, if the indexOf is -1, the >>> will flip the sign which makes the
        // splice do nothing.
        (_a = this._controllers) === null || _a === void 0 ? void 0 : _a.splice(this._controllers.indexOf(controller) >>> 0, 1);
    }
    requestUpdate() {
        if (!this._updatePending) {
            this._updatePending = true;
            // Trigger a React update by updating some state
            microtask.then(() => this._kick(++this._kickCount));
        }
    }
    get updateComplete() {
        return this._updateCompletePromise;
    }
    /* @internal */
    _connected() {
        this._controllers.forEach((c) => { var _a; return (_a = c.hostConnected) === null || _a === void 0 ? void 0 : _a.call(c); });
    }
    /* @internal */
    _disconnected() {
        this._controllers.forEach((c) => { var _a; return (_a = c.hostDisconnected) === null || _a === void 0 ? void 0 : _a.call(c); });
    }
    /* @internal */
    _update() {
        this._controllers.forEach((c) => { var _a; return (_a = c.hostUpdate) === null || _a === void 0 ? void 0 : _a.call(c); });
    }
    /* @internal */
    _updated() {
        this._updatePending = false;
        const resolve = this._resolveUpdate;
        // Create a new updateComplete Promise for the next update,
        // before resolving the current one.
        this._updateCompletePromise = new Promise((res, _rej) => {
            this._resolveUpdate = res;
        });
        this._controllers.forEach((c) => { var _a; return (_a = c.hostUpdated) === null || _a === void 0 ? void 0 : _a.call(c); });
        resolve(this._updatePending);
    }
}
/**
 * Creates and stores a stateful ReactiveController instance and provides it
 * with a ReactiveControllerHost that drives the controller lifecycle.
 *
 * Use this hook to convert a ReactiveController into a React hook.
 *
 * @param React the React module that provides the base hooks. Must provide
 * `useState` and `useLayoutEffect`.
 * @param createController A function that creates a controller instance. This
 * function is given a ReactControllerHost to pass to the controller. The
 * create function is only called once per component.
 */
export const useController = (React, createController) => {
    const { useState, useLayoutEffect } = React;
    // State to force updates of the React component
    const [kickCount, kick] = useState(0);
    // Create and store the controller instance. We use useState() instead of
    // useMemo() because React does not guarantee that it will preserve values
    // created with useMemo().
    // TODO (justinfagnani): since this controller are mutable, this may cause
    // issues such as "shearing" with React concurrent mode. The solution there
    // will likely be to shapshot the controller state with something like
    // `useMutableSource`:
    // https://github.com/reactjs/rfcs/blob/master/text/0147-use-mutable-source.md
    // We can address this when React's concurrent mode is closer to shipping.
    const [host] = useState(() => {
        const host = new ReactControllerHost(kickCount, kick);
        const controller = createController(host);
        host._primaryController = controller;
        host._connected();
        return host;
    });
    host._updatePending = true;
    // We use useLayoutEffect because we need updated() called synchronously
    // after rendering.
    useLayoutEffect(() => host._updated());
    // Returning a cleanup function simulates hostDisconnected timing. An empty
    // deps array tells React to only call this once: on mount with the cleanup
    // called on unmount.
    useLayoutEffect(() => () => host._disconnected(), []);
    // TODO (justinfagnani): don't call in SSR
    host._update();
    return host._primaryController;
};
//# sourceMappingURL=use-controller.js.map