/**
 * @license
 * Copyright 2018 Google LLC
 * SPDX-License-Identifier: BSD-3-Clause
 */
const DEV_MODE = true;
const reservedReactProperties = new Set([
    'children',
    'localName',
    'ref',
    'style',
    'className',
]);
const listenedEvents = new WeakMap();
/**
 * Adds an event listener for the specified event to the given node. In the
 * React setup, there should only ever be one event listener. Thus, for
 * efficiency only one listener is added and the handler for that listener is
 * updated to point to the given listener function.
 */
const addOrUpdateEventListener = (node, event, listener) => {
    let events = listenedEvents.get(node);
    if (events === undefined) {
        listenedEvents.set(node, (events = new Map()));
    }
    let handler = events.get(event);
    if (listener !== undefined) {
        // If necessary, add listener and track handler
        if (handler === undefined) {
            events.set(event, (handler = { handleEvent: listener }));
            node.addEventListener(event, handler);
            // Otherwise just update the listener with new value
        }
        else {
            handler.handleEvent = listener;
        }
        // Remove listener if one exists and value is undefined
    }
    else if (handler !== undefined) {
        events.delete(event);
        node.removeEventListener(event, handler);
    }
};
/**
 * Sets properties and events on custom elements. These properties and events
 * have been pre-filtered so we know they should apply to the custom element.
 */
const setProperty = (node, name, value, old, events) => {
    const event = events === null || events === void 0 ? void 0 : events[name];
    if (event !== undefined && value !== old) {
        // Dirty check event value.
        addOrUpdateEventListener(node, event, value);
        return;
    }
    // Note, the attribute removal here for `undefined` and `null` values is done
    // to match React's behavior on non-custom elements. It needs special
    // handling because it does not match platform behavior.  For example,
    // setting the `id` property to `undefined` sets the attribute to the string
    // "undefined." React "fixes" that odd behavior and the code here matches
    // React's convention.
    if ((value === undefined || value === null) &&
        name in HTMLElement.prototype) {
        node.removeAttribute(name);
        return;
    }
    // But don't dirty check properties; elements are assumed to do this.
    node[name] = value;
};
// Set a React ref. Note, there are 2 kinds of refs and there's no built in
// React API to set a ref.
const setRef = (ref, value) => {
    if (typeof ref === 'function') {
        ref(value);
    }
    else {
        ref.current = value;
    }
};
export function createComponent(ReactOrOptions = window.React, tagName, elementClass, events, displayName) {
    // digest overloaded parameters
    let React;
    let tag;
    let element;
    if (tagName === undefined) {
        const options = ReactOrOptions;
        ({ tagName: tag, elementClass: element, events, displayName } = options);
        React = options.react;
    }
    else {
        React = ReactOrOptions;
        element = elementClass;
        tag = tagName;
    }
    // Warn users when web components use reserved React properties
    if (DEV_MODE) {
        for (const p of reservedReactProperties) {
            if (p in element.prototype && !(p in HTMLElement.prototype)) {
                // Note, this effectively warns only for `ref` since the other
                // reserved props are on HTMLElement.prototype. To address this
                // would require crawling down the prototype, which doesn't feel worth
                // it since implementing these properties on an element is extremely
                // rare.
                console.warn(`${tagName} contains property ${p} which is a React
reserved property. It will be used by React and not set on
the element.`);
            }
        }
    }
    const Component = React.Component;
    const createElement = React.createElement;
    const eventProps = new Set(Object.keys(events !== null && events !== void 0 ? events : {}));
    class ReactComponent extends Component {
        constructor() {
            super(...arguments);
            this._element = null;
        }
        _updateElement(oldProps) {
            if (this._element === null) {
                return;
            }
            // Set element properties to the values in `this.props`
            for (const prop in this._elementProps) {
                setProperty(this._element, prop, this.props[prop], oldProps ? oldProps[prop] : undefined, events);
            }
            // Note, the spirit of React might be to "unset" any old values that
            // are no longer included; however, there's no reasonable value to set
            // them to so we just leave the previous state as is.
        }
        /**
         * Updates element properties correctly setting properties
         * on mount.
         */
        componentDidMount() {
            this._updateElement();
        }
        /**
         * Updates element properties correctly setting properties
         * on every update. Note, this does not include mount.
         */
        componentDidUpdate(old) {
            this._updateElement(old);
        }
        /**
         * Renders the custom element with a `ref` prop which allows this
         * component to reference the custom element.
         *
         * Standard attributes are passed to React and element properties and events
         * are updated in componentDidMount/componentDidUpdate.
         *
         */
        render() {
            // Extract and remove __forwardedRef from userProps in a rename-safe way
            const { __forwardedRef, ...userProps } = this.props;
            // Since refs only get fulfilled once, pass a new one if the user's ref
            // changed. This allows refs to be fulfilled as expected, going from
            // having a value to null.
            if (this._forwardedRef !== __forwardedRef) {
                this._ref = (value) => {
                    if (__forwardedRef !== null) {
                        setRef(__forwardedRef, value);
                    }
                    this._element = value;
                    this._forwardedRef = __forwardedRef;
                };
            }
            // Save element props while iterating to avoid the need to iterate again
            // when setting properties.
            this._elementProps = {};
            const props = { ref: this._ref };
            // Filters class properties and event properties out and passes the
            // remaining attributes to React. This allows attributes to use framework
            // rules for setting attributes and render correctly under SSR.
            for (const [k, v] of Object.entries(userProps)) {
                if (reservedReactProperties.has(k)) {
                    // React does *not* handle `className` for custom elements so
                    // coerce it to `class` so it's handled correctly.
                    props[k === 'className' ? 'class' : k] = v;
                    continue;
                }
                if (eventProps.has(k) || k in element.prototype) {
                    this._elementProps[k] = v;
                    continue;
                }
                props[k] = v;
            }
            return createElement(tag, props);
        }
    }
    ReactComponent.displayName = displayName !== null && displayName !== void 0 ? displayName : element.name;
    const ForwardedComponent = React.forwardRef((props, __forwardedRef) => createElement(ReactComponent, { ...props, __forwardedRef }, props === null || props === void 0 ? void 0 : props.children));
    // To ease debugging in the React Developer Tools
    ForwardedComponent.displayName = ReactComponent.displayName;
    return ForwardedComponent;
}
//# sourceMappingURL=create-component.js.map