/**
 * @license
 * Copyright 2018 Google LLC
 * SPDX-License-Identifier: BSD-3-Clause
 */
import type React from 'react';
/**
 * Creates a type to be used for the props of a web component used directly in
 * React JSX.
 *
 * Example:
 *
 * ```ts
 * declare module "react" {
 *   namespace JSX {
 *     interface IntrinsicElements {
 *       'x-foo': WebComponentProps<XFoo>;
 *     }
 *   }
 * }
 * ```
 */
export declare type WebComponentProps<I extends HTMLElement> = React.DetailedHTMLProps<React.HTMLAttributes<I>, I> & ElementProps<I>;
/**
 * Type of the React component wrapping the web component. This is the return
 * type of `createComponent`.
 */
export declare type ReactWebComponent<I extends HTMLElement, E extends EventNames = {}> = React.ForwardRefExoticComponent<React.PropsWithoutRef<ComponentProps<I, E>> & React.RefAttributes<I>>;
declare type ElementProps<I> = Partial<Omit<I, keyof HTMLElement>>;
declare type ComponentProps<I, E extends EventNames = {}> = Omit<React.HTMLAttributes<I>, keyof E> & ElementProps<I> & EventListeners<E>;
/**
 * Type used to cast an event name with an event type when providing the
 * `events` option to `createComponent` for better typing of the event handler
 * prop.
 *
 * Example:
 *
 * ```ts
 * const FooComponent = createCompoennt({
 *   ...
 *   events: {
 *     onfoo: 'foo' as EventName<FooEvent>,
 *   }
 * });
 * ```
 *
 * `onfoo` prop will have the type `(e: FooEvent) => void`.
 */
export declare type EventName<T extends Event = Event> = string & {
    __eventType: T;
};
declare type EventNames = Record<string, EventName | string>;
declare type EventListeners<R extends EventNames> = {
    [K in keyof R]?: R[K] extends EventName ? (e: R[K]['__eventType']) => void : (e: Event) => void;
};
interface Options<I extends HTMLElement, E extends EventNames = {}> {
    react: typeof React;
    tagName: string;
    elementClass: Constructor<I>;
    events?: E;
    displayName?: string;
}
declare type Constructor<T> = {
    new (): T;
};
/**
 * Creates a React component for a custom element. Properties are distinguished
 * from attributes automatically, and events can be configured so they are added
 * to the custom element as event listeners.
 *
 * @param options An options bag containing the parameters needed to generate a
 * wrapped web component.
 *
 * @param options.react The React module, typically imported from the `react`
 * npm package.
 * @param options.tagName The custom element tag name registered via
 * `customElements.define`.
 * @param options.elementClass The custom element class registered via
 * `customElements.define`.
 * @param options.events An object listing events to which the component can
 * listen. The object keys are the event property names passed in via React
 * props and the object values are the names of the corresponding events
 * generated by the custom element. For example, given `{onactivate:
 * 'activate'}` an event function may be passed via the component's `onactivate`
 * prop and will be called when the custom element fires its `activate` event.
 * @param options.displayName A React component display name, used in debugging
 * messages. Default value is inferred from the name of custom element class
 * registered via `customElements.define`.
 */
export declare const createComponent: <I extends HTMLElement, E extends EventNames = {}>({ react: React, tagName, elementClass, events, displayName, }: Options<I, E>) => ReactWebComponent<I, E>;
export {};
//# sourceMappingURL=create-component.d.ts.map