import { decorateProperty } from './base.js';

/**
 * @license
 * Copyright 2017 Google LLC
 * SPDX-License-Identifier: BSD-3-Clause
 */
/**
 * A property decorator that converts a class property into a getter that
 * returns the `assignedNodes` of the given `slot`.
 *
 * Can be passed an optional {@linkcode QueryAssignedNodesOptions} object.
 *
 * Example usage:
 * ```ts
 * class MyElement {
 *   @queryAssignedNodes({slot: 'list', flatten: true})
 *   listItems!: Array<Node>;
 *
 *   render() {
 *     return html`
 *       <slot name="list"></slot>
 *     `;
 *   }
 * }
 * ```
 *
 * Note the type of this property should be annotated as `Array<Node>`. Use the
 * queryAssignedElements decorator to list only elements, and optionally filter
 * the element list using a CSS selector.
 *
 * @category Decorator
 */
function queryAssignedNodes(options) {
    const slot = options?.slot;
    const assignedNodesOptions = options;
    return decorateProperty({
        descriptor: (_name) => ({
            get() {
                const slotSelector = `slot${slot ? `[name=${slot}]` : ':not([name])'}`;
                const slotEl = this.renderRoot?.querySelector(slotSelector);
                return slotEl?.assignedNodes(assignedNodesOptions) ?? [];
            },
            enumerable: true,
            configurable: true,
        }),
    });
}

export { queryAssignedNodes };
//# sourceMappingURL=query-assigned-nodes.js.map
