'use strict';

/**
 * Creates the region
 * @param {Object} options The following configuration options:
 * @option {String} `ariaLive`: "polite" or "assertive" (defaults to "polite")
 * @option {String} `role`: "status", "alert", or "log" (defaults to "log")
 * @option {String} `ariaRelevant`: "additions", "removals", "text", "all",
 *         or "additions text" (defaults to "additions")
 * @option {String} `ariaAtomic`: "true" or "false" (defaults to "false")
 */

function LiveRegion(options) {
  this.region = document.createElement('div');
  this.options = options || {};
  // set attrs / styles
  this.configure();
  // append it
  document.body.appendChild(this.region);
}

/**
 * configure
 * Sets attributes and offscreens the region
 */

LiveRegion.prototype.configure = function () {
  var opts = this.options;
  var region = this.region;
  // set attributes
  region.setAttribute('aria-live', opts.ariaLive || 'polite');
  region.setAttribute('role', opts.role || 'log');
  region.setAttribute('aria-relevant', opts.ariaRelevant || 'additions');
  region.setAttribute('aria-atomic', opts.ariaAtomic || 'false');

  // offscreen it
  this.region.style.position = 'absolute';
  this.region.style.width = '1px';
  this.region.style.height = '1px';
  this.region.style.marginTop = '-1px';
  this.region.style.clip = 'rect(1px, 1px, 1px, 1px)';
  this.region.style.overflow = 'hidden';
};

/**
 * announce
 * Creates a live region announcement
 * @param {String} msg The message to announce
 * @param {Number} `expire`: The number of ms before removing the announcement
 * node from the live region. This prevents the region from getting full useless
 * nodes (defaults to 7000)
 */

LiveRegion.prototype.announce = function (msg, expire) {
  var announcement = document.createElement('div');
  announcement.innerHTML = msg;
  // add it to the offscreen region
  this.region.appendChild(announcement);

  if (expire || typeof expire === 'undefined') {
    setTimeout(function () {
      this.region.removeChild(announcement);
    }.bind(this), expire || 7e3); // defaults to 7 seconds
  }
};

/**
 * destroy
 * Removes the live region DOM node inserted on initialization
 */

LiveRegion.prototype.destroy = function () {
  this.region.parentNode.removeChild(this.region)
};

/**
 * Expose LiveRegion
 */

if (typeof module !== 'undefined') {
  module.exports = LiveRegion;
}
