"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.normalizeImplicitTileData = normalizeImplicitTileData;
exports.normalizeImplicitTileHeaders = normalizeImplicitTileHeaders;
exports.normalizeTileData = normalizeTileData;
exports.normalizeTileHeaders = normalizeTileHeaders;
var _tile3dSubtreeLoader = require("../../tile-3d-subtree-loader");
var _core = require("@loaders.gl/core");
var _tiles = require("@loaders.gl/tiles");
var _parse3dImplicitTiles = require("./helpers/parse-3d-implicit-tiles");
var _s2CornersToObb = require("../utils/obb/s2-corners-to-obb");
function getTileType(tile) {
  if (!tile.contentUrl) {
    return _tiles.TILE_TYPE.EMPTY;
  }
  const contentUrl = tile.contentUrl.split('?')[0];
  const fileExtension = contentUrl.split('.').pop();
  switch (fileExtension) {
    case 'pnts':
      return _tiles.TILE_TYPE.POINTCLOUD;
    case 'i3dm':
    case 'b3dm':
    case 'glb':
    case 'gltf':
      return _tiles.TILE_TYPE.SCENEGRAPH;
    default:
      return fileExtension;
  }
}
function getRefine(refine) {
  switch (refine) {
    case 'REPLACE':
    case 'replace':
      return _tiles.TILE_REFINEMENT.REPLACE;
    case 'ADD':
    case 'add':
      return _tiles.TILE_REFINEMENT.ADD;
    default:
      return refine;
  }
}
function resolveUri(uri, basePath) {
  const urlSchemeRegex = /^[a-z][0-9a-z+.-]*:/i;
  if (urlSchemeRegex.test(basePath)) {
    const url = new URL(uri, "".concat(basePath, "/"));
    return decodeURI(url.toString());
  } else if (uri.startsWith('/')) {
    return uri;
  }
  return "".concat(basePath, "/").concat(uri);
}
function normalizeTileData(tile, options) {
  if (!tile) {
    return null;
  }
  if (tile.content) {
    const contentUri = tile.content.uri || tile.content.url;
    tile.contentUrl = resolveUri(contentUri, options.basePath);
  }
  tile.id = tile.contentUrl;
  tile.lodMetricType = _tiles.LOD_METRIC_TYPE.GEOMETRIC_ERROR;
  tile.lodMetricValue = tile.geometricError;
  tile.transformMatrix = tile.transform;
  tile.type = getTileType(tile);
  tile.refine = getRefine(tile.refine);
  return tile;
}
async function normalizeTileHeaders(tileset, options) {
  const basePath = tileset.basePath;
  let root;
  const rootImplicitTilingExtension = getImplicitTilingExtensionData(tileset === null || tileset === void 0 ? void 0 : tileset.root);
  if (rootImplicitTilingExtension && tileset.root) {
    root = await normalizeImplicitTileHeaders(tileset.root, tileset, rootImplicitTilingExtension, options);
  } else {
    root = normalizeTileData(tileset.root, tileset);
  }
  const stack = [];
  stack.push(root);
  while (stack.length > 0) {
    const tile = stack.pop() || {};
    const children = tile.children || [];
    for (let childHeader of children) {
      const childImplicitTilingExtension = getImplicitTilingExtensionData(childHeader);
      if (childImplicitTilingExtension) {
        childHeader = await normalizeImplicitTileHeaders(childHeader, tileset, childImplicitTilingExtension, options);
      } else {
        normalizeTileData(childHeader, {
          basePath
        });
      }
      stack.push(childHeader);
    }
  }
  return root;
}
async function normalizeImplicitTileHeaders(tile, tileset, implicitTilingExtension, options) {
  var _tileset$root, _tile$boundingVolume$;
  const basePath = tileset.basePath;
  const {
    subdivisionScheme,
    maximumLevel,
    subtreeLevels,
    subtrees: {
      uri: subtreesUriTemplate
    }
  } = implicitTilingExtension;
  const replacedUrlTemplate = (0, _parse3dImplicitTiles.replaceContentUrlTemplate)(subtreesUriTemplate, 0, 0, 0, 0);
  const subtreeUrl = resolveUri(replacedUrlTemplate, basePath);
  const subtree = await (0, _core.load)(subtreeUrl, _tile3dSubtreeLoader.Tile3DSubtreeLoader, options);
  const contentUrlTemplate = resolveUri(tile.content.uri, basePath);
  const refine = tileset === null || tileset === void 0 ? void 0 : (_tileset$root = tileset.root) === null || _tileset$root === void 0 ? void 0 : _tileset$root.refine;
  const rootLodMetricValue = tile.geometricError;
  const s2VolumeInfo = (_tile$boundingVolume$ = tile.boundingVolume.extensions) === null || _tile$boundingVolume$ === void 0 ? void 0 : _tile$boundingVolume$['3DTILES_bounding_volume_S2'];
  if (s2VolumeInfo) {
    const box = (0, _s2CornersToObb.convertS2BoundingVolumetoOBB)(s2VolumeInfo);
    const s2VolumeBox = {
      box,
      s2VolumeInfo
    };
    tile.boundingVolume = s2VolumeBox;
  }
  const rootBoundingVolume = tile.boundingVolume;
  const implicitOptions = {
    contentUrlTemplate,
    subtreesUriTemplate,
    subdivisionScheme,
    subtreeLevels,
    maximumLevel,
    refine,
    basePath,
    lodMetricType: _tiles.LOD_METRIC_TYPE.GEOMETRIC_ERROR,
    rootLodMetricValue,
    rootBoundingVolume,
    getTileType,
    getRefine
  };
  return await normalizeImplicitTileData(tile, subtree, implicitOptions);
}
async function normalizeImplicitTileData(tile, rootSubtree, options) {
  if (!tile) {
    return null;
  }
  tile.lodMetricType = _tiles.LOD_METRIC_TYPE.GEOMETRIC_ERROR;
  tile.lodMetricValue = tile.geometricError;
  tile.transformMatrix = tile.transform;
  const {
    children,
    contentUrl
  } = await (0, _parse3dImplicitTiles.parseImplicitTiles)({
    subtree: rootSubtree,
    options,
    s2VolumeBox: tile
  });
  if (contentUrl) {
    tile.contentUrl = contentUrl;
    tile.content = {
      uri: contentUrl.replace("".concat(options.basePath, "/"), '')
    };
  }
  tile.refine = getRefine(tile.refine);
  tile.type = getTileType(tile);
  tile.children = children;
  tile.id = tile.contentUrl;
  return tile;
}
function getImplicitTilingExtensionData(tile) {
  var _tile$extensions;
  return (tile === null || tile === void 0 ? void 0 : (_tile$extensions = tile.extensions) === null || _tile$extensions === void 0 ? void 0 : _tile$extensions['3DTILES_implicit_tiling']) || (tile === null || tile === void 0 ? void 0 : tile.implicitTiling);
}
//# sourceMappingURL=parse-3d-tile-header.js.map