"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseInstancedModel3DTile = parseInstancedModel3DTile;
var _core = require("@math.gl/core");
var _geospatial = require("@math.gl/geospatial");
var _math = require("@loaders.gl/math");
var _tile3dFeatureTable = _interopRequireDefault(require("../classes/tile-3d-feature-table"));
var _tile3dBatchTable = _interopRequireDefault(require("../classes/tile-3d-batch-table"));
var _parse3dTileHeader = require("./helpers/parse-3d-tile-header");
var _parse3dTileTables = require("./helpers/parse-3d-tile-tables");
var _parse3dTileGltfView = require("./helpers/parse-3d-tile-gltf-view");
async function parseInstancedModel3DTile(tile, arrayBuffer, byteOffset, options, context) {
  byteOffset = parseInstancedModel(tile, arrayBuffer, byteOffset, options, context);
  await (0, _parse3dTileGltfView.extractGLTF)(tile, tile.gltfFormat, options, context);
  return byteOffset;
}
function parseInstancedModel(tile, arrayBuffer, byteOffset, options, context) {
  byteOffset = (0, _parse3dTileHeader.parse3DTileHeaderSync)(tile, arrayBuffer, byteOffset);
  if (tile.version !== 1) {
    throw new Error("Instanced 3D Model version ".concat(tile.version, " is not supported"));
  }
  byteOffset = (0, _parse3dTileTables.parse3DTileTablesHeaderSync)(tile, arrayBuffer, byteOffset);
  const view = new DataView(arrayBuffer);
  tile.gltfFormat = view.getUint32(byteOffset, true);
  byteOffset += 4;
  byteOffset = (0, _parse3dTileTables.parse3DTileTablesSync)(tile, arrayBuffer, byteOffset, options);
  byteOffset = (0, _parse3dTileGltfView.parse3DTileGLTFViewSync)(tile, arrayBuffer, byteOffset, options);
  if (tile.featureTableJsonByteLength === 0) {
    throw new Error('i3dm parser: featureTableJsonByteLength is zero.');
  }
  const featureTable = new _tile3dFeatureTable.default(tile.featureTableJson, tile.featureTableBinary);
  const instancesLength = featureTable.getGlobalProperty('INSTANCES_LENGTH');
  featureTable.featuresLength = instancesLength;
  if (!Number.isFinite(instancesLength)) {
    throw new Error('i3dm parser: INSTANCES_LENGTH must be defined');
  }
  tile.eastNorthUp = featureTable.getGlobalProperty('EAST_NORTH_UP');
  tile.rtcCenter = featureTable.getGlobalProperty('RTC_CENTER', _math.GL.FLOAT, 3);
  const batchTable = new _tile3dBatchTable.default(tile.batchTableJson, tile.batchTableBinary, instancesLength);
  extractInstancedAttributes(tile, featureTable, batchTable, instancesLength);
  return byteOffset;
}
function extractInstancedAttributes(tile, featureTable, batchTable, instancesLength) {
  const collectionOptions = {
    instances: new Array(instancesLength),
    batchTable: tile._batchTable,
    cull: false,
    url: undefined,
    gltf: undefined,
    basePath: undefined,
    incrementallyLoadTextures: false,
    forwardAxis: [1, 0, 0]
  };
  const instances = collectionOptions.instances;
  const instancePosition = new _core.Vector3();
  const instanceNormalRight = new _core.Vector3();
  const instanceNormalUp = new _core.Vector3();
  const instanceNormalForward = new _core.Vector3();
  const instanceRotation = new _core.Matrix3();
  const instanceQuaternion = new _core.Quaternion();
  const instanceScale = new _core.Vector3();
  const instanceTranslationRotationScale = {};
  const instanceTransform = new _core.Matrix4();
  const scratch1 = [];
  const scratch2 = [];
  const scratchVector1 = new _core.Vector3();
  const scratchVector2 = new _core.Vector3();
  for (let i = 0; i < instancesLength; i++) {
    let position;
    if (featureTable.hasProperty('POSITION')) {
      position = featureTable.getProperty('POSITION', _math.GL.FLOAT, 3, i, instancePosition);
    } else if (featureTable.hasProperty('POSITION_QUANTIZED')) {
      position = featureTable.getProperty('POSITION_QUANTIZED', _math.GL.UNSIGNED_SHORT, 3, i, instancePosition);
      const quantizedVolumeOffset = featureTable.getGlobalProperty('QUANTIZED_VOLUME_OFFSET', _math.GL.FLOAT, 3, scratchVector1);
      if (!quantizedVolumeOffset) {
        throw new Error('i3dm parser: QUANTIZED_VOLUME_OFFSET must be defined for quantized positions.');
      }
      const quantizedVolumeScale = featureTable.getGlobalProperty('QUANTIZED_VOLUME_SCALE', _math.GL.FLOAT, 3, scratchVector2);
      if (!quantizedVolumeScale) {
        throw new Error('i3dm parser: QUANTIZED_VOLUME_SCALE must be defined for quantized positions.');
      }
      const MAX_UNSIGNED_SHORT = 65535.0;
      for (let j = 0; j < 3; j++) {
        position[j] = position[j] / MAX_UNSIGNED_SHORT * quantizedVolumeScale[j] + quantizedVolumeOffset[j];
      }
    }
    if (!position) {
      throw new Error('i3dm: POSITION or POSITION_QUANTIZED must be defined for each instance.');
    }
    instancePosition.copy(position);
    instanceTranslationRotationScale.translation = instancePosition;
    tile.normalUp = featureTable.getProperty('NORMAL_UP', _math.GL.FLOAT, 3, i, scratch1);
    tile.normalRight = featureTable.getProperty('NORMAL_RIGHT', _math.GL.FLOAT, 3, i, scratch2);
    const hasCustomOrientation = false;
    if (tile.normalUp) {
      if (!tile.normalRight) {
        throw new Error('i3dm: Custom orientation requires both NORMAL_UP and NORMAL_RIGHT.');
      }
      tile.hasCustomOrientation = true;
    } else {
      tile.octNormalUp = featureTable.getProperty('NORMAL_UP_OCT32P', _math.GL.UNSIGNED_SHORT, 2, scratch1);
      tile.octNormalRight = featureTable.getProperty('NORMAL_RIGHT_OCT32P', _math.GL.UNSIGNED_SHORT, 2, scratch2);
      if (tile.octNormalUp) {
        if (!tile.octNormalRight) {
          throw new Error('i3dm: oct-encoded orientation requires NORMAL_UP_OCT32P and NORMAL_RIGHT_OCT32P');
        }
        throw new Error('i3dm: oct-encoded orientation not implemented');
      } else if (tile.eastNorthUp) {
        _geospatial.Ellipsoid.WGS84.eastNorthUpToFixedFrame(instancePosition, instanceTransform);
        instanceTransform.getRotationMatrix3(instanceRotation);
      } else {
        instanceRotation.identity();
      }
    }
    if (hasCustomOrientation) {
      instanceNormalForward.copy(instanceNormalRight).cross(instanceNormalUp).normalize();
      instanceRotation.setColumn(0, instanceNormalRight);
      instanceRotation.setColumn(1, instanceNormalUp);
      instanceRotation.setColumn(2, instanceNormalForward);
    }
    instanceQuaternion.fromMatrix3(instanceRotation);
    instanceTranslationRotationScale.rotation = instanceQuaternion;
    instanceScale.set(1.0, 1.0, 1.0);
    const scale = featureTable.getProperty('SCALE', _math.GL.FLOAT, 1, i);
    if (Number.isFinite(scale)) {
      instanceScale.multiplyByScalar(scale);
    }
    const nonUniformScale = featureTable.getProperty('SCALE_NON_UNIFORM', _math.GL.FLOAT, 3, i, scratch1);
    if (nonUniformScale) {
      instanceScale.scale(nonUniformScale);
    }
    instanceTranslationRotationScale.scale = instanceScale;
    let batchId = featureTable.getProperty('BATCH_ID', _math.GL.UNSIGNED_SHORT, 1, i);
    if (batchId === undefined) {
      batchId = i;
    }
    const rotationMatrix = new _core.Matrix4().fromQuaternion(instanceTranslationRotationScale.rotation);
    instanceTransform.identity();
    instanceTransform.translate(instanceTranslationRotationScale.translation);
    instanceTransform.multiplyRight(rotationMatrix);
    instanceTransform.scale(instanceTranslationRotationScale.scale);
    const modelMatrix = instanceTransform.clone();
    instances[i] = {
      modelMatrix,
      batchId
    };
  }
  tile.instances = instances;
}
//# sourceMappingURL=parse-3d-tile-instanced-model.js.map