import { RowTableBatch } from '@loaders.gl/tables';
import { geojsonToBinary } from '@loaders.gl/gis';
import parseJSONSync from './lib/parse-json';
import parseJSONInBatches from './lib/parse-json-in-batches';
const VERSION = typeof "2.3.13" !== 'undefined' ? "2.3.13" : 'latest';
const GeoJSONLoaderOptions = {
  geojson: {
    TableBatch: RowTableBatch,
    batchSize: 'auto',
    workerUrl: "https://unpkg.com/@loaders.gl/json@".concat(VERSION, "/dist/geojson-loader.worker.js")
  },
  json: {
    jsonpaths: ['$', '$.features']
  },
  gis: {
    format: 'geojson'
  }
};
export const GeoJSONWorkerLoader = {
  id: 'geojson',
  name: 'GeoJSON',
  version: VERSION,
  extensions: ['geojson'],
  mimeTypes: ['application/geo+json'],
  category: 'geometry',
  text: true,
  options: GeoJSONLoaderOptions
};
export const GeoJSONLoader = { ...GeoJSONWorkerLoader,
  parse,
  parseTextSync,
  parseInBatches
};

async function parse(arrayBuffer, options) {
  return parseTextSync(new TextDecoder().decode(arrayBuffer), options);
}

function parseTextSync(text, options) {
  options = { ...GeoJSONLoaderOptions,
    ...options
  };
  options.json = { ...GeoJSONLoaderOptions.geojson,
    ...options.geojson
  };
  options.gis = options.gis || {};
  const json = parseJSONSync(text, options);

  switch (options.gis.format) {
    case 'binary':
      return geojsonToBinary(json);

    default:
      return json;
  }
}

async function parseInBatches(asyncIterator, options) {
  options = { ...GeoJSONLoaderOptions,
    ...options
  };
  options.json = { ...GeoJSONLoaderOptions.geojson,
    ...options.geojson
  };
  const geojsonIterator = parseJSONInBatches(asyncIterator, options);

  switch (options.gis.format) {
    case 'binary':
      return makeBinaryGeometryIterator(geojsonIterator);

    default:
      return geojsonIterator;
  }
}

async function* makeBinaryGeometryIterator(geojsonIterator) {
  for await (const batch of geojsonIterator) {
    batch.data = geojsonToBinary(batch.data);
    yield batch;
  }
}
//# sourceMappingURL=geojson-loader.js.map