import ChildProcess from 'child_process';
import assert from '../env-utils/assert';
import { getAvailablePort } from './process-utils';
const DEFAULT_PROCESS_OPTIONS = {
  command: null,
  arguments: [],
  portArg: null,
  port: 'auto',
  basePort: 5000,
  wait: 2000,
  nodeSpawnOptions: {
    maxBuffer: 5000 * 1024
  },
  onSuccess: processProxy => {
    console.log("Started ".concat(processProxy.options.command));
  }
};
export default class ChildProcessProxy {
  constructor({
    id = 'browser-driver'
  } = {}) {
    this.id = id;
    this.childProcess = null;
    this.port = null;
  }

  async start(options = {}) {
    options = { ...DEFAULT_PROCESS_OPTIONS,
      ...options
    };
    assert(options.command && typeof options.command === 'string');
    this.options = options;
    const args = [...options.arguments];
    this.port = options.port;

    if (options.portArg) {
      if (this.port === 'auto') {
        this.port = await getAvailablePort(options.basePort);
      }

      args.push(options.portArg, this.port);
    }

    return await new Promise((resolve, reject) => {
      try {
        const successTimer = setTimeout(() => {
          if (options.onSuccess) {
            options.onSuccess(this);
          }

          resolve({});
        }, options.wait);
        console.log("Spawning ".concat(options.command, " ").concat(options.arguments.join(' ')));
        this.childProcess = ChildProcess.spawn(options.command, args, options.spawn);
        this.childProcess.stderr.on('data', data => {
          console.log("Child process wrote to stderr: \"".concat(data, "\"."));
          clearTimeout(successTimer);
          reject(new Error(data));
        });
        this.childProcess.on('error', error => {
          console.log("Child process errored with ".concat(error));
          clearTimeout(successTimer);
          reject(error);
        });
        this.childProcess.on('close', code => {
          console.log("Child process exited with ".concat(code));
          this.childProcess = null;
        });
      } catch (error) {
        reject(error);
      }
    });
  }

  async stop() {
    if (this.childProcess) {
      this.childProcess.kill();
      this.childProcess = null;
    }
  }

  async exit(statusCode = 0) {
    try {
      await this.stop();
      process.exit(statusCode);
    } catch (error) {
      console.error(error.message || error);
      process.exit(1);
    }
  }

}
//# sourceMappingURL=child-process-proxy.js.map