import WorkerThread from './worker-thread';
export default class WorkerPool {
  constructor({
    source,
    name = 'unnamed',
    maxConcurrency = 1,
    onMessage,
    onDebug = () => {},
    reuseWorkers = true
  }) {
    this.source = source;
    this.name = name;
    this.maxConcurrency = maxConcurrency;
    this.onMessage = onMessage;
    this.onDebug = onDebug;
    this.jobQueue = [];
    this.idleQueue = [];
    this.count = 0;
    this.isDestroyed = false;
    this.reuseWorkers = reuseWorkers;
  }

  destroy() {
    this.idleQueue.forEach(worker => worker.destroy());
    this.isDestroyed = true;
  }

  process(data, jobName) {
    return new Promise((resolve, reject) => {
      this.jobQueue.push({
        data,
        jobName,
        resolve,
        reject
      });

      this._startQueuedJob();
    });
  }

  async _startQueuedJob() {
    if (!this.jobQueue.length) {
      return;
    }

    const worker = this._getAvailableWorker();

    if (!worker) {
      return;
    }

    const job = this.jobQueue.shift();
    this.onDebug({
      message: 'processing',
      worker: worker.name,
      job: job.jobName,
      backlog: this.jobQueue.length
    });

    try {
      job.resolve(await worker.process(job.data));
    } catch (error) {
      job.reject(error);
    } finally {
      this._onWorkerDone(worker);
    }
  }

  _onWorkerDone(worker) {
    if (this.isDestroyed) {
      worker.destroy();
      return;
    }

    if (this.reuseWorkers) {
      this.idleQueue.push(worker);
    } else {
      worker.destroy();
      this.count--;
    }

    this._startQueuedJob();
  }

  _getAvailableWorker() {
    if (this.idleQueue.length > 0) {
      return this.idleQueue.shift();
    }

    if (this.count < this.maxConcurrency) {
      this.count++;
      const name = "".concat(this.name.toLowerCase(), " (#").concat(this.count, " of ").concat(this.maxConcurrency, ")");
      return new WorkerThread({
        source: this.source,
        onMessage: this.onMessage,
        name
      });
    }

    return null;
  }

}
//# sourceMappingURL=worker-pool.js.map