"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _constants = require("../constants");
const GL_TYPE_TO_ARRAY_TYPE = {
  [_constants.GL_TYPE.DOUBLE]: Float64Array,
  [_constants.GL_TYPE.FLOAT]: Float32Array,
  [_constants.GL_TYPE.UNSIGNED_SHORT]: Uint16Array,
  [_constants.GL_TYPE.UNSIGNED_INT]: Uint32Array,
  [_constants.GL_TYPE.UNSIGNED_BYTE]: Uint8Array,
  [_constants.GL_TYPE.BYTE]: Int8Array,
  [_constants.GL_TYPE.SHORT]: Int16Array,
  [_constants.GL_TYPE.INT]: Int32Array
};
const NAME_TO_GL_TYPE = {
  DOUBLE: _constants.GL_TYPE.DOUBLE,
  FLOAT: _constants.GL_TYPE.FLOAT,
  UNSIGNED_SHORT: _constants.GL_TYPE.UNSIGNED_SHORT,
  UNSIGNED_INT: _constants.GL_TYPE.UNSIGNED_INT,
  UNSIGNED_BYTE: _constants.GL_TYPE.UNSIGNED_BYTE,
  BYTE: _constants.GL_TYPE.BYTE,
  SHORT: _constants.GL_TYPE.SHORT,
  INT: _constants.GL_TYPE.INT
};
const ERR_TYPE_CONVERSION = 'Failed to convert GL type';
class GLType {
  static fromTypedArray(arrayOrType) {
    arrayOrType = ArrayBuffer.isView(arrayOrType) ? arrayOrType.constructor : arrayOrType;
    for (const glType in GL_TYPE_TO_ARRAY_TYPE) {
      const ArrayType = GL_TYPE_TO_ARRAY_TYPE[glType];
      if (ArrayType === arrayOrType) {
        return glType;
      }
    }
    throw new Error(ERR_TYPE_CONVERSION);
  }
  static fromName(name) {
    const glType = NAME_TO_GL_TYPE[name];
    if (!glType) {
      throw new Error(ERR_TYPE_CONVERSION);
    }
    return glType;
  }
  static getArrayType(glType) {
    switch (glType) {
      case _constants.GL_TYPE.UNSIGNED_SHORT_5_6_5:
      case _constants.GL_TYPE.UNSIGNED_SHORT_4_4_4_4:
      case _constants.GL_TYPE.UNSIGNED_SHORT_5_5_5_1:
        return Uint16Array;
      default:
        const ArrayType = GL_TYPE_TO_ARRAY_TYPE[glType];
        if (!ArrayType) {
          throw new Error(ERR_TYPE_CONVERSION);
        }
        return ArrayType;
    }
  }
  static getByteSize(glType) {
    const ArrayType = GLType.getArrayType(glType);
    return ArrayType.BYTES_PER_ELEMENT;
  }
  static validate(glType) {
    return Boolean(GLType.getArrayType(glType));
  }
  static createTypedArray(glType, buffer) {
    let byteOffset = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;
    let length = arguments.length > 3 ? arguments[3] : undefined;
    if (length === undefined) {
      length = (buffer.byteLength - byteOffset) / GLType.getByteSize(glType);
    }
    const ArrayType = GLType.getArrayType(glType);
    return new ArrayType(buffer, byteOffset, length);
  }
}
exports.default = GLType;
//# sourceMappingURL=gl-type.js.map