"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = loadTerrain;
var _schema = require("@loaders.gl/schema");
var _martini = _interopRequireDefault(require("@mapbox/martini"));
var _delatin = _interopRequireDefault(require("./delatin"));
var _skirt = require("./helpers/skirt");
function getTerrain(imageData, width, height, elevationDecoder, tesselator) {
  const {
    rScaler,
    bScaler,
    gScaler,
    offset
  } = elevationDecoder;
  const terrain = new Float32Array((width + 1) * (height + 1));
  for (let i = 0, y = 0; y < height; y++) {
    for (let x = 0; x < width; x++, i++) {
      const k = i * 4;
      const r = imageData[k + 0];
      const g = imageData[k + 1];
      const b = imageData[k + 2];
      terrain[i + y] = r * rScaler + g * gScaler + b * bScaler + offset;
    }
  }
  if (tesselator === 'martini') {
    for (let i = (width + 1) * width, x = 0; x < width; x++, i++) {
      terrain[i] = terrain[i - width - 1];
    }
    for (let i = height, y = 0; y < height + 1; y++, i += height + 1) {
      terrain[i] = terrain[i - 1];
    }
  }
  return terrain;
}
function getMeshAttributes(vertices, terrain, width, height, bounds) {
  const gridSize = width + 1;
  const numOfVerticies = vertices.length / 2;
  const positions = new Float32Array(numOfVerticies * 3);
  const texCoords = new Float32Array(numOfVerticies * 2);
  const [minX, minY, maxX, maxY] = bounds || [0, 0, width, height];
  const xScale = (maxX - minX) / width;
  const yScale = (maxY - minY) / height;
  for (let i = 0; i < numOfVerticies; i++) {
    const x = vertices[i * 2];
    const y = vertices[i * 2 + 1];
    const pixelIdx = y * gridSize + x;
    positions[3 * i + 0] = x * xScale + minX;
    positions[3 * i + 1] = -y * yScale + maxY;
    positions[3 * i + 2] = terrain[pixelIdx];
    texCoords[2 * i + 0] = x / width;
    texCoords[2 * i + 1] = y / height;
  }
  return {
    POSITION: {
      value: positions,
      size: 3
    },
    TEXCOORD_0: {
      value: texCoords,
      size: 2
    }
  };
}
function getMesh(terrainImage, terrainOptions) {
  if (terrainImage === null) {
    return null;
  }
  const {
    meshMaxError,
    bounds,
    elevationDecoder
  } = terrainOptions;
  const {
    data,
    width,
    height
  } = terrainImage;
  let terrain;
  let mesh;
  switch (terrainOptions.tesselator) {
    case 'martini':
      terrain = getTerrain(data, width, height, elevationDecoder, terrainOptions.tesselator);
      mesh = getMartiniTileMesh(meshMaxError, width, terrain);
      break;
    case 'delatin':
      terrain = getTerrain(data, width, height, elevationDecoder, terrainOptions.tesselator);
      mesh = getDelatinTileMesh(meshMaxError, width, height, terrain);
      break;
    default:
      if (width === height && !(height & width - 1)) {
        terrain = getTerrain(data, width, height, elevationDecoder, 'martini');
        mesh = getMartiniTileMesh(meshMaxError, width, terrain);
      } else {
        terrain = getTerrain(data, width, height, elevationDecoder, 'delatin');
        mesh = getDelatinTileMesh(meshMaxError, width, height, terrain);
      }
      break;
  }
  const {
    vertices
  } = mesh;
  let {
    triangles
  } = mesh;
  let attributes = getMeshAttributes(vertices, terrain, width, height, bounds);
  const boundingBox = (0, _schema.getMeshBoundingBox)(attributes);
  if (terrainOptions.skirtHeight) {
    const {
      attributes: newAttributes,
      triangles: newTriangles
    } = (0, _skirt.addSkirt)(attributes, triangles, terrainOptions.skirtHeight);
    attributes = newAttributes;
    triangles = newTriangles;
  }
  return {
    loaderData: {
      header: {}
    },
    header: {
      vertexCount: triangles.length,
      boundingBox
    },
    mode: 4,
    indices: {
      value: Uint32Array.from(triangles),
      size: 1
    },
    attributes
  };
}
function getMartiniTileMesh(meshMaxError, width, terrain) {
  const gridSize = width + 1;
  const martini = new _martini.default(gridSize);
  const tile = martini.createTile(terrain);
  const {
    vertices,
    triangles
  } = tile.getMesh(meshMaxError);
  return {
    vertices,
    triangles
  };
}
function getDelatinTileMesh(meshMaxError, width, height, terrain) {
  const tin = new _delatin.default(terrain, width + 1, height + 1);
  tin.run(meshMaxError);
  const {
    coords,
    triangles
  } = tin;
  const vertices = coords;
  return {
    vertices,
    triangles
  };
}
async function loadTerrain(arrayBuffer, options, context) {
  const loadImageOptions = {
    ...options,
    mimeType: 'application/x.image',
    image: {
      ...options.image,
      type: 'data'
    }
  };
  const image = await context.parse(arrayBuffer, loadImageOptions);
  return getMesh(image, options.terrain);
}
//# sourceMappingURL=parse-terrain.js.map