/*!
MIT License

Copyright (c) 2018 Arturas Molcanovas <a.molcanovas@gmail.com>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

*/


import { clone, getKeyPrefix, serialiser, executeCallback, dropInstanceCommon, normaliseKey } from 'localforage-driver-commons';

const _driver = 'localforage-driver-memory';

const stores = {};
/** @internal */
class Store {
    constructor(kp) {
        this.kp = kp;
        this.data = {};
    }
    static resolve(kp) {
        if (!stores[kp]) {
            stores[kp] = new Store(kp);
        }
        return stores[kp];
    }
    clear() {
        this.data = {};
    }
    drop() {
        this.clear();
        delete stores[this.kp];
    }
    get(key) {
        return this.data[key];
    }
    key(idx) {
        return this.keys()[idx];
    }
    keys() {
        return Object.keys(this.data);
    }
    rm(k) {
        delete this.data[k];
    }
    set(k, v) {
        this.data[k] = v;
    }
}

function _initStorage(options) {
    const opts = options ? clone(options) : {};
    const kp = getKeyPrefix(opts, this._defaultConfig);
    const store = Store.resolve(kp);
    this._dbInfo = opts;
    this._dbInfo.serializer = serialiser;
    this._dbInfo.keyPrefix = kp;
    this._dbInfo.mStore = store;
    return Promise.resolve();
}

function clear(callback) {
    const promise = this.ready().then(() => {
        this._dbInfo.mStore.clear();
    });
    executeCallback(promise, callback);
    return promise;
}

function dropInstance(_options, _cb) {
    const { promise, callback } = dropInstanceCommon.apply(this, arguments);
    const outPromise = promise.then(keyPrefix => {
        Store.resolve(keyPrefix).drop();
    });
    executeCallback(outPromise, callback);
    return promise;
}

function getItem(key$, callback) {
    key$ = normaliseKey(key$);
    const promise = this.ready().then(() => {
        const result = this._dbInfo.mStore.get(key$);
        // Deserialise if the result is not null or undefined
        return result == null ? null : this._dbInfo.serializer.deserialize(result); //tslint:disable-line:triple-equals
    });
    executeCallback(promise, callback);
    return promise;
}

function iterate(iterator, callback) {
    const promise = this.ready().then(() => {
        const store = this._dbInfo.mStore;
        const keys = store.keys();
        for (let i = 0; i < keys.length; i++) {
            let value = store.get(keys[i]);
            // If a result was found, parse it from the serialized
            // string into a JS object. If result isn't truthy, the
            // key is likely undefined and we'll pass it straight
            // to the iterator.
            if (value) {
                value = this._dbInfo.serializer.deserialize(value);
            }
            value = iterator(value, keys[i], i + 1);
            if (value !== undefined) {
                return value;
            }
        }
    });
    executeCallback(promise, callback);
    return promise;
}

function key(idx, callback) {
    const promise = this.ready().then(() => {
        let result;
        try {
            result = this._dbInfo.mStore.key(idx);
            if (result === undefined) {
                result = null;
            }
        }
        catch {
            result = null;
        }
        return result;
    });
    executeCallback(promise, callback);
    return promise;
}

function keys(callback) {
    const promise = this.ready().then(() => {
        return this._dbInfo.mStore.keys();
    });
    executeCallback(promise, callback);
    return promise;
}

function length(callback) {
    const promise = this.keys().then((keys$) => keys$.length);
    executeCallback(promise, callback);
    return promise;
}

function removeItem(key$, callback) {
    key$ = normaliseKey(key$);
    const promise = this.ready().then(() => {
        this._dbInfo.mStore.rm(key$);
    });
    executeCallback(promise, callback);
    return promise;
}

function setItem(key$, value, callback) {
    key$ = normaliseKey(key$);
    const promise = this.ready().then(() => {
        if (value === undefined) {
            value = null;
        }
        // Save the original value to pass to the callback.
        const originalValue = value;
        return new Promise((resolve, reject) => {
            this._dbInfo.serializer.serialize(value, (value$, error) => {
                if (error) {
                    reject(error);
                }
                else {
                    try {
                        this._dbInfo.mStore.set(key$, value$);
                        resolve(originalValue);
                    }
                    catch (e) {
                        reject(e);
                    }
                }
            });
        });
    });
    executeCallback(promise, callback);
    return promise;
}

const _support = true;

export { _support, _driver, _initStorage, clear, dropInstance, getItem, iterate, key, keys, length, removeItem, setItem };
//# sourceMappingURL=fesm2015.js.map
