/*!
MIT License

Copyright (c) 2018 Arturas Molcanovas <a.molcanovas@gmail.com>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

*/


import { clone, getKeyPrefix, serialiser, executeCallback, dropInstanceCommon, normaliseKey } from 'localforage-driver-commons';

var _driver = 'localforage-driver-memory';

var stores = {};
/** @internal */
var Store = /** @class */ (function () {
    function Store(kp) {
        this.kp = kp;
        this.data = {};
    }
    Store.resolve = function (kp) {
        if (!stores[kp]) {
            stores[kp] = new Store(kp);
        }
        return stores[kp];
    };
    Store.prototype.clear = function () {
        this.data = {};
    };
    Store.prototype.drop = function () {
        this.clear();
        delete stores[this.kp];
    };
    Store.prototype.get = function (key) {
        return this.data[key];
    };
    Store.prototype.key = function (idx) {
        return this.keys()[idx];
    };
    Store.prototype.keys = function () {
        return Object.keys(this.data);
    };
    Store.prototype.rm = function (k) {
        delete this.data[k];
    };
    Store.prototype.set = function (k, v) {
        this.data[k] = v;
    };
    return Store;
}());

function _initStorage(options) {
    var opts = options ? clone(options) : {};
    var kp = getKeyPrefix(opts, this._defaultConfig);
    var store = Store.resolve(kp);
    this._dbInfo = opts;
    this._dbInfo.serializer = serialiser;
    this._dbInfo.keyPrefix = kp;
    this._dbInfo.mStore = store;
    return Promise.resolve();
}

function clear(callback) {
    var _this = this;
    var promise = this.ready().then(function () {
        _this._dbInfo.mStore.clear();
    });
    executeCallback(promise, callback);
    return promise;
}

function dropInstance(_options, _cb) {
    var _a = dropInstanceCommon.apply(this, arguments), promise = _a.promise, callback = _a.callback;
    var outPromise = promise.then(function (keyPrefix) {
        Store.resolve(keyPrefix).drop();
    });
    executeCallback(outPromise, callback);
    return promise;
}

function getItem(key$, callback) {
    var _this = this;
    key$ = normaliseKey(key$);
    var promise = this.ready().then(function () {
        var result = _this._dbInfo.mStore.get(key$);
        // Deserialise if the result is not null or undefined
        return result == null ? null : _this._dbInfo.serializer.deserialize(result); //tslint:disable-line:triple-equals
    });
    executeCallback(promise, callback);
    return promise;
}

function iterate(iterator, callback) {
    var _this = this;
    var promise = this.ready().then(function () {
        var store = _this._dbInfo.mStore;
        var keys = store.keys();
        for (var i = 0; i < keys.length; i++) {
            var value = store.get(keys[i]);
            // If a result was found, parse it from the serialized
            // string into a JS object. If result isn't truthy, the
            // key is likely undefined and we'll pass it straight
            // to the iterator.
            if (value) {
                value = _this._dbInfo.serializer.deserialize(value);
            }
            value = iterator(value, keys[i], i + 1);
            if (value !== undefined) {
                return value;
            }
        }
    });
    executeCallback(promise, callback);
    return promise;
}

function key(idx, callback) {
    var _this = this;
    var promise = this.ready().then(function () {
        var result;
        try {
            result = _this._dbInfo.mStore.key(idx);
            if (result === undefined) {
                result = null;
            }
        }
        catch (_a) {
            result = null;
        }
        return result;
    });
    executeCallback(promise, callback);
    return promise;
}

function keys(callback) {
    var _this = this;
    var promise = this.ready().then(function () {
        return _this._dbInfo.mStore.keys();
    });
    executeCallback(promise, callback);
    return promise;
}

function length(callback) {
    var promise = this.keys().then(function (keys$) { return keys$.length; });
    executeCallback(promise, callback);
    return promise;
}

function removeItem(key$, callback) {
    var _this = this;
    key$ = normaliseKey(key$);
    var promise = this.ready().then(function () {
        _this._dbInfo.mStore.rm(key$);
    });
    executeCallback(promise, callback);
    return promise;
}

function setItem(key$, value, callback) {
    var _this = this;
    key$ = normaliseKey(key$);
    var promise = this.ready().then(function () {
        if (value === undefined) {
            value = null;
        }
        // Save the original value to pass to the callback.
        var originalValue = value;
        return new Promise(function (resolve, reject) {
            _this._dbInfo.serializer.serialize(value, function (value$, error) {
                if (error) {
                    reject(error);
                }
                else {
                    try {
                        _this._dbInfo.mStore.set(key$, value$);
                        resolve(originalValue);
                    }
                    catch (e) {
                        reject(e);
                    }
                }
            });
        });
    });
    executeCallback(promise, callback);
    return promise;
}

var _support = true;

export { _support, _driver, _initStorage, clear, dropInstance, getItem, iterate, key, keys, length, removeItem, setItem };
//# sourceMappingURL=fesm5.js.map
