'use strict';

var _tapeCatch = require('tape-catch');

var _tapeCatch2 = _interopRequireDefault(_tapeCatch);

var _webgl = require('../../../webgl');

var _shaderTools = require('../../shader-tools');

var _index = require('./index');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// Special utility functions for df64 tests

// Copyright (c) 2016 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

/* eslint-disable camelcase */
function getRandomFloat64() {
  var upper = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 256;

  return Math.random() * Math.pow(2.0, (Math.random() - 0.5) * upper);
}

var TEST_LOOP = 20;

// Actual tests for different arithmetic functions

var TEST_CASES = {
  add_fp64: function add_fp64() {
    var float0 = getRandomFloat64();
    var float1 = getRandomFloat64();
    var float_ref = float0 + float1;
    var float0_vec2 = (0, _index.fp64ify)(float0);
    var float1_vec2 = (0, _index.fp64ify)(float1);

    return {
      uniforms: {
        a: float0_vec2,
        b: float1_vec2
      },
      line: '(' + float0_vec2 + ') + (' + float1_vec2 + ')',
      result: float_ref
    };
  },

  sub_fp64: function sub_fp64() {
    var float0 = getRandomFloat64();
    var float1 = getRandomFloat64();
    var float_ref = float0 - float1;
    var float0_vec2 = (0, _index.fp64ify)(float0);
    var float1_vec2 = (0, _index.fp64ify)(float1);

    return {
      uniforms: {
        a: float0_vec2,
        b: float1_vec2
      },
      line: '(' + float0_vec2 + ') - (' + float1_vec2 + ')',
      result: float_ref
    };
  },

  mul_fp64: function mul_fp64() {
    var float0 = getRandomFloat64(128);
    var float1 = getRandomFloat64(128);
    var float_ref = float0 * float1;

    var float0_vec2 = (0, _index.fp64ify)(float0);
    var float1_vec2 = (0, _index.fp64ify)(float1);

    return {
      uniforms: {
        a: float0_vec2,
        b: float1_vec2
      },
      line: '(' + float0_vec2 + ') * (' + float1_vec2 + ')',
      result: float_ref
    };
  },

  div_fp64: function div_fp64() {
    var float0 = getRandomFloat64(128);
    var float1 = getRandomFloat64(128);
    var float_ref = float0 / float1;

    var float0_vec2 = (0, _index.fp64ify)(float0);
    var float1_vec2 = (0, _index.fp64ify)(float1);

    return {
      uniforms: {
        a: float0_vec2,
        b: float1_vec2
      },
      line: '(' + float0_vec2 + ') / (' + float1_vec2 + ')',
      result: float_ref
    };
  },

  sqrt_fp64: function sqrt_fp64() {
    var float0 = getRandomFloat64(128);
    var float_ref = Math.sqrt(float0);

    var float0_vec2 = (0, _index.fp64ify)(float0);

    return {
      uniforms: {
        a: float0_vec2
      },
      line: 'sqrt(' + float0_vec2 + ')',
      result: float_ref
    };
  },

  exp_fp64: function exp_fp64() {
    var float0 = getRandomFloat64(6);
    var float_ref = Math.exp(float0);

    var float0_vec2 = (0, _index.fp64ify)(float0);

    return {
      uniforms: {
        a: float0_vec2
      },
      line: 'exp(' + float0_vec2 + ')',
      result: float_ref
    };
  },

  log_fp64: function log_fp64() {
    var float0 = getRandomFloat64(24);
    var float_ref = Math.log(float0);

    var float0_vec2 = (0, _index.fp64ify)(float0);

    return {
      uniforms: {
        a: float0_vec2
      },
      line: 'log(' + float0_vec2 + ')',
      result: float_ref
    };
  },

  sin_fp64: function sin_fp64() {
    var float0 = getRandomFloat64(3);
    var float_ref = Math.sin(float0);

    var float0_vec2 = (0, _index.fp64ify)(float0);

    return {
      uniforms: {
        a: float0_vec2
      },
      line: 'sin(' + float0_vec2 + ')',
      result: float_ref
    };
  },

  cos_fp64: function cos_fp64() {
    var float0 = getRandomFloat64(3);
    var float_ref = Math.cos(float0);

    var float0_vec2 = (0, _index.fp64ify)(float0);

    return {
      uniforms: {
        a: float0_vec2
      },
      line: 'cos(' + float0_vec2 + ')',
      result: float_ref
    };
  },

  tan_fp64: function tan_fp64() {
    var float0 = getRandomFloat64(3);
    var float_ref = Math.tan(float0);

    var float0_vec2 = (0, _index.fp64ify)(float0);

    return {
      uniforms: {
        a: float0_vec2
      },
      line: 'tan(' + float0_vec2 + ')',
      result: float_ref
    };
  }
};

// Utilities

function initializeGL(canvas) {
  var gl = (0, _webgl.createGLContext)();
  gl.viewport(0, 0, 16, 16);
  gl.clearColor(0, 0, 0, 1);
  gl.clearDepth(1);
  gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

  if (!gl.getExtension('OES_texture_float')) {
    // console.error('no floating point texture support!');
    throw new Error('no floating point texture support!');
  }

  return gl;
}

function initializeTexTarget(gl) {
  var framebuffer = gl.createFramebuffer();
  gl.bindFramebuffer(gl.FRAMEBUFFER, framebuffer);
  framebuffer.width = 10;
  framebuffer.height = 10;

  var tex = gl.createTexture();
  gl.bindTexture(gl.TEXTURE_2D, tex);
  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.NEAREST);
  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.NEAREST);

  gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, framebuffer.width, framebuffer.height, 0, gl.RGBA, gl.FLOAT, null);

  var renderbuffer = gl.createRenderbuffer();
  gl.bindRenderbuffer(gl.RENDERBUFFER, renderbuffer);
  gl.renderbufferStorage(gl.RENDERBUFFER, gl.DEPTH_COMPONENT16, framebuffer.width, framebuffer.height);
  gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.TEXTURE_2D, tex, 0);
  gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.DEPTH_ATTACHMENT, gl.RENDERBUFFER, renderbuffer);
}

function getGPUOutput(gl) {
  var width = 16;
  var height = gl.canvas.height;
  var buf = new Float32Array(width * height * 4);
  gl.readPixels(0, 0, width, height, gl.RGBA, gl.FLOAT, buf);
  return buf;
}

function checkError(t, result, reference) {
  var reference64 = reference[0] + reference[1];
  var result64 = result[0] + result[1];

  t.comment('------------------------');
  t.comment('CPU output: (' + reference[0] + ',' + reference[1] + ') = ' + reference64);
  t.comment('GPU output: (' + result[0] + ',' + result[1] + ',' + result[2] + ',' + result[3] + ') = ' + result64);
  t.comment('error: "' + Math.abs((reference64 - result64) / reference64) + '"');
}

function getProgram(gl, func, uniforms) {

  var functionUnderTest = '' + func + (uniforms.b ? '(a, b)' : '(a)');

  var vs = '\nattribute vec3 positions;\nuniform vec2 a;\nuniform vec2 b;\nvarying vec4 vColor;\nvoid main(void) {\n  gl_Position = vec4(positions, 1.0);\n  vec2 result = ' + functionUnderTest + ';\n  vColor = vec4(result.x, result.y, 0.0, 1.0);\n}\n';

  var fs = '\n#ifdef GL_ES\nprecision highp float;\n#endif\nvarying vec4 vColor;\nvoid main(void) {\n  gl_FragColor = vColor;\n}\n';

  return new _webgl.Program(gl, (0, _shaderTools.assembleShaders)(gl, {
    vs: vs,
    fs: fs,
    modules: ['fp64']
  })).setBuffers({
    positions: new _webgl.Buffer(gl).setData({
      data: new Float32Array([1, 1, -1, 1, 1, -1, -1, -1]),
      size: 2
    })
  }).setUniforms({ ONE: 1.0 }).setUniforms(uniforms);
}

// Test cases

(0, _tapeCatch2.default)('fp64', function (t) {
  // Initialize GL
  var gl = initializeGL();

  initializeTexTarget(gl);

  var testCount = 0;

  for (var testName in TEST_CASES) {

    var tc = TEST_CASES[testName];

    for (var idx0 = 0; idx0 < TEST_LOOP; idx0++) {
      t.comment('------------------------');
      t.comment('Loop No. ' + testCount++ + ' - ' + testName);

      // Note: Test case is a function that returns object with test data
      var testData = tc();

      var program = getProgram(gl, testName, testData.uniforms);
      program.use();
      gl.drawArrays(gl.TRIANGLE_STRIP, 0, 4);

      var cpuResult = testData.result;
      var gpuResult = getGPUOutput(gl);

      checkError(gpuResult, cpuResult);

      t.comment('------------------------');
    }
  }
  t.end();
});
//# sourceMappingURL=data:application/json;base64,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