import { getShaderDependencies, getShaderModule } from './register-shaders';
import { glGetDebugInfo } from '../../webgl';
import assert from 'assert';

var VERTEX_SHADER = 'vertexShader';
var FRAGMENT_SHADER = 'fragmentShader';

export function checkRendererVendor(debugInfo, gpuVendor) {
  var vendor = debugInfo.vendor,
      renderer = debugInfo.renderer;

  var result = void 0;
  switch (gpuVendor) {
    case 'nvidia':
      result = vendor.match(/NVIDIA/i) || renderer.match(/NVIDIA/i);
      break;
    case 'intel':
      result = vendor.match(/INTEL/i) || renderer.match(/INTEL/i);
      break;
    case 'amd':
      result = vendor.match(/AMD/i) || renderer.match(/AMD/i) || vendor.match(/ATI/i) || renderer.match(/ATI/i);
      break;
    default:
      result = false;
  }
  return result;
}

export function getPlatformShaderDefines(gl) {
  /* eslint-disable */
  var platformDefines = '';
  var debugInfo = glGetDebugInfo(gl);

  if (checkRendererVendor(debugInfo, 'nvidia')) {
    platformDefines += '#define NVIDIA_GPU\n#define NVIDIA_FP64_WORKAROUND 1\n#define NVIDIA_EQUATION_WORKAROUND 1\n';
  } else if (checkRendererVendor(debugInfo, 'intel')) {
    platformDefines += '#define INTEL_GPU\n#define INTEL_FP64_WORKAROUND 1\n#define NVIDIA_EQUATION_WORKAROUND 1\n #define INTEL_TAN_WORKAROUND 1\n';
  } else if (checkRendererVendor(debugInfo, 'amd')) {
    platformDefines += '#define AMD_GPU\n';
  } else {
    platformDefines += '#define DEFAULT_GPU\n';
  }

  return platformDefines;
}

function assembleShader(gl, _ref) {
  var source = _ref.source,
      type = _ref.type,
      _ref$modules = _ref.modules,
      modules = _ref$modules === undefined ? [] : _ref$modules;

  assert(typeof source === 'string', 'shader source must be a string');

  // Add platform defines
  var assembledSource = getPlatformShaderDefines(gl) + '\n';

  // Add dependent modules in resolved order
  var _iteratorNormalCompletion = true;
  var _didIteratorError = false;
  var _iteratorError = undefined;

  try {
    for (var _iterator = modules[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
      var moduleName = _step.value;

      var shaderModule = getShaderModule(moduleName);
      if (!shaderModule) {
        assert(shaderModule, 'shader module is not defined');
      }
      var moduleSource = shaderModule[type];
      assembledSource += '// BEGIN SHADER MODULE ' + moduleName + '\n#define MODULE_' + moduleName.toUpperCase() + '\n' + moduleSource + '\n// END SHADER MODULE ' + moduleName;
    }

    // Add actual source of shader
  } catch (err) {
    _didIteratorError = true;
    _iteratorError = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion && _iterator.return) {
        _iterator.return();
      }
    } finally {
      if (_didIteratorError) {
        throw _iteratorError;
      }
    }
  }

  assembledSource += source;

  return assembledSource;
}

/**
 * Apply set of modules
 */
export function assembleShaders(gl) {
  var opts = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  var vs = opts.vs,
      fs = opts.fs;

  var modules = getShaderDependencies(opts.modules || []);
  return {
    gl: gl,
    vs: assembleShader(gl, Object.assign({}, opts, { source: vs, type: VERTEX_SHADER, modules: modules })),
    fs: assembleShader(gl, Object.assign({}, opts, { source: fs, type: FRAGMENT_SHADER, modules: modules }))
  };
}
//# sourceMappingURL=data:application/json;base64,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