import assert from 'assert';

var shaderModules = {};

/**
 * Registers an array of shader modules
 * @param {Object[]} shaderModuleList - Array of shader modules
 */
export function registerShaderModules(shaderModuleList) {
  var _iteratorNormalCompletion = true;
  var _didIteratorError = false;
  var _iteratorError = undefined;

  try {
    for (var _iterator = shaderModuleList[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
      var shaderModule = _step.value;

      assert(shaderModule.name, 'shader module has no name');
      if (shaderModules[shaderModule.name]) {
        throw new Error('shader module ' + shaderModule.name + ' already registered');
      }
      shaderModules[shaderModule.name] = shaderModule;
      shaderModule.dependencies = shaderModule.dependencies || [];
    }
  } catch (err) {
    _didIteratorError = true;
    _iteratorError = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion && _iterator.return) {
        _iterator.return();
      }
    } finally {
      if (_didIteratorError) {
        throw _iteratorError;
      }
    }
  }
}

export function getShaderModule(moduleName) {
  var shaderModule = shaderModules[moduleName];
  if (!shaderModule) {
    // console.log(`${moduleName} not in registered modules:`, shaderModules);
    assert(false, 'Unknown shader module ' + moduleName);
  }
  return shaderModule;
}

/**
 * Takes a list of shader module names and returns a new list of
 * shader module names that includes all dependencies, sorted so
 * that modules that are dependencies of other modules come first.
 *
 * If the shader glsl code from the returned modules is concatenated
 * in the reverse order, it is guaranteed that all functions be resolved and
 * that all function and variable definitions come before use.
 *
 * @param {String[]} moduleNames - Array of module names
 * @return {String[]} - Array of modules
 */
export function getShaderDependencies(moduleNames) {
  var result = {};
  getDependencyGraph({
    moduleNames: moduleNames,
    level: 0,
    result: result
  });
  return Object.keys(result).sort(function (a, b) {
    return result[a] - result[b];
  });
}

// Adds another level of dependencies to the result map
function getDependencyGraph(_ref) {
  var moduleNames = _ref.moduleNames,
      level = _ref.level,
      result = _ref.result;

  if (level >= 5) {
    throw new Error('Possible loop in shader dependency graph');
  }
  var _iteratorNormalCompletion2 = true;
  var _didIteratorError2 = false;
  var _iteratorError2 = undefined;

  try {
    for (var _iterator2 = moduleNames[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
      var moduleName = _step2.value;

      result[moduleName] = level;
    }
  } catch (err) {
    _didIteratorError2 = true;
    _iteratorError2 = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion2 && _iterator2.return) {
        _iterator2.return();
      }
    } finally {
      if (_didIteratorError2) {
        throw _iteratorError2;
      }
    }
  }

  var _iteratorNormalCompletion3 = true;
  var _didIteratorError3 = false;
  var _iteratorError3 = undefined;

  try {
    for (var _iterator3 = moduleNames[Symbol.iterator](), _step3; !(_iteratorNormalCompletion3 = (_step3 = _iterator3.next()).done); _iteratorNormalCompletion3 = true) {
      var _moduleName = _step3.value;

      var shaderModule = getShaderModule(_moduleName);

      getDependencyGraph({
        moduleNames: shaderModule.dependencies,
        level: level,
        result: result
      });
    }
  } catch (err) {
    _didIteratorError3 = true;
    _iteratorError3 = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion3 && _iterator3.return) {
        _iterator3.return();
      }
    } finally {
      if (_didIteratorError3) {
        throw _iteratorError3;
      }
    }
  }

  return result;
}
//# sourceMappingURL=data:application/json;base64,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