function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

// WebGLRenderingContext related methods
/* eslint-disable no-try-catch, no-loop-func */
import WebGLDebug from 'webgl-debug';
import { WebGLRenderingContext, webGLTypesAvailable } from './webgl-types';
import { assertWebGLContext, isWebGL2Context } from './webgl-checks';
import queryManager from './helpers/query-manager';
import { log, isBrowser, isPageLoaded, pageLoadPromise } from '../utils';
import { global } from '../utils/globals';
import assert from 'assert';
/* global document */

var luma = global.luma;


var ERR_WEBGL_MISSING_BROWSER = 'WebGL API is missing. Check your if your browser supports WebGL or\ninstall a recent version of a major browser.';

var ERR_WEBGL_MISSING_NODE = 'WebGL API is missing. To run luma.gl under Node.js, please "npm install gl"\nand import \'luma.gl/headless\' before importing \'luma.gl\'.';

var ERR_HEADLESSGL_NOT_AVAILABLE = 'Cannot create headless WebGL context, headlessGL not available';

var ERR_HEADLESSGL_FAILED = 'headlessGL failed to create headless WebGL context';

// Checks if WebGL is enabled and creates a context for using WebGL.
/* eslint-disable complexity, max-statements */
export function createGLContext() {
  var opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
  var canvas = opts.canvas;
  var _opts$width = opts.width,
      width = _opts$width === undefined ? 800 : _opts$width,
      _opts$height = opts.height,
      height = _opts$height === undefined ? 600 : _opts$height,
      _opts$webgl = opts.webgl2,
      webgl2 = _opts$webgl === undefined ? false : _opts$webgl,
      _opts$debug = opts.debug,
      debug = _opts$debug === undefined ? true : _opts$debug;


  var gl = void 0;

  if (!isBrowser) {
    gl = _createHeadlessContext(width, height, opts);
  } else {
    // Create browser gl context
    if (!webGLTypesAvailable) {
      throw new Error(ERR_WEBGL_MISSING_BROWSER);
    }
    // Make sure we have a canvas
    canvas = canvas;
    if (typeof canvas === 'string') {
      if (!isPageLoaded) {
        throw new Error('createGLContext called on canvas \'' + canvas + '\' before page was loaded');
      }
      canvas = document.getElementById(canvas);
    }
    if (!canvas) {
      canvas = _createCanvas();
    }

    canvas.addEventListener('webglcontextcreationerror', function (e) {
      log.log(0, e.statusMessage || 'Unknown error');
    }, false);

    // Prefer webgl2 over webgl1, prefer conformant over experimental
    if (webgl2) {
      gl = gl || canvas.getContext('webgl2', opts);
      gl = gl || canvas.getContext('experimental-webgl2', opts);
    } else {
      gl = gl || canvas.getContext('webgl', opts);
      gl = gl || canvas.getContext('experimental-webgl', opts);
    }

    assert(gl, 'Failed to create WebGLRenderingContext');
  }

  if (isBrowser && debug) {
    var debugGL = WebGLDebug.makeDebugContext(gl, throwOnError, validateArgsAndLog);

    var WebGLDebugContext = function WebGLDebugContext() {
      _classCallCheck(this, WebGLDebugContext);
    };

    Object.assign(WebGLDebugContext.prototype, debugGL);
    gl = debugGL;
    gl.debug = true;
    log.priority = log.priority < 1 ? 1 : log.priority;

    logInfo(gl);
  }

  return gl;
}

// Create a canvas set to 100%
// TODO - remove
function _createCanvas() {
  var canvas = document.createElement('canvas');
  canvas.id = 'lumagl-canvas';
  canvas.style.width = '100%';
  canvas.style.height = '100%';
  // adds the canvas to the body element
  pageLoadPromise.then(function (document) {
    var body = document.body;
    body.insertBefore(canvas, body.firstChild);
  });
  return canvas;
}

function _createHeadlessContext(width, height, opts) {
  // Create headless gl context
  if (!webGLTypesAvailable) {
    throw new Error(ERR_WEBGL_MISSING_NODE);
  }
  if (!luma.globals.headlessGL) {
    throw new Error(ERR_HEADLESSGL_NOT_AVAILABLE);
  }
  var gl = luma.globals.headlessGL(width, height, opts);
  if (!gl) {
    throw new Error(ERR_HEADLESSGL_FAILED);
  }
  return gl;
}

// Returns the extension or throws an error
export function getGLExtension(gl, extensionName) {
  // assertWebGLContext(gl);
  var ERROR = 'Illegal arg to getExtension';
  assert(gl instanceof WebGLRenderingContext, ERROR);
  assert(typeof extensionName === 'string', ERROR);
  var extension = gl.getExtension(extensionName);
  assert(extension, extensionName + ' not supported!');
  return extension;
}

// POLLING FOR PENDING QUERIES

// Calling this function checks all pending queries for completion
export function poll(gl) {
  assertWebGLContext(gl);
  queryManager.poll(gl);
}

// VERY LIMITED / BASIC GL STATE MANAGEMENT

// Executes a function with gl states temporarily set, exception safe
// Currently support scissor test and framebuffer binding
export function glContextWithState(gl, _ref, func) {
  var scissorTest = _ref.scissorTest,
      framebuffer = _ref.framebuffer;

  // assertWebGLContext(gl);

  var scissorTestWasEnabled = void 0;
  if (scissorTest) {
    scissorTestWasEnabled = gl.isEnabled(gl.SCISSOR_TEST);
    var x = scissorTest.x,
        y = scissorTest.y,
        w = scissorTest.w,
        h = scissorTest.h;

    gl.enable(gl.SCISSOR_TEST);
    gl.scissor(x, y, w, h);
  }

  if (framebuffer) {
    // TODO - was there any previously set frame buffer we need to remember?
    framebuffer.bind();
  }

  var value = void 0;
  try {
    value = func(gl);
  } finally {
    if (!scissorTestWasEnabled) {
      gl.disable(gl.SCISSOR_TEST);
    }
    if (framebuffer) {
      // TODO - was there any previously set frame buffer?
      // TODO - delegate "unbind" to Framebuffer object?
      gl.bindFramebuffer(gl.FRAMEBUFFER, null);
    }
  }

  return value;
}

// DEBUG INFO

/**
 * Provides strings identifying the GPU vendor and driver.
 * https://www.khronos.org/registry/webgl/extensions/WEBGL_debug_renderer_info/
 * @param {WebGLRenderingContext} gl - context
 * @return {Object} - 'vendor' and 'renderer' string fields.
 */
export function glGetDebugInfo(gl) {
  var info = gl.getExtension('WEBGL_debug_renderer_info');
  // We can't determine if 'WEBGL_debug_renderer_info' is supported by
  // checking whether info is null here. Firefox doesn't follow the
  // specs by returning null for unsupported extension. Instead,
  // it returns an object without GL_UNMASKED_VENDOR_WEBGL and GL_UNMASKED_RENDERER_WEBGL.
  return {
    vendor: info && info.UNMASKED_VENDOR_WEBGL ? gl.getParameter(info.UNMASKED_VENDOR_WEBGL) : 'unknown',
    renderer: info && info.UNMASKED_RENDERER_WEBGL ? gl.getParameter(info.UNMASKED_RENDERER_WEBGL) : 'unknown'
  };
}

function logInfo(gl) {
  var webGL = isWebGL2Context(gl) ? 'WebGL2' : 'WebGL1';
  var info = glGetDebugInfo(gl);
  var driver = info ? '(' + info.vendor + ' ' + info.renderer + ')' : '';
  var debug = gl.debug ? 'debug' : '';
  log.log(0, 'luma.gl: Created ' + webGL + ' ' + debug + ' context ' + driver, gl);

  // const extensions = gl.getSupportedExtensions();
  // log.log(0, `Supported extensions: [${extensions.join(', ')}]`);
}

// DEBUG TRACING

function getFunctionString(functionName, functionArgs) {
  var args = WebGLDebug.glFunctionArgsToString(functionName, functionArgs);
  args = '' + args.slice(0, 100) + (args.length > 100 ? '...' : '');
  return 'gl.' + functionName + '(' + args + ')';
}

function throwOnError(err, functionName, args) {
  var errorMessage = WebGLDebug.glEnumToString(err);
  var functionArgs = WebGLDebug.glFunctionArgsToString(functionName, args);
  throw new Error(errorMessage + ' was caused by call to: ' + ('gl.' + functionName + '(' + functionArgs + ')'));
}

// Don't generate function string until it is needed
function validateArgsAndLog(functionName, functionArgs) {
  var functionString = void 0;
  if (log.priority >= 4) {
    functionString = getFunctionString(functionName, functionArgs);
    log.info(4, '' + functionString);
  }

  var _iteratorNormalCompletion = true;
  var _didIteratorError = false;
  var _iteratorError = undefined;

  try {
    for (var _iterator = functionArgs[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
      var arg = _step.value;

      if (arg === undefined) {
        functionString = functionString || getFunctionString(functionName, functionArgs);
        throw new Error('Undefined argument: ' + functionString);
      }
    }
  } catch (err) {
    _didIteratorError = true;
    _iteratorError = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion && _iterator.return) {
        _iterator.return();
      }
    } finally {
      if (_didIteratorError) {
        throw _iteratorError;
      }
    }
  }

  if (log.break) {
    functionString = functionString || getFunctionString(functionName, functionArgs);
    var isBreakpoint = log.break && log.break.every(function (breakString) {
      return functionString.indexOf(breakString) !== -1;
    });

    /* eslint-disable no-debugger */
    if (isBreakpoint) {
      debugger;
    }
    /* eslint-enable no-debugger */
  }
}
//# sourceMappingURL=data:application/json;base64,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