var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

import { GL, glGet, glTypeToArray, glTypeFromArray } from './webgl';
import { assertWebGLContext, assertWebGL2Context } from './webgl-checks';
import Texture2D from './texture-2d';
import Renderbuffer from './renderbuffer';
import assert from 'assert';
import { uid, log } from '../utils';

// Returns number of components in a specific WebGL format
function glFormatToComponents(format) {
  switch (format) {
    case GL.ALPHA:
      return 1;
    case GL.RGB:
      return 3;
    case GL.RGBA:
      return 4;
    default:
      throw new Error('Unknown format');
  }
}

var Framebuffer = function () {
  _createClass(Framebuffer, null, [{
    key: 'makeFrom',
    value: function makeFrom(gl) {
      var object = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      return object instanceof Framebuffer ? object :
      // Use .handle (e.g from stack.gl's gl-buffer), else use buffer directly
      new Framebuffer(gl, { handle: object.handle || object });
    }

    /* eslint-disable max-statements */

  }]);

  function Framebuffer(gl) {
    var opts = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    _classCallCheck(this, Framebuffer);

    assertWebGLContext(gl);

    var id = opts.id;


    var handle = gl.createFramebuffer();
    if (!handle) {
      throw new Error('Failed to create WebGL Framebuffer');
    }

    this.gl = gl;
    this.id = uid(id);
    this.handle = handle;
    this.colorBuffer = null;
    this.depthBuffer = null;
    this.stencilBuffer = null;
    this.texture = null;
    this.userData = {};
    this.width = 0;
    this.height = 0;
    Object.seal(this);

    this.resize(opts);
  }
  /* eslint-enable max-statements */

  _createClass(Framebuffer, [{
    key: 'delete',
    value: function _delete() {
      var gl = this.gl;

      gl.deleteFramebuffer(this.handle);
    }

    // SIMPLIFIED INTERFACE

  }, {
    key: 'resize',
    value: function resize(_ref) {
      var width = _ref.width,
          height = _ref.height;

      this.update({ width: width, height: height });
    }

    /* eslint-disable max-statements */

  }, {
    key: 'update',
    value: function update(_ref2) {
      var _ref2$width = _ref2.width,
          width = _ref2$width === undefined ? 1 : _ref2$width,
          _ref2$height = _ref2.height,
          height = _ref2$height === undefined ? 1 : _ref2$height,
          _ref2$depth = _ref2.depth,
          depth = _ref2$depth === undefined ? true : _ref2$depth,
          _ref2$minFilter = _ref2.minFilter,
          minFilter = _ref2$minFilter === undefined ? GL.NEAREST : _ref2$minFilter,
          _ref2$magFilter = _ref2.magFilter,
          magFilter = _ref2$magFilter === undefined ? GL.NEAREST : _ref2$magFilter,
          _ref2$format = _ref2.format,
          format = _ref2$format === undefined ? GL.RGBA : _ref2$format,
          _ref2$type = _ref2.type,
          type = _ref2$type === undefined ? GL.UNSIGNED_BYTE : _ref2$type;

      assert(width >= 0 && height >= 0, 'Width and height need to be integers');
      if (width === this.width && height === this.height) {
        return;
      }

      log.log(2, 'Resizing framebuffer ' + this.id + ' to ' + width + 'x' + height);

      var gl = this.gl;

      // TODO - do we need to reallocate the framebuffer?

      var colorBuffer = new Texture2D(gl, {
        minFilter: this.minFilter,
        magFilter: this.magFilter
      })
      // TODO - should be handled by Texture2D constructor?
      .setImageData({
        data: null,
        width: width,
        height: height,
        type: type,
        format: format
      });

      this.attachTexture({
        attachment: GL.COLOR_ATTACHMENT0,
        texture: colorBuffer
      });

      if (this.colorBuffer) {
        this.colorBuffer.delete();
      }
      this.colorBuffer = colorBuffer;
      this.texture = colorBuffer;

      // Add a depth buffer if requested
      if (depth) {
        var depthBuffer = new Renderbuffer(gl).storage({
          internalFormat: GL.DEPTH_COMPONENT16,
          width: width,
          height: height
        });
        this.attachRenderbuffer({
          attachment: GL.DEPTH_ATTACHMENT,
          renderbuffer: depthBuffer
        });

        if (this.depthBuffer) {
          this.depthBuffer.delete();
        }
        this.depthBuffer = depthBuffer;
      }

      // Checks that framebuffer was properly set up,
      // if not, throws an explanatory error
      this.checkStatus();

      this.width = width;
      this.height = height;
    }

    // WEBGL INTERFACE

  }, {
    key: 'bind',
    value: function bind() {
      var _ref3 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref3$target = _ref3.target,
          target = _ref3$target === undefined ? GL.FRAMEBUFFER : _ref3$target;

      var gl = this.gl;

      gl.bindFramebuffer(target, this.handle);
      return this;
    }
  }, {
    key: 'unbind',
    value: function unbind() {
      var _ref4 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref4$target = _ref4.target,
          target = _ref4$target === undefined ? GL.FRAMEBUFFER : _ref4$target;

      var gl = this.gl;

      gl.bindFramebuffer(target, null);
      return this;
    }

    //
    // NOTE: Slow requires roundtrip to GPU
    // App can provide pixelArray or have it auto allocated by this method
    // @returns {Uint8Array|Uint16Array|FloatArray} - pixel array,
    //  newly allocated by this method unless provided by app.

  }, {
    key: 'readPixels',
    value: function readPixels(_ref5) {
      var _ref5$x = _ref5.x,
          x = _ref5$x === undefined ? 0 : _ref5$x,
          _ref5$y = _ref5.y,
          y = _ref5$y === undefined ? 0 : _ref5$y,
          width = _ref5.width,
          height = _ref5.height,
          _ref5$format = _ref5.format,
          format = _ref5$format === undefined ? GL.RGBA : _ref5$format,
          type = _ref5.type,
          _ref5$pixelArray = _ref5.pixelArray,
          pixelArray = _ref5$pixelArray === undefined ? null : _ref5$pixelArray;
      var gl = this.gl;

      // Deduce type and allocated pixelArray if needed

      if (!pixelArray) {
        // Allocate pixel array if not already available, using supplied type
        type = type || GL.UNSIGNED_BYTE;
        var ArrayType = glTypeToArray(type);
        var components = glFormatToComponents(format);
        // TODO - check for composite type (components = 1).
        pixelArray = pixelArray || new ArrayType(width * height * components);
      }

      // Pixel array available, if necessary, deduce type from it.
      type = type || glTypeFromArray(pixelArray);

      this.bind();
      gl.readPixels(x, y, width, height, format, type, pixelArray);
      this.unbind();

      return pixelArray;
    }

    /**
     * Used to attach textures to a framebuffer, the textures will store
     * the various buffers.
     *
     *  The set of available attachments is larger in WebGL2, and also the
     *  extensions WEBGL_draw_buffers and WEBGL_depth_texture provide additional
     *  attachments that match or exceed the WebGL2 set.
     *
     * @param {Texture2D|TextureCube|WebGLTexture|null} opt.texture=null -
     *    default is null which unbinds the texture for the attachment
     * @param {String|Number} opt.attachment= - which attachment to bind
     *    defaults to gl.COLOR_ATTACHMENT0.
     * @param {String|Number} opt.target= - bind point, normally gl.FRAMEBUFFER
     *    (WebGL2 support separating bet)
     * @param {String|Number} opt.textureTarget= - can be used to specify
     *    faces of a cube map.
     * @returns {FrameBuffer} returns itself to enable chaining
     */

  }, {
    key: 'attachTexture',
    value: function attachTexture() {
      var _ref6 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref6$texture = _ref6.texture,
          texture = _ref6$texture === undefined ? null : _ref6$texture,
          _ref6$target = _ref6.target,
          target = _ref6$target === undefined ? GL.FRAMEBUFFER : _ref6$target,
          _ref6$attachment = _ref6.attachment,
          attachment = _ref6$attachment === undefined ? GL.COLOR_ATTACHMENT0 : _ref6$attachment,
          _ref6$textureTarget = _ref6.textureTarget,
          textureTarget = _ref6$textureTarget === undefined ? GL.TEXTURE_2D : _ref6$textureTarget,
          _ref6$mipmapLevel = _ref6.mipmapLevel,
          mipmapLevel = _ref6$mipmapLevel === undefined ? 0 : _ref6$mipmapLevel;

      var gl = this.gl;


      texture = texture && Texture2D.makeFrom(gl, texture);

      this.bind({ target: target });

      gl.framebufferTexture2D(target, glGet(attachment), glGet(textureTarget), texture.handle, mipmapLevel);

      this.unbind();
      return this;
    }

    /**
     * Used to attach a framebuffer to a framebuffer, the textures will store
     * the various buffers.
     * @param {Object} opts= - named parameters
     * @param {RenderBuffer|WebGLRenderBuffer|null} opts.renderbuffer=null -
     *    renderbuffer to bind
     *    default is null which unbinds the renderbuffer for the attachment
     * @param {String|Number} opts.attachment= - which buffer to bind
     * @returns {FrameBuffer} returns itself to enable chaining
     */

  }, {
    key: 'attachRenderbuffer',
    value: function attachRenderbuffer() {
      var _ref7 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref7$renderbuffer = _ref7.renderbuffer,
          renderbuffer = _ref7$renderbuffer === undefined ? null : _ref7$renderbuffer,
          _ref7$attachment = _ref7.attachment,
          attachment = _ref7$attachment === undefined ? GL.COLOR_ATTACHMENT0 : _ref7$attachment,
          _ref7$target = _ref7.target,
          target = _ref7$target === undefined ? GL.FRAMEBUFFER : _ref7$target,
          _ref7$renderbufferTar = _ref7.renderbufferTarget,
          renderbufferTarget = _ref7$renderbufferTar === undefined ? GL.RENDERBUFFER : _ref7$renderbufferTar;

      var gl = this.gl;

      renderbuffer = renderbuffer && Renderbuffer.makeFrom(gl, renderbuffer);

      this.bind({ target: target });

      gl.framebufferRenderbuffer(target, glGet(attachment), glGet(renderbufferTarget), renderbuffer.handle);

      this.unbind({ target: target });

      return this;
    }
  }, {
    key: 'checkStatus',
    value: function checkStatus() {
      var _ref8 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref8$target = _ref8.target,
          target = _ref8$target === undefined ? GL.FRAMEBUFFER : _ref8$target;

      var gl = this.gl;


      this.bind({ target: target });

      var status = gl.checkFramebufferStatus(target);

      this.unbind({ target: target });

      if (status !== gl.FRAMEBUFFER_COMPLETE) {
        throw new Error(this._getFrameBufferStatus(status));
      }

      return this;
    }

    // WEBGL2 INTERFACE

  }, {
    key: 'blit',
    value: function blit(_ref9) {
      var srcX0 = _ref9.srcX0,
          srcY0 = _ref9.srcY0,
          srcX1 = _ref9.srcX1,
          srcY1 = _ref9.srcY1,
          dstX0 = _ref9.dstX0,
          dstY0 = _ref9.dstY0,
          dstX1 = _ref9.dstX1,
          dstY1 = _ref9.dstY1,
          mask = _ref9.mask,
          _ref9$filter = _ref9.filter,
          filter = _ref9$filter === undefined ? GL.NEAREST : _ref9$filter;
      var gl = this.gl;

      assertWebGL2Context(gl);
      gl.blitFramebuffer(srcX0, srcY0, srcX1, srcY1, dstX0, dstY0, dstX1, dstY1, mask, filter);
      return this;
    }
  }, {
    key: 'textureLayer',
    value: function textureLayer() {
      var _ref10 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref10$target = _ref10.target,
          target = _ref10$target === undefined ? GL.FRAMEBUFFER : _ref10$target,
          attachment = _ref10.attachment,
          texture = _ref10.texture,
          level = _ref10.level,
          layer = _ref10.layer;

      var gl = this.gl;

      assertWebGL2Context(gl);
      gl.framebufferTextureLayer(target, attachment, texture, level, layer);
      return this;
    }
  }, {
    key: 'invalidate',
    value: function invalidate(_ref11) {
      var _ref11$target = _ref11.target,
          target = _ref11$target === undefined ? GL.FRAMEBUFFER : _ref11$target,
          _ref11$attachments = _ref11.attachments,
          attachments = _ref11$attachments === undefined ? [] : _ref11$attachments;
      var gl = this.gl;

      assertWebGL2Context(gl);
      gl.invalidateFramebuffer(target, attachments);
      return this;
    }
  }, {
    key: 'invalidateSub',
    value: function invalidateSub(_ref12) {
      var _ref12$target = _ref12.target,
          target = _ref12$target === undefined ? GL.FRAMEBUFFER : _ref12$target,
          _ref12$attachments = _ref12.attachments,
          attachments = _ref12$attachments === undefined ? [] : _ref12$attachments,
          _ref12$x = _ref12.x,
          x = _ref12$x === undefined ? 0 : _ref12$x,
          _ref12$y = _ref12.y,
          y = _ref12$y === undefined ? 0 : _ref12$y,
          width = _ref12.width,
          height = _ref12.height;
      var gl = this.gl;

      assertWebGL2Context(gl);
      gl.invalidateFramebuffer(target, attachments, x, y, width, height);
      return this;
    }

    // Selects a color buffer as the source for pixels for subsequent calls to
    // copyTexImage2D, copyTexSubImage2D, copyTexSubImage3D or readPixels.
    // src
    //  gl.BACK: Reads from the back color buffer.
    //  gl.NONE: Reads from no color buffer.
    //  gl.COLOR_ATTACHMENT{0-15}: Reads from one of 16 color attachment buffers.

  }, {
    key: 'readBuffer',
    value: function readBuffer(_ref13) {
      var src = _ref13.src;
      var gl = this.gl;

      assertWebGL2Context(gl);
      gl.readBuffer(src);
      return this;
    }

    // @returns {GLint}

  }, {
    key: 'alphaSize',
    value: function alphaSize() {
      return this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_ALPHA_SIZE);
    }

    // @returns {GLint}

  }, {
    key: 'blueSize',
    value: function blueSize() {
      return this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_BLUE_SIZE);
    }

    // @returns {GLenum}

  }, {
    key: 'colorEncoding',
    value: function colorEncoding() {
      return this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING);
    }

    // @returns {GLenum}

  }, {
    key: 'componentType',
    value: function componentType() {
      return this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE);
    }

    // @returns {GLint}

  }, {
    key: 'depthSize',
    value: function depthSize() {
      return this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_DEPTH_SIZE);
    }

    // @returns {GLint}

  }, {
    key: 'greenSize',
    value: function greenSize() {
      return this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_GREEN_SIZE);
    }

    // @returns {WebGLRenderbuffer|WebGLTexture}

  }, {
    key: 'objectName',
    value: function objectName() {
      return this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_OBJECT_NAME);
    }

    // @returns {GLenum}

  }, {
    key: 'objectType',
    value: function objectType() {
      return this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE);
    }

    // @returns {GLint}

  }, {
    key: 'redSize',
    value: function redSize() {
      return this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_RED_SIZE);
    }

    // @returns {GLint}

  }, {
    key: 'stencilSize',
    value: function stencilSize() {
      return this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_STENCIL_SIZE);
    }

    // @returns {GLint}

  }, {
    key: 'cubeMapFace',
    value: function cubeMapFace() {
      return this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE);
    }

    // @returns {GLint}

  }, {
    key: 'layer',
    value: function layer() {
      return this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER);
    }

    // @returns {GLint}

  }, {
    key: 'level',
    value: function level() {
      return this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL);
    }
  }, {
    key: 'getParameters',
    value: function getParameters() {
      return {
        redSize: this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_RED_SIZE),
        greenSize: this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_GREEN_SIZE),
        blueSize: this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_BLUE_SIZE),
        alphaSize: this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_ALPHA_SIZE),

        depthSize: this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_DEPTH_SIZE),
        stencilSize: this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_STENCIL_SIZE),

        colorEncoding: this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING),
        componentType: this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE),
        objectName: this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_OBJECT_NAME),
        objectType: this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE),
        cubeMapFace: this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE),
        layer: this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER),
        level: this.getAttachmentParameter(GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL)
      };
    }

    // (OpenGL ES 3.0.4 §6.1.13, similar to glGetFramebufferAttachmentParameteriv)
    // Return the value for the passed pname given the target and attachment.
    // The type returned is the natural type for the requested pname:
    // pname returned type
    // FRAMEBUFFER_ATTACHMENT_ALPHA_SIZE GLint
    // FRAMEBUFFER_ATTACHMENT_BLUE_SIZE  GLint
    // FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING GLenum
    // FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE GLenum
    // FRAMEBUFFER_ATTACHMENT_DEPTH_SIZE GLint
    // FRAMEBUFFER_ATTACHMENT_GREEN_SIZE GLint
    // FRAMEBUFFER_ATTACHMENT_OBJECT_NAME  WebGLRenderbuffer or WebGLTexture
    // FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE  GLenum
    // FRAMEBUFFER_ATTACHMENT_RED_SIZE GLint
    // FRAMEBUFFER_ATTACHMENT_STENCIL_SIZE GLint
    // FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE  GLint
    // FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER  GLint
    // FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL  GLint
    // If pname is not in the table above, generates an INVALID_ENUM error.
    // If an OpenGL error is generated, returns null.

  }, {
    key: 'getAttachmentParameter',
    value: function getAttachmentParameter() {
      var _ref14 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          pname = _ref14.pname,
          target = _ref14.target,
          attachment = _ref14.attachment;

      var gl = this.gl;

      assertWebGL2Context(gl);
      var value = gl.getFramebufferAttachmentParameter(target, attachment, pname);
      return value;
    }

    /* eslint-disable max-len */

  }, {
    key: '_getFrameBufferStatus',
    value: function _getFrameBufferStatus(status) {
      var error = void 0;
      switch (status) {
        case GL.FRAMEBUFFER_COMPLETE:
          error = 'Success. Framebuffer is correctly set up';
          break;
        case GL.FRAMEBUFFER_INCOMPLETE_ATTACHMENT:
          error = 'The attachment types are mismatched or not all framebuffer attachment points are framebuffer attachment complete.';
          break;
        case GL.FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT:
          error = 'There is no attachment.';
          break;
        case GL.FRAMEBUFFER_INCOMPLETE_DIMENSIONS:
          error = 'Height and width of the attachment are not the same.';
          break;
        case GL.FRAMEBUFFER_UNSUPPORTED:
          error = 'The format of the attachment is not supported or if depth and stencil attachments are not the same renderbuffer.';
          break;
        // When using a WebGL 2 context, the following values can be returned
        case GL.FRAMEBUFFER_INCOMPLETE_MULTISAMPLE:
          error = 'The values of GL.RENDERBUFFER_SAMPLES are different among attached renderbuffers, or are non-zero if the attached images are a mix of renderbuffers and textures.';
          break;
        default:
          error = 'Framebuffer error ' + status;
          break;
      }
      return error;
    }
    /* eslint-enable max-len */

  }]);

  return Framebuffer;
}();

export { Framebuffer as default };
//# sourceMappingURL=data:application/json;base64,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