var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

import { GL } from './webgl';
import { assertWebGLContext, assertWebGL2Context } from './webgl-checks';
import * as VertexAttributes from './vertex-attributes';
import Buffer from './buffer';
import Texture from './texture';
import { parseUniformName, getUniformSetter } from './uniforms';
import { VertexShader, FragmentShader } from './shader';
import { log, uid } from '../utils';
import assert from 'assert';

var Program = function () {
  _createClass(Program, null, [{
    key: 'makeFrom',


    /**
     * Returns a Program wrapped WebGLProgram from a variety of inputs.
     * Allows other functions to transparently accept raw WebGLPrograms etc
     * and manipulate them using the methods in the `Program` class.
     * Checks for ".handle"
     *
     * @param {WebGLRenderingContext} gl - if a new buffer needs to be initialized
     * @param {*} object - candidate that will be coerced to a buffer
     * @returns {Program} - Program object that wraps the buffer parameter
     */
    value: function makeFrom(gl) {
      var object = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      return object instanceof Program ? object :
      // Use .handle if available, else use 'program' directly
      new Program(gl).setData({ handle: object.handle || object });
    }

    /*
     * @classdesc
     * Handles creation of programs, mapping of attributes and uniforms
     *
     * @class
     * @param {WebGLRenderingContext} gl - gl context
     * @param {Object} opts - options
     * @param {String} opts.vs - Vertex shader source
     * @param {String} opts.fs - Fragment shader source
     * @param {String} opts.id= - Id
     */
    /* eslint-disable max-statements */

  }]);

  function Program(gl) {
    var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
        id = _ref.id,
        vs = _ref.vs,
        fs = _ref.fs,
        defaultUniforms = _ref.defaultUniforms,
        handle = _ref.handle;

    _classCallCheck(this, Program);

    assertWebGLContext(gl);

    // Create shaders if needed
    this.vs = typeof vs === 'string' ? new VertexShader(gl, vs) : vs;
    this.fs = typeof vs === 'string' ? new FragmentShader(gl, fs) : fs;

    assert(this.vs instanceof VertexShader, 'Program: bad vertex shader');
    assert(this.fs instanceof FragmentShader, 'Program: bad fragment shader');

    // If program is not named, name it after shader names
    var programName = this.vs.getName() || this.fs.getName();
    programName = programName ? programName + '-program' : 'program';
    this.id = id || uid(programName);

    this.gl = gl;
    this.defaultUniforms = defaultUniforms;
    this.handle = handle;
    if (!this.handle) {
      this.handle = gl.createProgram();
      this._compileAndLink();
    }
    if (!this.handle) {
      throw new Error('Failed to create program');
    }

    // determine attribute locations (i.e. indices)
    this._attributeLocations = this._getAttributeLocations();
    this._attributeCount = this.getAttributeCount();
    this._warn = [];
    this._filledLocations = {};

    // prepare uniform setters
    this._uniformSetters = this._getUniformSetters();
    this._uniformCount = this.getUniformCount();
    this._textureIndexCounter = 0;

    this.userData = {};
    Object.seal(this);
  }
  /* eslint-enable max-statements */

  _createClass(Program, [{
    key: 'delete',
    value: function _delete() {
      var gl = this.gl;

      if (this.handle) {
        gl.deleteProgram(this.handle);
      }
      this.handle = null;
      return this;
    }
  }, {
    key: 'use',
    value: function use() {
      var gl = this.gl;

      gl.useProgram(this.handle);
      return this;
    }

    // DEPRECATED METHODS

  }, {
    key: 'clearBuffers',
    value: function clearBuffers() {
      this._filledLocations = {};
      return this;
    }
  }, {
    key: '_print',
    value: function _print(bufferName) {
      return 'Program ' + this.id + ': Attribute ' + bufferName;
    }

    /**
     * Attach a map of Buffers values to a program
     * Only attributes with names actually present in the linked program
     * will be updated. Other supplied buffers will be ignored.
     *
     * @param {Object} buffers - An object map with attribute names being keys
     *  and values are expected to be instances of Buffer.
     * @returns {Program} Returns itself for chaining.
     */
    /* eslint-disable max-statements */

  }, {
    key: 'setBuffers',
    value: function setBuffers(buffers) {
      var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
          _ref2$clear = _ref2.clear,
          clear = _ref2$clear === undefined ? true : _ref2$clear,
          _ref2$check = _ref2.check,
          check = _ref2$check === undefined ? true : _ref2$check,
          _ref2$drawParams = _ref2.drawParams,
          drawParams = _ref2$drawParams === undefined ? {} : _ref2$drawParams;

      var gl = this.gl;

      if (Array.isArray(buffers)) {
        throw new Error('Program.setBuffers expects map of buffers');
      }

      if (clear) {
        this.clearBuffers();
      }

      // indexing is autodetected - buffer with target gl.ELEMENT_ARRAY_BUFFER
      // index type is saved for drawElement calls
      drawParams.isInstanced = false;
      drawParams.isIndexed = false;
      drawParams.indexType = null;

      var _sortBuffersByLocatio = this._sortBuffersByLocation(buffers),
          locations = _sortBuffersByLocatio.locations,
          elements = _sortBuffersByLocatio.elements;

      // Process locations in order


      for (var location = 0; location < locations.length; ++location) {
        var bufferName = locations[location];
        var buffer = buffers[bufferName];
        // DISABLE MISSING ATTRIBUTE
        if (!buffer) {
          VertexAttributes.disable(gl, location);
        } else {
          var divisor = buffer.layout.instanced ? 1 : 0;
          VertexAttributes.enable(gl, location);
          VertexAttributes.setBuffer({ gl: gl, location: location, buffer: buffer });
          VertexAttributes.setDivisor(gl, location, divisor);
          drawParams.isInstanced = buffer.layout.instanced > 0;
          this._filledLocations[bufferName] = true;
        }
      }

      // SET ELEMENTS ARRAY BUFFER
      if (elements) {
        var _buffer = buffers[elements];
        _buffer.bind();
        drawParams.isIndexed = true;
        drawParams.indexType = _buffer.layout.type;
      }

      if (check) {
        this.checkBuffers();
      }

      return this;
    }
    /* eslint-enable max-statements */

  }, {
    key: 'checkBuffers',
    value: function checkBuffers() {
      for (var attributeName in this._attributeLocations) {
        if (!this._filledLocations[attributeName] && !this._warn[attributeName]) {
          var location = this._attributeLocations[attributeName];
          // throw new Error(`Program ${this.id}: ` +
          //   `Attribute ${location}:${attributeName} not supplied`);
          log.warn(0, 'Program ' + this.id + ': ' + ('Attribute ' + location + ':' + attributeName + ' not supplied'));
          this._warn[attributeName] = true;
        }
      }
      return this;
    }

    /*
     * @returns {Program} Returns itself for chaining.
     */

  }, {
    key: 'unsetBuffers',
    value: function unsetBuffers() {
      var gl = this.gl;

      var length = this._attributeCount;
      for (var i = 1; i < length; ++i) {
        // VertexAttributes.setDivisor(gl, i, 0);
        VertexAttributes.disable(gl, i);
      }
      gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, null);
      return this;
    }

    /**
     * Apply a set of uniform values to a program
     * Only uniforms with names actually present in the linked program
     * will be updated.
     * other uniforms will be ignored
     *
     * @param {Object} uniformMap - An object with names being keys
     * @returns {Program} - returns itself for chaining.
     */
    /* eslint-disable max-depth */

  }, {
    key: 'setUniforms',
    value: function setUniforms(uniforms) {
      for (var uniformName in uniforms) {
        var uniform = uniforms[uniformName];
        var uniformSetter = this._uniformSetters[uniformName];
        if (uniformSetter) {
          if (uniform instanceof Texture) {
            if (uniformSetter.textureIndex === undefined) {
              uniformSetter.textureIndex = this._textureIndexCounter++;
            }
            // Bind texture to index, and set the uniform sampler to the index
            var texture = uniform;
            var textureIndex = uniformSetter.textureIndex;
            // console.debug('setting texture', textureIndex, texture);

            texture.bind(textureIndex);
            uniformSetter(textureIndex);
          } else {
            // Just set the value
            uniformSetter(uniform);
          }
        }
      }
      return this;
    }
    /* eslint-enable max-depth */

    // RAW WEBGL METHODS

  }, {
    key: 'getAttachedShadersCount',
    value: function getAttachedShadersCount() {
      return this.getProgramParameter(GL.ATTACHED_SHADERS);
    }

    // ATTRIBUTES API
    // Note: Locations are numeric indices

  }, {
    key: 'getAttributeCount',
    value: function getAttributeCount() {
      return this.getProgramParameter(GL.ACTIVE_ATTRIBUTES);
    }

    /**
     * Returns an object with info about attribute at index "location"/
     * @param {int} location - index of an attribute
     * @returns {WebGLActiveInfo} - info about an active attribute
     *   fields: {name, size, type}
     */

  }, {
    key: 'getAttributeInfo',
    value: function getAttributeInfo(location) {
      return this.gl.getActiveAttrib(this.handle, location);
    }
  }, {
    key: 'getAttributeName',
    value: function getAttributeName(location) {
      return this.getAttributeInfo(location).name;
    }

    /**
     * Returns location (index) of a name
     * @param {String} attributeName - name of an attribute
     *   (matches name in a linked shader)
     * @returns {String[]} - array of actual attribute names from shader linking
     */

  }, {
    key: 'getAttributeLocation',
    value: function getAttributeLocation(attributeName) {
      return this.gl.getAttribLocation(this.handle, attributeName);
    }

    // UNIFORMS API
    // Note: locations are opaque structures

  }, {
    key: 'getUniformCount',
    value: function getUniformCount() {
      return this.getProgramParameter(GL.ACTIVE_UNIFORMS);
    }

    /*
     * @returns {WebGLActiveInfo} - object with {name, size, type}
     */

  }, {
    key: 'getUniformInfo',
    value: function getUniformInfo(index) {
      return this.gl.getActiveUniform(this.handle, index);
    }

    /*
     * @returns {WebGLUniformLocation} - opaque object representing location
     * of uniform, used by setter methods
     */

  }, {
    key: 'getUniformLocation',
    value: function getUniformLocation(name) {
      return this.gl.getUniformLocation(this.handle, name);
    }
  }, {
    key: 'getUniformValue',
    value: function getUniformValue(location) {
      return this.gl.getUniform(this.handle, location);
    }

    // PROGRAM API

  }, {
    key: 'isFlaggedForDeletion',
    value: function isFlaggedForDeletion() {
      return this.getProgramParameter(GL.DELETE_STATUS);
    }
  }, {
    key: 'getLastLinkStatus',
    value: function getLastLinkStatus() {
      return this.getProgramParameter(GL.LINK_STATUS);
    }
  }, {
    key: 'getLastValidationStatus',
    value: function getLastValidationStatus() {
      return this.getProgramParameter(GL.VALIDATE_STATUS);
    }

    // WEBGL2 INTERFACE

    // This may be gl.SEPARATE_ATTRIBS or gl.INTERLEAVED_ATTRIBS.

  }, {
    key: 'getTransformFeedbackBufferMode',
    value: function getTransformFeedbackBufferMode() {
      assertWebGL2Context(this.gl);
      return this.getProgramParameter(this.gl.TRANSFORM_FEEDBACK_BUFFER_MODE);
    }
  }, {
    key: 'getTransformFeedbackVaryingsCount',
    value: function getTransformFeedbackVaryingsCount() {
      assertWebGL2Context(this.gl);
      return this.getProgramParameter(this.gl.TRANSFORM_FEEDBACK_VARYINGS);
    }
  }, {
    key: 'getActiveUniformBlocksCount',
    value: function getActiveUniformBlocksCount() {
      assertWebGL2Context(this.gl);
      return this.getProgramParameter(this.gl.ACTIVE_UNIFORM_BLOCKS);
    }

    // Retrieves the assigned color number binding for the user-defined varying
    // out variable name for program. program must have previously been linked.
    // [WebGLHandlesContextLoss]

  }, {
    key: 'getFragDataLocation',
    value: function getFragDataLocation(varyingName) {
      assertWebGL2Context(this.gl);
      return this.gl.getFragDataLocation(this.handle, varyingName);
    }

    // Return the value for the passed pname given the passed program.
    // The type returned is the natural type for the requested pname,
    // as given in the following table:
    // pname returned type
    // DELETE_STATUS GLboolean
    // LINK_STATUS GLboolean
    // VALIDATE_STATUS GLboolean
    // ATTACHED_SHADERS  GLint
    // ACTIVE_ATTRIBUTES GLint
    // ACTIVE_UNIFORMS GLint
    // TRANSFORM_FEEDBACK_BUFFER_MODE  GLenum
    // TRANSFORM_FEEDBACK_VARYINGS GLint
    // ACTIVE_UNIFORM_BLOCKS GLint

  }, {
    key: 'getProgramParameter',
    value: function getProgramParameter(pname) {
      return this.gl.getProgramParameter(this.handle, pname);
    }

    // PRIVATE METHODS

  }, {
    key: '_compileAndLink',
    value: function _compileAndLink() {
      var gl = this.gl;

      gl.attachShader(this.handle, this.vs.handle);
      gl.attachShader(this.handle, this.fs.handle);
      gl.linkProgram(this.handle);
      // Program linking error is checked only when debug context is used
      if (gl.debug) {
        gl.validateProgram(this.handle);
        var linked = gl.getProgramParameter(this.handle, gl.LINK_STATUS);
        if (!linked) {
          throw new Error('Error linking ' + gl.getProgramInfoLog(this.handle));
        }
      }
    }
  }, {
    key: '_sortBuffersByLocation',
    value: function _sortBuffersByLocation(buffers) {
      var elements = null;
      var locations = new Array(this._attributeCount);

      for (var bufferName in buffers) {
        var buffer = Buffer.makeFrom(this.gl, buffers[bufferName]);
        var location = this._attributeLocations[bufferName];
        if (location === undefined) {
          if (buffer.target === GL.ELEMENT_ARRAY_BUFFER && elements) {
            throw new Error(this._print(bufferName) + ' duplicate gl.ELEMENT_ARRAY_BUFFER');
          } else if (buffer.target === GL.ELEMENT_ARRAY_BUFFER) {
            elements = bufferName;
          } else if (!this._warn[bufferName]) {
            log.warn(2, this._print(bufferName) + ' not used');
            this._warn[bufferName] = true;
          }
        } else {
          if (buffer.target === GL.ELEMENT_ARRAY_BUFFER) {
            throw new Error(this._print(bufferName) + ':' + location + ' ' + 'has both location and type gl.ELEMENT_ARRAY_BUFFER');
          }
          locations[location] = bufferName;
        }
      }

      return { locations: locations, elements: elements };
    }

    // Check that all active attributes are enabled

  }, {
    key: '_areAllAttributesEnabled',
    value: function _areAllAttributesEnabled() {
      var gl = this.gl;

      var length = this._attributeCount;
      for (var i = 0; i < length; ++i) {
        if (!VertexAttributes.isEnabled(gl, i)) {
          return false;
        }
      }
      return true;
    }

    // determine attribute locations (maps attribute name to index)

  }, {
    key: '_getAttributeLocations',
    value: function _getAttributeLocations() {
      var attributeLocations = {};
      var length = this.getAttributeCount();
      for (var location = 0; location < length; location++) {
        var name = this.getAttributeName(location);
        attributeLocations[name] = this.getAttributeLocation(name);
      }
      return attributeLocations;
    }

    // create uniform setters
    // Map of uniform names to setter functions

  }, {
    key: '_getUniformSetters',
    value: function _getUniformSetters() {
      var gl = this.gl;

      var uniformSetters = {};
      var length = this.getUniformCount();
      for (var i = 0; i < length; i++) {
        var info = this.getUniformInfo(i);
        var parsedName = parseUniformName(info.name);
        var location = this.getUniformLocation(parsedName.name);
        uniformSetters[parsedName.name] = getUniformSetter(gl, location, info, parsedName.isArray);
      }
      return uniformSetters;
    }
  }]);

  return Program;
}();

// create uniform setters
// Map of uniform names to setter functions


export { Program as default };
export function getUniformDescriptors(gl, program) {
  var uniformDescriptors = {};
  var length = program.getUniformCount();
  for (var i = 0; i < length; i++) {
    var info = program.getUniformInfo(i);
    var location = program.getUniformLocation(info.name);
    var descriptor = getUniformSetter(gl, location, info);
    uniformDescriptors[descriptor.name] = descriptor;
  }
  return uniformDescriptors;
}
//# sourceMappingURL=data:application/json;base64,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