var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

// WebGL2 VertexArray Objects Helper
import { assertWebGLContext, glCheckError } from '../webgl/webgl-checks';
import queryManager from './helpers/query-manager';

/* eslint-disable no-multi-spaces */
var GL_QUERY_COUNTER_BITS_EXT = 0x8864;
var GL_QUERY_RESULT_EXT = 0x8866;
var GL_QUERY_RESULT_AVAILABLE_EXT = 0x8867;
var GL_TIME_ELAPSED_EXT = 0x88BF;
var GL_TIMESTAMP_EXT = 0x8E28;
var GL_GPU_DISJOINT_EXT = 0x8FBB;
/* eslint-enable no-multi-spaces */

var noop = function noop(x) {
  return x;
};

var ERR_GPU_DISJOINT = 'Disjoint GPU operation invalidated timer queries';
var ERR_TIMER_QUERY_NOT_SUPPORTED = 'Timer queries require "EXT_disjoint_timer_query" extension';

var TimerQuery = function () {
  _createClass(TimerQuery, null, [{
    key: 'isSupported',

    /**
     * Returns true if TimerQuery is supported by the WebGL implementation
     * (depends on the EXT_disjoint_timer_query extension)/
     * Can also check whether timestamp queries are available.
     *
     * @param {WebGLRenderingContext} gl - gl context
     * @param {Object} opts= - options
     * @param {Object} opts.requireTimestamps=false -
     *   If true, checks if timestamps are supported
     * @return {Boolean} - TimerQueries are supported with specified configuration
     */
    value: function isSupported(gl) {
      var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
          _ref$requireTimestamp = _ref.requireTimestamps,
          requireTimestamps = _ref$requireTimestamp === undefined ? false : _ref$requireTimestamp;

      assertWebGLContext(gl);
      var ext = gl.getExtension('EXT_disjoint_timer_query');
      var queryCounterBits = ext ? ext.getQueryEXT(GL_TIMESTAMP_EXT, GL_QUERY_COUNTER_BITS_EXT) : 0;
      var timestampsSupported = queryCounterBits > 0;
      return Boolean(ext) && (!requireTimestamps || timestampsSupported);
    }

    /**
     * @classdesc
     * Provides a way to measure the duration of a set of GL commands,
     * without stalling the rendering pipeline.
     *
     * Exposes a `promise` member that tracks the state of the query
     * when `poll` is used to update queries.
     *
     * @example
        const timerQuery = new TimerQuery({
          onComplete: timestamp => console.log(timestamp)
          onError: error => console.warn(error)
        });
         timerQuery.begin();
         // Issue GPU calls
         timerQuery.end();
         // Poll for resolved queries
        requestAnimationFrame(() => TimerQuery.poll(gl))
     *
     * Remarks:
     * - On Chrome, go to chrome:flags and enable "WebGL Draft Extensions"
     *
     * - For full functionality, TimerQuery depends on a `poll` function being
     *   called regularly. When this is done, completed queries will be
     *   automatically detected and any callbacks called.
     *
     * - TimerQuery instance creation will always succeed, even when the required
     *   extension is not supported. Instead any issued queries will fail
     *   immediately. This allows applications to unconditionally use TimerQueries
     *   without adding logic to check whether they are supported; the
     *   difference being that the `onComplete` callback never gets called,
     *   (the `onError` callback, if supplied, will be called instead).
     *
     * - Even when supported, timer queries can fail whenever a change in the
     *   GPU occurs that will make the values returned by this extension unusable
     *   for performance metrics. Power conservation might cause the GPU to go to
     *   sleep at the lower levers. TimerQuery will detect this condition and
     *   fail any outstanding queries (which calls the `onError` function,
     *   if supplied).
     *
     * @class
     * @param {WebGLRenderingContext} gl - gl context
     * @param {Object} opts - options
     * @param {Function} opts.onComplete - called with a timestamp.
     *   Specifying this parameter causes a timestamp query to be initiated
     * @param {Function} opts.onComplete - called with a timestamp.
     *   Specifying this parameter causes a timestamp query to be initiated
     */

  }]);

  function TimerQuery(gl) {
    var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
        _ref2$onComplete = _ref2.onComplete,
        onComplete = _ref2$onComplete === undefined ? noop : _ref2$onComplete,
        _ref2$onError = _ref2.onError,
        onError = _ref2$onError === undefined ? noop : _ref2$onError;

    _classCallCheck(this, TimerQuery);

    assertWebGLContext(gl);

    this.gl = gl;
    this.ext = this.gl.getExtension('EXT_disjoint_timer_query');
    this.handle = this.ext ? this.ext.createQueryEXT() : null;
    this.target = null;
    this.userData = {};

    this.onComplete = onComplete;
    this.onError = onError;

    // query manager needs a promise field
    this.promise = null;

    Object.seal(this);
  }

  /**
   * Destroys the WebGL object
   * Rejects any pending query
   *
   * @return {TimerQuery} - returns itself, to enable chaining of calls.
   */


  _createClass(TimerQuery, [{
    key: 'delete',
    value: function _delete() {
      if (this.handle) {
        queryManager.deleteQuery(this);
        this.ext.deleteQueryEXT(this.handle);
        glCheckError(this.gl);
        this.handle = null;
      }
      return this;
    }

    /**
     * Measures GPU time delta between this call and a matching `end` call in the
     * GPU instruction stream.
     *
     * Remarks:
     * - Due to OpenGL API limitations, after calling `begin()` on one TimerQuery
     *   instance, `end()` must be called on that same instance before
     *   calling `begin()` on another query. While there can be multiple
     *   outstanding queries representing disjoint `begin()`/`end()` intervals.
     *   It is not possible to interleave or overlap `begin` and `end` calls.
     *
     * - Triggering a new query when a TimerQuery is already tracking an
     *   unresolved query causes that query to be cancelled.
     *
     * @return {TimerQuery} - returns itself, to enable chaining of calls.
     */

  }, {
    key: 'begin',
    value: function begin() {
      queryManager.beginQuery(this, this.onComplete, this.onError);
      if (this.ext) {
        this.target = GL_TIME_ELAPSED_EXT;
        this.ext.beginQueryEXT(this.target, this.handle);
      } else {
        queryManager.rejectQuery(this, ERR_TIMER_QUERY_NOT_SUPPORTED);
      }
      return this;
    }

    /**
     * Inserts a query end marker into the GPU instruction stream.
     * Can be called multiple times.
     *
     * @return {TimerQuery} - returns itself, to enable chaining of calls.
     */

  }, {
    key: 'end',
    value: function end() {
      // Note: calling end does not affect the pending promise
      if (this.target) {
        this.ext.endQueryEXT(this.target);
        this.target = null;
      }
      return this;
    }

    /**
     * Generates a GPU time stamp when the GPU instruction stream reaches
     * this instruction.
     * To measure time deltas, two queries are needed.
     *
     * Remarks:
     * - timestamp() queries may not be available even when the timer query
     *   extension is. See TimeQuery.isSupported() flags.
     *
     * - Triggering a new query when a TimerQuery is already tracking an
     *   unresolved query causes that query to be cancelled.
     *
     * @return {TimerQuery} - returns itself, to enable chaining of calls.
     */

  }, {
    key: 'timestamp',
    value: function timestamp() {
      queryManager.beginQuery(this, this.onComplete, this.onError);
      if (this.ext) {
        this.ext.queryCounterEXT(this.handle, GL_TIMESTAMP_EXT);
      } else {
        queryManager.rejectQuery(this, ERR_TIMER_QUERY_NOT_SUPPORTED);
      }
      return this;
    }

    /**
     * Cancels a pending query
     * Note - Cancel's the promise and calls end on the current query if needed.
     *
     * @return {TimerQuery} - returns itself, to enable chaining of calls.
     */

  }, {
    key: 'cancel',
    value: function cancel() {
      this.end();
      queryManager.cancelQuery(this);
      return this;
    }

    /**
     * @return {Boolean} - true if query result is available
     */

  }, {
    key: 'isResultAvailable',
    value: function isResultAvailable() {
      return this.ext && this.ext.getQueryObjectEXT(this.handle, GL_QUERY_RESULT_AVAILABLE_EXT);
    }

    /**
     * Returns the query result, converted to milliseconds to match
     * JavaScript conventions.
     *
     * @return {Number} - measured time or timestamp, in milliseconds
     */

  }, {
    key: 'getResult',
    value: function getResult() {
      return this.ext ? this.ext.getQueryObjectEXT(this.handle, GL_QUERY_RESULT_EXT) / 1e6 : 0;
    }
  }]);

  return TimerQuery;
}();

// NOTE: This call lets the queryManager know how to detect disjoint GPU state
// It will check dsjoint state on polls and before adding a new query
// and reject any outstanding TimerQueries with our supplied error message.


export { TimerQuery as default };
queryManager.setInvalidator({
  queryType: TimerQuery,
  errorMessage: ERR_GPU_DISJOINT,
  // Note: Querying the disjoint state resets it
  checkInvalid: function checkInvalid(gl) {
    return gl.getParameter(GL_GPU_DISJOINT_EXT);
  }
});
//# sourceMappingURL=data:application/json;base64,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