/* eslint-disable dot-notation*/
import { glGet } from './webgl';
import { assertWebGLContext, assertWebGL2Context, isWebGL2Context } from './webgl-checks';
import Buffer from './buffer';
import { log } from '../utils';
import assert from 'assert';

/**
 * Methods for manipulating the vertex attributes array, which is where
 * vertex data is staged for shader execution.
 *
 * Vertex attributes are stored in "arrays" with indices from 0 and up.
 * During shader execution, these indices (or 'locations') are matched to
 * the indices assigned to shader attributes during WebGLProgram linking.
 *
 * Note: The global state contains an implicit vertex attribute array which
 * the methods in this class manipulate by default. It is also possible to
 * create and bind a VertexArrayObject to manage multiple arrays.
 *
 * Each vertex attribute has these properties:
 * - Can be enabled or disabled (Only enable attrs actually used by a program)
 * - Has an instance `divisor` (usually 1 or 0 to enable/disable instancing)
 * - Have a size (1-4 values per vertex)
 * - Has a value or values that is accessible in shaders
 *
 * Attribute values are either
 * - Generic: a constant value for all vertices/instances, or
 * - Bound to a WebGLBuffer with unique values for each vertex/instance
 *
 * When binding to a WebGLBuffer it is necessary to specify the layout of
 * data in the buffer:
 * - size (1-4 values per vertex)
 * - data type (e.g. gl.FLOAT)
 * - stride, offset, and integer normalization policy can also be specified
 *
 * Note: All methods in this class take a `location` index to specify which
 * vertex attribute in the array they are operating on.
 *
 * Note: Attribute 0 can sometimes be treated specially by the driver,
 * to be safe we avoid disabling it.
 *
 * Note: WebGL2
 * - Improves support for integer attributes, both generic and buffered.
 * - Setting instance "divisors" no longer require using a WebGL extension.
 *
 */

function glGetLumaInfo(gl) {
  if (!gl.luma) {
    gl.luma = { extensions: {} };
  }
  if (gl.luma.extensions['ANGLE_instanced_arrays'] === undefined) {
    gl.luma.extensions['ANGLE_instanced_arrays'] = gl.getExtension('ANGLE_instanced_arrays');
  }
  return gl.luma;
}

// ACCESSORS

/**
 * The max number of attributes in the vertex attribute array is an
 * implementation defined limit, but never smaller than 8
 * @param {WebGLRenderingContext} gl - webgl context
 * @returns {GLuint} - (max) number of attributes in the vertex attribute array
 */
export function getMaxAttributes(gl) {
  assertWebGLContext(gl);
  return gl.getParameter(gl.MAX_VERTEX_ATTRIBS);
}

/**
 * Is instance divisor availble (checks for WebGL2 or ANGLE extension)
 * @param {WebGLRenderingContext} gl - webgl context
 * @returns {Boolean} - is divisor available?
 */
export function hasDivisor(gl) {
  assertWebGLContext(gl);
  return Boolean(isWebGL2Context(gl) || gl.getExtension(gl, 'ANGLE_instanced_arrays'));
}

/**
 * Returns true if the vertex attribute is enabled at this index.
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @returns {Boolean} - enabled status
 */
export function isEnabled(gl, location) {
  return Boolean(get(gl, location, gl.VERTEX_ATTRIB_ARRAY_ENABLED));
}

/**
 * Returns the currently bound buffer
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @returns {WebGLBuffer} Returns the currently bound buffer
 */
export function getBuffer(gl, location) {
  return get(gl, location, gl.VERTEX_ATTRIB_ARRAY_BUFFER_BINDING);
}

/**
 * Get values for generic vertex attributes
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @returns {Float32Array} (with 4 elements) representing the current value
 * of the vertex attribute at the given index.
 */
export function getGeneric(gl, location) {
  return get(gl, gl.CURRENT_VERTEX_ATTRIB);
}

/**
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 */
// @returns {GLint} the size of an element of the vertex array.
export function getSize(gl, location) {
  return get(location, gl.VERTEX_ATTRIB_ARRAY_SIZE);
}

/**
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 */
// @returns {GLenum} representing the array type.
export function getType(gl, location) {
  return get(location, gl.VERTEX_ATTRIB_ARRAY_TYPE);
}

/**
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 */
// @returns {GLboolean} true if fixed-point data types are normalized
// for the vertex attribute array at the given index.
export function isNormalized(gl, location) {
  return get(location, gl.VERTEX_ATTRIB_ARRAY_NORMALIZED);
}

/**
 * check if an integer data type in the vertex attribute at index
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - index of the vertex attribute.
 * @returns {GLboolean} - true if an integer data type is in the
 * vertex attribute array at the given index.
 */
export function isInteger(gl, location) {
  assertWebGL2Context(gl);
  return get(location, gl.VERTEX_ATTRIB_ARRAY_INTEGER);
}

/**
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @returns {GLint} number of bytes between successive elements in the array.
 * 0 means that the elements are sequential.
 */
export function getStride(gl, location) {
  return get(location, gl.VERTEX_ATTRIB_ARRAY_STRIDE);
}

/**
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @param {GLuint} pname - enum specifying which offset to return
 * @returns {GLuint} the address of a specified vertex attribute.
 */
export function getOffset(gl, location) {
  var pname = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : gl.VERTEX_ATTRIB_ARRAY_POINTER;

  return gl.getVertexAttribOffset(location, pname);
}

/**
 * @private
 * Generic getter for information about a vertex attribute at a given position
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - index of the vertex attribute.
 * @param {GLenum} pname - specifies the information to query.
 * @returns {*} - requested vertex attribute information (specified by pname)
 */
function get(gl, location, pname) {
  assertWebGLContext(gl);
  return gl.getVertexAttrib(location, pname);
}

// MODIFIERS

/**
 * Enable the attribute
 * Note: By default all attributes are disabled. Only attributes
 * used by a program's shaders should be enabled.
 *
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 */
export function enable(gl, location) {
  gl.enableVertexAttribArray(location);
}

/**
 * Disable the attribute
 * Note: Only attributes used by a program's shaders should be enabled.
 *
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 */
export function disable(gl, location) {
  // Don't disable location 0
  if (location > 0) {
    gl.disableVertexAttribArray(location);
  }
}

/**
 * Set the frequency divisor used for instanced rendering.
 * Note: Usually simply set to 1 or 0 to enable/disable instanced rendering
 * for a specific attribute.
 *
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @param {GLuint} divisor - instances that pass between updates of attribute
 */
export function setDivisor(gl, location, divisor) {
  if (isWebGL2Context(gl)) {
    gl.vertexAttribDivisor(location, divisor);
    return;
  }
  var ext = glGetLumaInfo(gl).extensions['ANGLE_instanced_arrays'];
  if (ext) {
    ext.vertexAttribDivisorANGLE(location, divisor);
    return;
  }
  // Accept divisor 0 even if instancing is not supported (0 = no instancing)
  if (divisor !== 0) {
    throw new Error('WebGL instanced rendering not supported');
  }
}

/**
 * Returns the frequency divisor used for instanced rendering.
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @returns {GLuint} divisor
 */
export function getDivisor(gl, location) {
  assert(location > 0);
  if (isWebGL2Context(gl)) {
    var divisor = get(location, gl.VERTEX_ATTRIB_ARRAY_DIVISOR);
    return divisor;
  }
  var ext = glGetLumaInfo(gl).extensions['ANGLE_instanced_arrays'];
  if (ext) {
    var _divisor = get(location, ext.VERTEX_ATTRIB_ARRAY_DIVISOR_ANGLE);
    return _divisor;
  }
  // if instancing is not available, return 0 meaning divisor has not been set
  return 0;
}

/**
 * Set a location in vertex attributes array to a buffer, specifying
 * its data layout and integer to float conversion and normalization flags
 *
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @param {WebGLBuffer|Buffer} buffer - WebGL buffer to set as value
 * @param {GLuint} target=gl.ARRAY_BUFFER - which target to bind to
 * @param {Object} layout= Optional data layout, defaults to buffer's layout
 * @param {GLuint} layout.size - number of values per element (1-4)
 * @param {GLuint} layout.type - type of values (e.g. gl.FLOAT)
 * @param {GLbool} layout.normalized=false - normalize integers to [-1,1], [0,1]
 * @param {GLuint} layout.integer=false - WebGL2 only, disable int-to-float conv
 * @param {GLuint} layout.stride=0 - supports strided arrays
 * @param {GLuint} layout.offset=0 - supports strided arrays
 */
export function setBuffer() {
  var _ref = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
      gl = _ref.gl,
      location = _ref.location,
      buffer = _ref.buffer,
      target = _ref.target,
      layout = _ref.layout;

  assertWebGLContext(gl);
  buffer = Buffer.makeFrom(gl, buffer);

  // Copy main data characteristics from buffer
  target = glGet(target !== undefined ? target : buffer.target);
  layout = layout !== undefined ? layout : buffer.layout;
  assert(target, 'setBuffer needs target');
  assert(layout, 'setBuffer called on uninitialized buffer');

  // a non-zero named buffer object must be bound to the GL_ARRAY_BUFFER target
  buffer.bind({ target: gl.ARRAY_BUFFER });

  // Attach bound ARRAY_BUFFER with specified buffer format to location
  if (!layout.integer) {
    gl.vertexAttribPointer(location, layout.size, glGet(layout.type), layout.normalized, layout.stride, layout.offset);
  } else {
    // specifies *integer* data formats and locations of vertex attributes
    // For glVertexAttribIPointer, Values are always left as integer values.
    // Only accepts the integer types gl.BYTE, gl.UNSIGNED_BYTE,
    // gl.SHORT, gl.UNSIGNED_SHORT, gl.INT, gl.UNSIGNED_INT
    assertWebGL2Context(gl);
    gl.vertexAttribIPointer(location, layout.size, glGet(layout.type), layout.stride, layout.offset);
  }

  buffer.unbind({ target: gl.ARRAY_BUFFER });
}

/*
 * Specify values for generic vertex attributes
 * Generic vertex attributes are constant for all vertices
 * Up to 4 values depending on attribute size
 *
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @param {GLuint} divisor - instances that pass between updates of attribute
 */
export function setGeneric(_ref2) {
  var gl = _ref2.gl,
      location = _ref2.location,
      array = _ref2.array;

  log.warn(0, 'VertexAttributes.setGeneric is not well tested');
  // throw new Error('vertex attribute size must be between 1 and 4');

  if (array instanceof Float32Array) {
    gl.vertexAttrib4fv(location, array);
  } else if (array instanceof Int32Array) {
    assertWebGL2Context(gl);
    gl.vertexAttribI4iv(location, array);
  } else if (array instanceof Uint32Array) {
    assertWebGL2Context(gl);
    gl.vertexAttribI4uiv(location, array);
  }
}

/*
 * Specify values for generic vertex attributes
 * Generic vertex attributes are constant for all vertices
 * Up to 4 values depending on attribute size
 *
 * @param {GLuint} location - ordinal number of the attribute
 * @param {GLuint} divisor - instances that pass between updates of attribute
 */
/* eslint-disable max-params */
export function setGenericValues(gl, location, v0, v1, v2, v3) {
  log.warn(0, 'VertexAttributes.setGenericValues is not well tested');
  switch (arguments.length - 1) {
    case 1:
      gl.vertexAttrib1f(location, v0);break;
    case 2:
      gl.vertexAttrib2f(location, v0, v1);break;
    case 3:
      gl.vertexAttrib3f(location, v0, v1, v2);break;
    case 4:
      gl.vertexAttrib4f(location, v0, v1, v2, v3);break;
    default:
      throw new Error('vertex attribute size must be between 1 and 4');
  }

  // assert(gl instanceof WebGL2RenderingContext, 'WebGL2 required');
  // Looks like these will check how many arguments were supplied?
  // gl.vertexAttribI4i(location, v0, v1, v2, v3);
  // gl.vertexAttribI4ui(location, v0, v1, v2, v3);
}
//# sourceMappingURL=data:application/json;base64,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