// Helper definitions for validation of webgl parameters
/* eslint-disable no-inline-comments, max-len */
import { WebGLRenderingContext, WebGL2RenderingContext } from './webgl-types';
import GL, { glKey } from './webgl-constants';
import { glTypeFromArray } from './webgl-arrays';
import assert from 'assert';

var ERR_CONTEXT = 'Invalid WebGLRenderingContext';
var ERR_WEBGL2 = 'Requires WebGL2';

export function isWebGLContext(gl) {
  return gl instanceof WebGLRenderingContext || gl && gl.ARRAY_BUFFER === 0x8892;
}

export function isWebGL2Context(gl) {
  return gl instanceof WebGL2RenderingContext || gl && gl.TEXTURE_BINDING_3D === 0x806A;
}

export function assertWebGLContext(gl) {
  // Need to handle debug context
  assert(isWebGLContext(gl), ERR_CONTEXT);
}

export function assertWebGL2Context(gl) {
  // Need to handle debug context
  assert(isWebGL2Context(gl), ERR_WEBGL2);
}

// INDEX TYPES

// For drawElements, size of indices
export var GL_INDEX_TYPES = [GL.UNSIGNED_BYTE, GL.UNSIGNED_SHORT, GL.UNSIGNED_INT];

export function isIndexType(type) {
  return GL_INDEX_TYPES.indexOf(type) !== -1;
}

export function assertIndexType(glType, source) {
  assert(isIndexType(glType), 'Bad index type ' + glKey(glType) + ' ' + source);
}

// DRAW MODES

export var GL_DRAW_MODES = [GL.POINTS, GL.LINE_STRIP, GL.LINE_LOOP, GL.LINES, GL.TRIANGLE_STRIP, GL.TRIANGLE_FAN, GL.TRIANGLES];

export function isDrawMode(glMode) {
  return GL_DRAW_MODES.indexOf(glMode) !== -1;
}

export function assertDrawMode(glType, source) {
  assert(isDrawMode(glType), 'Bad draw mode ' + glKey(glType) + ' ' + source);
}

// TARGET TYPES

export var GL_TARGETS = [GL.ARRAY_BUFFER, // vertex attributes (e.g. vertex/texture coords or color)
GL.ELEMENT_ARRAY_BUFFER, // Buffer used for element indices.
// For WebGL 2 contexts
GL.COPY_READ_BUFFER, // Buffer for copying from one buffer object to another
GL.COPY_WRITE_BUFFER, // Buffer for copying from one buffer object to another
GL.TRANSFORM_FEEDBACK_BUFFER, // Buffer for transform feedback operations
GL.UNIFORM_BUFFER, // Buffer used for storing uniform blocks
GL.PIXEL_PACK_BUFFER, // Buffer used for pixel transfer operations
GL.PIXEL_UNPACK_BUFFER // Buffer used for pixel transfer operations
];

// USAGE TYPES

export var GL_BUFFER_USAGE = [GL.STATIC_DRAW, // Buffer used often and not change often. Contents are written to the buffer, but not read.
GL.DYNAMIC_DRAW, // Buffer used often and change often. Contents are written to the buffer, but not read.
GL.STREAM_DRAW, // Buffer not used often. Contents are written to the buffer, but not read.
// For WebGL 2 contexts
GL.STATIC_READ, // Buffer used often and not change often. Contents are read from the buffer, but not written.
GL.DYNAMIC_READ, // Buffer used often and change often. Contents are read from the buffer, but not written.
GL.STREAM_READ, // Contents of the buffer are likely to not be used often. Contents are read from the buffer, but not written.
GL.STATIC_COPY, // Buffer used often and not change often. Contents are neither written or read by the user.
GL.DYNAMIC_COPY, // Buffer used often and change often. Contents are neither written or read by the user.
GL.STREAM_COPY // Buffer used often and not change often. Contents are neither written or read by the user.
];

export function assertArrayTypeMatch(array, type, source) {
  assert(type === glTypeFromArray(array), (array.constructor.name || 'Array') + ' ' + ('does not match element type ' + glKey(type) + ' ' + source));
}

// GL errors

// Returns an Error representing the Latest webGl error or null
export function glGetError(gl) {
  // Loop to ensure all errors are cleared
  var errorStack = [];
  var glError = gl.getError();
  while (glError !== gl.NO_ERROR) {
    errorStack.push(glGetErrorMessage(gl, glError));
    glError = gl.getError();
  }
  return errorStack.length ? new Error(errorStack.join('\n')) : null;
}

export function glCheckError(gl) {
  if (gl.debug) {
    var error = glGetError(gl);
    if (error) {
      throw error;
    }
  }
}

function glGetErrorMessage(gl, glError) {
  switch (glError) {
    case GL.CONTEXT_LOST_WEBGL:
      //  If the WebGL context is lost, this error is returned on the
      // first call to getError. Afterwards and until the context has been
      // restored, it returns gl.NO_ERROR.
      return 'WebGL context lost';
    case GL.INVALID_ENUM:
      // An unacceptable value has been specified for an enumerated argument.
      return 'WebGL invalid enumerated argument';
    case GL.INVALID_VALUE:
      // A numeric argument is out of range.
      return 'WebGL invalid value';
    case GL.INVALID_OPERATION:
      // The specified command is not allowed for the current state.
      return 'WebGL invalid operation';
    case GL.INVALID_FRAMEBUFFER_OPERATION:
      // The currently bound framebuffer is not framebuffer complete
      // when trying to render to or to read from it.
      return 'WebGL invalid framebuffer operation';
    case GL.OUT_OF_MEMORY:
      // Not enough memory is left to execute the command.
      return 'WebGL out of memory';
    default:
      return 'WebGL unknown error ' + glError;
  }
}
//# sourceMappingURL=data:application/json;base64,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