/* eslint-disable no-continue, max-statements */
import GL from './webgl-constants';

// import shaderName from 'glsl-shader-name';
// replace with regexp
export function getShaderName(shader) {
  return 'unnamed';
}

/**
 * Formats a GLSL shader compiler error and generates a string
 * showing the source code around the error.
 *
 * From https://github.com/wwwtyro/gl-format-compiler-error (public domain)
 *
 * @param {String} errLog - error log from gl.getShaderInfoLog
 * @param {String} src - original shader source code
 * @param {Number} shaderType - shader type (GL constant)
 * @return {String} - Formatted strings has the error marked inline with src.
 */
export function formatGLSLCompilerError(errLog, src, shaderType) {
  var errorStrings = errLog.split(/\r?\n/);
  var errors = {};

  // Parse the error - note: browser and driver dependent
  for (var i = 0; i < errorStrings.length; i++) {
    var errorString = errorStrings[i];
    if (errorString.length <= 1) {
      continue;
    }
    var lineNo = parseInt(errorString.split(':')[2], 10);
    if (isNaN(lineNo)) {
      return 'Could not parse GLSL compiler error: ' + errLog;
    }
    errors[lineNo] = errorString;
  }

  // Format the error inline with the code
  var message = '';
  var lines = addLineNumbers(src).split(/\r?\n/);

  for (var _i = 0; _i < lines.length; _i++) {
    var line = lines[_i];
    if (!errors[_i + 3] && !errors[_i + 2] && !errors[_i + 1]) {
      continue;
    }
    message += line + '\n';
    if (errors[_i + 1]) {
      var e = errors[_i + 1];
      e = e.substr(e.split(':', 3).join(':').length + 1).trim();
      message += '^^^ ' + e + '\n\n';
    }
  }

  var name = getShaderName(src) || 'unknown name';
  var type = getShaderTypeName(shaderType);
  return 'GLSL error in ' + type + ' shader ' + name + '\n' + message;
}

/**
 * Prepends line numbers to each line of a string.
 * The line numbers will be left-padded with spaces to ensure an
 * aligned layout when rendered using monospace fonts.
 *
 * Adapted from https://github.com/Jam3/add-line-numbers, MIT license
 *
 * @param {String} string - multi-line string to add line numbers to
 * @param {Number} start=1 - number of spaces to add
 * @param {String} delim =': ' - injected between line number and original line
 * @return {String} string - The original string with line numbers added
 */
function addLineNumbers(string) {
  var start = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 1;
  var delim = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : ': ';

  var lines = string.split(/\r?\n/);
  var maxDigits = String(lines.length + start - 1).length;
  return lines.map(function (line, i) {
    var lineNumber = i + start;
    var digits = String(lineNumber).length;
    var prefix = padLeft(lineNumber, maxDigits - digits);
    return prefix + delim + line;
  }).join('\n');
}

/**
 * Pads a string with a number of spaces (space characters) to the left
 * @param {String} string - string to pad
 * @param {Number} digits - number of spaces to add
 * @return {String} string - The padded string
 */
function padLeft(string, digits) {
  var result = '';
  for (var i = 0; i < digits; ++i) {
    result += ' ';
  }
  return '' + result + string;
}

function getShaderTypeName(type) {
  switch (type) {
    case GL.FRAGMENT_SHADER:
      return 'fragment';
    case GL.VERTEX_SHADER:
      return 'vertex';
    default:
      return 'unknown type';
  }
}
//# sourceMappingURL=data:application/json;base64,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