'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Keys = exports.EventsProxy = exports.stop = undefined;

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }(); // event.js
// Handle keyboard/mouse/touch events in the Canvas
// TODO - this will not work under node

/* eslint-disable dot-notation, max-statements, no-loop-func */
/* global window, document */


exports.get = get;
exports.getWheel = getWheel;
exports.getKey = getKey;
exports.isRightClick = isRightClick;
exports.getPos = getPos;
exports.addEvents = addEvents;

var _utils = require('../utils');

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var KEYS = {
  enter: 13,
  up: 38,
  down: 40,
  left: 37,
  right: 39,
  esc: 27,
  space: 32,
  backspace: 8,
  tab: 9,
  delete: 46
};

// returns an O3D object or false otherwise.
function toO3D(n) {
  return n !== true ? n : false;
}

// Returns an element position
function _getPos(elem) {
  var bbox = elem.getBoundingClientRect();
  return {
    x: bbox.left,
    y: bbox.top,
    bbox: bbox
  };
}

// event object wrapper
function get(e, win) {
  win = win || window;
  return e || win.event;
}

function getWheel(e) {
  return e.wheelDelta ? e.wheelDelta / 120 : -(e.detail || 0) / 3;
}

function getKey(e) {
  var code = e.which || e.keyCode;
  var key = keyOf(code);
  // onkeydown
  var fKey = code - 111;
  if (fKey > 0 && fKey < 13) {
    key = 'f' + fKey;
  }
  key = key || String.fromCharCode(code).toLowerCase();

  return {
    code: code,
    key: key,
    shift: e.shiftKey,
    control: e.ctrlKey,
    alt: e.altKey,
    meta: e.metaKey
  };
}

function isRightClick(e) {
  return e.which === 3 || e.button === 2;
}

function getPos(e, win) {
  // get mouse position
  win = win || window;
  e = e || win.event;
  var doc = win.document;
  doc = doc.documentElement || doc.body;
  // TODO(nico): make touch event handling better
  if (e.touches && e.touches.length) {
    var touchesPos = [];
    var l = e.touches.length;
    var evt = void 0;
    for (var i = 0; i < l; ++i) {
      evt = e.touches[i];
      touchesPos.push({
        x: evt.pageX || evt.clientX + doc.scrollLeft,
        y: evt.pageY || evt.clientY + doc.scrollTop
      });
    }
    return touchesPos;
  }
  var page = {
    x: e.pageX || e.clientX + doc.scrollLeft,
    y: e.pageY || e.clientY + doc.scrollTop
  };
  return [page];
}

function _stop(e) {
  if (e.stopPropagation) {
    e.stopPropagation();
  }
  e.cancelBubble = true;
  if (e.preventDefault) {
    e.preventDefault();
  } else {
    e.returnValue = false;
  }
}

exports.stop = _stop;

var EventsProxy = exports.EventsProxy = function () {
  function EventsProxy(domElem, opt) {
    _classCallCheck(this, EventsProxy);

    this.scene = opt.scene;
    this.domElem = domElem;
    this.pos = _getPos(domElem);
    this.opt = this.callbacks = opt;

    this.size = {
      width: domElem.width || domElem.offsetWidth,
      height: domElem.height || domElem.offsetHeight
    };

    this.attachEvents();
  }

  _createClass(EventsProxy, [{
    key: 'attachEvents',
    value: function attachEvents() {
      var _this = this;

      var domElem = this.domElem;
      var opt = this.opt;

      if (opt.disableContextMenu) {
        domElem.oncontextmenu = function () {
          return false;
        };
      }

      if (opt.enableMouse) {
        ['mouseup', 'mousedown', 'mousemove', 'mouseover', 'mouseout'].forEach(function (action) {
          domElem.addEventListener(action, function (e, win) {
            _this[action](_this.eventInfo(action, e, win));
          }, false);
        });

        // "well, this is embarrassing..."
        var type = '';
        if (!document.getBoxObjectFor && window.mozInnerScreenX === null) {
          type = 'mousewheel';
        } else {
          type = 'DOMMouseScroll';
        }
        domElem.addEventListener(type, function (e, win) {
          _this['mousewheel'](_this.eventInfo('mousewheel', e, win));
        }, false);
      }

      if (opt.enableTouch) {
        ['touchstart', 'touchmove', 'touchend'].forEach(function (action) {
          domElem.addEventListener(action, function (e, win) {
            _this[action](_this.eventInfo(action, e, win));
          }, false);
        });
      }

      if (opt.enableKeyboard) {
        ['keydown', 'keyup'].forEach(function (action) {
          document.addEventListener(action, function (e, win) {
            _this[action](_this.eventInfo(action, e, win));
          }, false);
        });
      }
    }
  }, {
    key: 'eventInfo',
    value: function eventInfo(type, e, win) {
      var domElem = this.domElem;
      var scene = this.scene;
      var opt = this.opt;
      var size = this.getSize();
      var relative = opt.relative;
      var centerOrigin = opt.centerOrigin;
      var pos = opt.cachePosition && this.pos || _getPos(domElem);
      var ge = get(e, win);
      var epos = getPos(e, win);
      var origPos = { x: epos[0].x, y: epos[0].y };
      var evt = {};
      var x = void 0;
      var y = void 0;

      // get Position
      for (var i = 0, l = epos.length; i < l; ++i) {
        x = epos[i].x;
        y = epos[i].y;
        if (relative) {
          x -= pos.x;y -= pos.y;
          if (centerOrigin) {
            x -= size.width / 2;
            y -= size.height / 2;
            // y axis now points to the top of the screen
            y *= -1;
          }
        }
        epos[i].x = x;
        epos[i].y = y;
      }

      switch (type) {
        case 'mousewheel':
          evt.wheel = getWheel(ge);
          break;
        case 'keydown':
        case 'keyup':
          Object.assign(evt, getKey(ge));
          break;
        case 'mouseup':
          evt.isRightClick = isRightClick(ge);
          break;
        default:
          break;
      }

      var cacheTarget = void 0;

      Object.assign(evt, {
        x: epos[0].x,
        y: epos[0].y,
        posArray: epos,

        cache: false,
        // stop event propagation
        stop: function stop() {
          _stop(ge);
        },

        // get the target element of the event
        getTarget: function getTarget() {
          if (cacheTarget) {
            return cacheTarget;
          }
          return cacheTarget = opt.picking && scene.pick(origPos.x - pos.x, origPos.y - pos.y) || true;
        }
      });
      // wrap native event
      evt.event = ge;

      return evt;
    }
  }, {
    key: 'getSize',
    value: function getSize() {
      if (this.cacheSize) {
        return this.size;
      }
      var domElem = this.domElem;
      return {
        width: domElem.width || domElem.offsetWidth,
        height: domElem.height || domElem.offsetHeight
      };
    }
  }, {
    key: 'mouseup',
    value: function mouseup(e) {
      if (!this.moved) {
        if (e.isRightClick) {
          this.callbacks.onRightClick(e, this.hovered);
        } else {
          this.callbacks.onClick(e, toO3D(this.pressed));
        }
      }
      if (this.pressed) {
        if (this.moved) {
          this.callbacks.onDragEnd(e, toO3D(this.pressed));
        } else {
          this.callbacks.onDragCancel(e, toO3D(this.pressed));
        }
        this.pressed = this.moved = false;
      }
    }
  }, {
    key: 'mouseout',
    value: function mouseout(e) {
      // mouseout canvas
      var rt = e.relatedTarget;
      var domElem = this.domElem;
      while (rt && rt.parentNode) {
        if (domElem === rt.parentNode) {
          return;
        }
        rt = rt.parentNode;
      }
      if (this.hovered) {
        this.callbacks.onMouseLeave(e, this.hovered);
        this.hovered = false;
      }
      if (this.pressed && this.moved) {
        this.callbacks.onDragEnd(e);
        this.pressed = this.moved = false;
      }
    }
  }, {
    key: 'mouseover',
    value: function mouseover(e) {}
  }, {
    key: 'mousemove',
    value: function mousemove(e) {
      if (this.pressed) {
        this.moved = true;
        this.callbacks.onDragMove(e, toO3D(this.pressed));
        return;
      }
      if (this.hovered) {
        var target = toO3D(e.getTarget());
        if (!target || target.hash !== this.hash) {
          this.callbacks.onMouseLeave(e, this.hovered);
          this.hovered = target;
          this.hash = target;
          if (target) {
            this.hash = target.hash;
            this.callbacks.onMouseEnter(e, this.hovered);
          }
        } else {
          this.callbacks.onMouseMove(e, this.hovered);
        }
      } else {
        this.hovered = toO3D(e.getTarget());
        this.hash = this.hovered;
        if (this.hovered) {
          this.hash = this.hovered.hash;
          this.callbacks.onMouseEnter(e, this.hovered);
        }
      }
      if (!this.opt.picking) {
        this.callbacks.onMouseMove(e);
      }
    }
  }, {
    key: 'mousewheel',
    value: function mousewheel(e) {
      this.callbacks.onMouseWheel(e);
    }
  }, {
    key: 'mousedown',
    value: function mousedown(e) {
      this.pressed = e.getTarget();
      this.callbacks.onDragStart(e, toO3D(this.pressed));
    }
  }, {
    key: 'touchstart',
    value: function touchstart(e) {
      this.touched = e.getTarget();
      this.touchedLastPosition = { x: e.x, y: e.y };
      this.callbacks.onTouchStart(e, toO3D(this.touched));
    }
  }, {
    key: 'touchmove',
    value: function touchmove(e) {
      if (this.touched) {
        this.touchMoved = true;
        this.callbacks.onTouchMove(e, toO3D(this.touched));
      }
    }
  }, {
    key: 'touchend',
    value: function touchend(e) {
      if (this.touched) {
        if (this.touchMoved) {
          this.callbacks.onTouchEnd(e, toO3D(this.touched));
        } else {
          e.x = isNaN(e.x) ? this.touchedLastPosition.x : e.x;
          e.y = isNaN(e.y) ? this.touchedLastPosition.y : e.y;
          this.callbacks.onTap(e, toO3D(this.touched));
          this.callbacks.onTouchCancel(e, toO3D(this.touched));
        }
        this.touched = this.touchMoved = false;
      }
    }
  }, {
    key: 'keydown',
    value: function keydown(e) {
      this.callbacks.onKeyDown(e);
    }
  }, {
    key: 'keyup',
    value: function keyup(e) {
      this.callbacks.onKeyUp(e);
    }
  }]);

  return EventsProxy;
}();

Object.assign(EventsProxy.prototype, {
  hovered: false,
  pressed: false,
  touched: false,
  touchedLastPosition: { x: 0, y: 0 },
  touchMoved: false,
  moved: false
});

var DEFAULT_OPTS = {
  cachePosition: true,
  cacheSize: true,
  relative: true,
  centerOrigin: true,
  disableContextMenu: true,
  bind: false,
  picking: false,

  enableTouch: true,
  enableMouse: true,
  enableKeyboard: true,

  onClick: _utils.noop,
  onRightClick: _utils.noop,
  onDragStart: _utils.noop,
  onDragMove: _utils.noop,
  onDragEnd: _utils.noop,
  onDragCancel: _utils.noop,
  onTouchStart: _utils.noop,
  onTouchMove: _utils.noop,
  onTouchEnd: _utils.noop,
  onTouchCancel: _utils.noop,
  onTap: _utils.noop,
  onMouseMove: _utils.noop,
  onMouseEnter: _utils.noop,
  onMouseLeave: _utils.noop,
  onMouseWheel: _utils.noop,
  onKeyDown: _utils.noop,
  onKeyUp: _utils.noop
};

var Keys = exports.Keys = KEYS;

function keyOf(code) {
  var keyMap = Keys;
  for (var name in keyMap) {
    if (keyMap[name] === code) {
      return name;
    }
  }
  return null;
}

function addEvents(domElement) {
  var opt = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

  opt = Object.assign({}, DEFAULT_OPTS, opt);

  var bind = opt.bind;
  if (bind) {
    for (var name in opt) {
      if (name.match(/^on[a-zA-Z0-9]+$/)) {
        (function (fname, fn) {
          opt[fname] = function f() {
            fn.apply(bind, Array.prototype.slice.call(arguments));
          };
        })(name, opt[name]);
      }
    }
  }

  return new EventsProxy(domElement, opt);
}
//# sourceMappingURL=data:application/json;base64,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