'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.breakOnConsoleWarnings = breakOnConsoleWarnings;
exports.throwOnConsoleWarnings = throwOnConsoleWarnings;
exports.interceptRejectedPromises = interceptRejectedPromises;

var _env = require('./env');

/**
 * Ensure that your debugger stops when code issues warnings so that
 * you can see what is going on in othercomponents when they decide
 * to issue warnings.
 *
 * @param {Array} consoleBlacklist - array of strings to match against
 */
function breakOnConsoleWarnings() {
  var consoleBlacklist = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [/.*/];

  function breakOnConsole(log, msg, param1) {
    for (var _len = arguments.length, params = Array(_len > 3 ? _len - 3 : 0), _key = 3; _key < _len; _key++) {
      params[_key - 3] = arguments[_key];
    }

    if (typeof msg === 'string' && msg.indexOf('Unhandled promise rejection') === 0) {
      log.apply(undefined, [msg, param1].concat(params));
      throw new Error(param1);
    } else if (consoleBlacklist.some(function (pattern) {
      return pattern.test(msg);
    })) {
      log.apply(undefined, [msg, param1].concat(params));
    } else {
      log.apply(undefined, [msg, param1].concat(params));
    }
  }
  _env.logger.warn = breakOnConsole.bind(null, _env.logger.native.warn);
  _env.logger.error = breakOnConsole.bind(null, _env.logger.native.error);

  window.onerror = function (message, url, line, column, error) {
    if (error) {
      _env.logger.native.error(error + ' ' + url + ':' + line + ':' + (column || 0));
    } else {
      _env.logger.native.error(message + ' ' + url + ':' + line + ':' + (column || 0));
    }
    debugger;
  };
}

/**
 * Throw exceptions when code issues warnings so that
 * you can access them in your normal exception handling setup, perhaps
 * displaying them in the UI or logging them in a different way.
 *
 * @param {Array} consoleBlacklist - array of strings to match against
 */
/**
 * Utilities for dev-mode error handling
 */
/* eslint-disable no-console, no-debugger */
/* global window */
function throwOnConsoleWarnings() {
  var consoleBlacklist = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [/.*/];

  _env.logger.warn = function throwOnWarning(msg) {
    var _console$native;

    if (consoleBlacklist.some(function (patt) {
      return patt.test(msg);
    })) {
      throw new Error('Unacceptable warning: ' + msg);
    }
    (_console$native = _env.logger.native).warn.apply(_console$native, arguments);
  };
}

// Chrome has yet to implement onRejectedPromise, so trigger onerror instead
function interceptRejectedPromises() {
  var _arguments = arguments;

  _env.logger.error = function (msg, error) {
    for (var _len2 = arguments.length, params = Array(_len2 > 2 ? _len2 - 2 : 0), _key2 = 2; _key2 < _len2; _key2++) {
      params[_key2 - 2] = arguments[_key2];
    }

    var _console$native3;

    if (typeof msg === 'string' && msg.indexOf('Unhandled promise rejection') === 0) {
      var _console$native2;

      error.unhandledPromise = true;
      // Use different message to avoid triggering again
      (_console$native2 = _env.logger.native).error.apply(_console$native2, ['Rejected promise', error].concat(params));
      throw error;
    }
    (_console$native3 = _env.logger.native).error.apply(_console$native3, _arguments);
  };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uL3NyYy9leHBlcmltZW50YWwvcHJvYmUvc3JjL2Vycm9yLXV0aWxzLmpzIl0sIm5hbWVzIjpbImJyZWFrT25Db25zb2xlV2FybmluZ3MiLCJ0aHJvd09uQ29uc29sZVdhcm5pbmdzIiwiaW50ZXJjZXB0UmVqZWN0ZWRQcm9taXNlcyIsImNvbnNvbGVCbGFja2xpc3QiLCJicmVha09uQ29uc29sZSIsImxvZyIsIm1zZyIsInBhcmFtMSIsInBhcmFtcyIsImluZGV4T2YiLCJFcnJvciIsInNvbWUiLCJwYXR0ZXJuIiwidGVzdCIsIndhcm4iLCJiaW5kIiwibmF0aXZlIiwiZXJyb3IiLCJ3aW5kb3ciLCJvbmVycm9yIiwibWVzc2FnZSIsInVybCIsImxpbmUiLCJjb2x1bW4iLCJ0aHJvd09uV2FybmluZyIsInBhdHQiLCJhcmd1bWVudHMiLCJ1bmhhbmRsZWRQcm9taXNlIl0sIm1hcHBpbmdzIjoiOzs7OztRQWNnQkEsc0IsR0FBQUEsc0I7UUFnQ0FDLHNCLEdBQUFBLHNCO1FBVUFDLHlCLEdBQUFBLHlCOztBQW5EaEI7O0FBRUE7Ozs7Ozs7QUFPTyxTQUFTRixzQkFBVCxHQUEyRDtBQUFBLE1BQTNCRyxnQkFBMkIsdUVBQVIsQ0FBQyxJQUFELENBQVE7O0FBQ2hFLFdBQVNDLGNBQVQsQ0FBd0JDLEdBQXhCLEVBQTZCQyxHQUE3QixFQUFrQ0MsTUFBbEMsRUFBcUQ7QUFBQSxzQ0FBUkMsTUFBUTtBQUFSQSxZQUFRO0FBQUE7O0FBQ25ELFFBQUksT0FBT0YsR0FBUCxLQUFlLFFBQWYsSUFDRkEsSUFBSUcsT0FBSixDQUFZLDZCQUFaLE1BQStDLENBRGpELEVBQ29EO0FBQ2xESiw0QkFBSUMsR0FBSixFQUFTQyxNQUFULFNBQW9CQyxNQUFwQjtBQUNBLFlBQU0sSUFBSUUsS0FBSixDQUFVSCxNQUFWLENBQU47QUFDRCxLQUpELE1BSU8sSUFBSUosaUJBQWlCUSxJQUFqQixDQUFzQjtBQUFBLGFBQVdDLFFBQVFDLElBQVIsQ0FBYVAsR0FBYixDQUFYO0FBQUEsS0FBdEIsQ0FBSixFQUF5RDtBQUM5REQsNEJBQUlDLEdBQUosRUFBU0MsTUFBVCxTQUFvQkMsTUFBcEI7QUFDRCxLQUZNLE1BRUE7QUFDTEgsNEJBQUlDLEdBQUosRUFBU0MsTUFBVCxTQUFvQkMsTUFBcEI7QUFDRDtBQUNGO0FBQ0QsY0FBUU0sSUFBUixHQUFlVixlQUFlVyxJQUFmLENBQW9CLElBQXBCLEVBQTBCLFlBQVFDLE1BQVIsQ0FBZUYsSUFBekMsQ0FBZjtBQUNBLGNBQVFHLEtBQVIsR0FBZ0JiLGVBQWVXLElBQWYsQ0FBb0IsSUFBcEIsRUFBMEIsWUFBUUMsTUFBUixDQUFlQyxLQUF6QyxDQUFoQjs7QUFFQUMsU0FBT0MsT0FBUCxHQUFpQixVQUFDQyxPQUFELEVBQVVDLEdBQVYsRUFBZUMsSUFBZixFQUFxQkMsTUFBckIsRUFBNkJOLEtBQTdCLEVBQXVDO0FBQ3RELFFBQUlBLEtBQUosRUFBVztBQUNULGtCQUFRRCxNQUFSLENBQWVDLEtBQWYsQ0FBd0JBLEtBQXhCLFNBQWlDSSxHQUFqQyxTQUF3Q0MsSUFBeEMsVUFBZ0RDLFVBQVUsQ0FBMUQ7QUFDRCxLQUZELE1BRU87QUFDTCxrQkFBUVAsTUFBUixDQUFlQyxLQUFmLENBQXdCRyxPQUF4QixTQUFtQ0MsR0FBbkMsU0FBMENDLElBQTFDLFVBQWtEQyxVQUFVLENBQTVEO0FBQ0Q7QUFDRDtBQUNELEdBUEQ7QUFRRDs7QUFFRDs7Ozs7OztBQXZDQTs7O0FBR0E7QUFDQTtBQTBDTyxTQUFTdEIsc0JBQVQsR0FBMkQ7QUFBQSxNQUEzQkUsZ0JBQTJCLHVFQUFSLENBQUMsSUFBRCxDQUFROztBQUNoRSxjQUFRVyxJQUFSLEdBQWUsU0FBU1UsY0FBVCxDQUF3QmxCLEdBQXhCLEVBQTZCO0FBQUE7O0FBQzFDLFFBQUlILGlCQUFpQlEsSUFBakIsQ0FBc0I7QUFBQSxhQUFRYyxLQUFLWixJQUFMLENBQVVQLEdBQVYsQ0FBUjtBQUFBLEtBQXRCLENBQUosRUFBbUQ7QUFDakQsWUFBTSxJQUFJSSxLQUFKLDRCQUFtQ0osR0FBbkMsQ0FBTjtBQUNEO0FBQ0QsbUNBQVFVLE1BQVIsRUFBZUYsSUFBZix3QkFBdUJZLFNBQXZCO0FBQ0QsR0FMRDtBQU1EOztBQUVEO0FBQ08sU0FBU3hCLHlCQUFULEdBQXFDO0FBQUE7O0FBQzFDLGNBQVFlLEtBQVIsR0FBZ0IsVUFBQ1gsR0FBRCxFQUFNVyxLQUFOLEVBQTJCO0FBQUEsdUNBQVhULE1BQVc7QUFBWEEsWUFBVztBQUFBOztBQUFBOztBQUN6QyxRQUFJLE9BQU9GLEdBQVAsS0FBZSxRQUFmLElBQ0ZBLElBQUlHLE9BQUosQ0FBWSw2QkFBWixNQUErQyxDQURqRCxFQUNvRDtBQUFBOztBQUNsRFEsWUFBTVUsZ0JBQU4sR0FBeUIsSUFBekI7QUFDQTtBQUNBLHNDQUFRWCxNQUFSLEVBQWVDLEtBQWYsMEJBQXFCLGtCQUFyQixFQUF5Q0EsS0FBekMsU0FBbURULE1BQW5EO0FBQ0EsWUFBTVMsS0FBTjtBQUNEO0FBQ0Qsb0NBQVFELE1BQVIsRUFBZUMsS0FBZjtBQUNELEdBVEQ7QUFVRCIsImZpbGUiOiJlcnJvci11dGlscy5qcyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogVXRpbGl0aWVzIGZvciBkZXYtbW9kZSBlcnJvciBoYW5kbGluZ1xuICovXG4vKiBlc2xpbnQtZGlzYWJsZSBuby1jb25zb2xlLCBuby1kZWJ1Z2dlciAqL1xuLyogZ2xvYmFsIHdpbmRvdyAqL1xuaW1wb3J0IHtsb2dnZXIgYXMgY29uc29sZX0gZnJvbSAnLi9lbnYnO1xuXG4vKipcbiAqIEVuc3VyZSB0aGF0IHlvdXIgZGVidWdnZXIgc3RvcHMgd2hlbiBjb2RlIGlzc3VlcyB3YXJuaW5ncyBzbyB0aGF0XG4gKiB5b3UgY2FuIHNlZSB3aGF0IGlzIGdvaW5nIG9uIGluIG90aGVyY29tcG9uZW50cyB3aGVuIHRoZXkgZGVjaWRlXG4gKiB0byBpc3N1ZSB3YXJuaW5ncy5cbiAqXG4gKiBAcGFyYW0ge0FycmF5fSBjb25zb2xlQmxhY2tsaXN0IC0gYXJyYXkgb2Ygc3RyaW5ncyB0byBtYXRjaCBhZ2FpbnN0XG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBicmVha09uQ29uc29sZVdhcm5pbmdzKGNvbnNvbGVCbGFja2xpc3QgPSBbLy4qL10pIHtcbiAgZnVuY3Rpb24gYnJlYWtPbkNvbnNvbGUobG9nLCBtc2csIHBhcmFtMSwgLi4ucGFyYW1zKSB7XG4gICAgaWYgKHR5cGVvZiBtc2cgPT09ICdzdHJpbmcnICYmXG4gICAgICBtc2cuaW5kZXhPZignVW5oYW5kbGVkIHByb21pc2UgcmVqZWN0aW9uJykgPT09IDApIHtcbiAgICAgIGxvZyhtc2csIHBhcmFtMSwgLi4ucGFyYW1zKTtcbiAgICAgIHRocm93IG5ldyBFcnJvcihwYXJhbTEpO1xuICAgIH0gZWxzZSBpZiAoY29uc29sZUJsYWNrbGlzdC5zb21lKHBhdHRlcm4gPT4gcGF0dGVybi50ZXN0KG1zZykpKSB7XG4gICAgICBsb2cobXNnLCBwYXJhbTEsIC4uLnBhcmFtcyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGxvZyhtc2csIHBhcmFtMSwgLi4ucGFyYW1zKTtcbiAgICB9XG4gIH1cbiAgY29uc29sZS53YXJuID0gYnJlYWtPbkNvbnNvbGUuYmluZChudWxsLCBjb25zb2xlLm5hdGl2ZS53YXJuKTtcbiAgY29uc29sZS5lcnJvciA9IGJyZWFrT25Db25zb2xlLmJpbmQobnVsbCwgY29uc29sZS5uYXRpdmUuZXJyb3IpO1xuXG4gIHdpbmRvdy5vbmVycm9yID0gKG1lc3NhZ2UsIHVybCwgbGluZSwgY29sdW1uLCBlcnJvcikgPT4ge1xuICAgIGlmIChlcnJvcikge1xuICAgICAgY29uc29sZS5uYXRpdmUuZXJyb3IoYCR7ZXJyb3J9ICR7dXJsfToke2xpbmV9OiR7Y29sdW1uIHx8IDB9YCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnNvbGUubmF0aXZlLmVycm9yKGAke21lc3NhZ2V9ICR7dXJsfToke2xpbmV9OiR7Y29sdW1uIHx8IDB9YCk7XG4gICAgfVxuICAgIGRlYnVnZ2VyO1xuICB9O1xufVxuXG4vKipcbiAqIFRocm93IGV4Y2VwdGlvbnMgd2hlbiBjb2RlIGlzc3VlcyB3YXJuaW5ncyBzbyB0aGF0XG4gKiB5b3UgY2FuIGFjY2VzcyB0aGVtIGluIHlvdXIgbm9ybWFsIGV4Y2VwdGlvbiBoYW5kbGluZyBzZXR1cCwgcGVyaGFwc1xuICogZGlzcGxheWluZyB0aGVtIGluIHRoZSBVSSBvciBsb2dnaW5nIHRoZW0gaW4gYSBkaWZmZXJlbnQgd2F5LlxuICpcbiAqIEBwYXJhbSB7QXJyYXl9IGNvbnNvbGVCbGFja2xpc3QgLSBhcnJheSBvZiBzdHJpbmdzIHRvIG1hdGNoIGFnYWluc3RcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHRocm93T25Db25zb2xlV2FybmluZ3MoY29uc29sZUJsYWNrbGlzdCA9IFsvLiovXSkge1xuICBjb25zb2xlLndhcm4gPSBmdW5jdGlvbiB0aHJvd09uV2FybmluZyhtc2cpIHtcbiAgICBpZiAoY29uc29sZUJsYWNrbGlzdC5zb21lKHBhdHQgPT4gcGF0dC50ZXN0KG1zZykpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFVuYWNjZXB0YWJsZSB3YXJuaW5nOiAke21zZ31gKTtcbiAgICB9XG4gICAgY29uc29sZS5uYXRpdmUud2FybiguLi5hcmd1bWVudHMpO1xuICB9O1xufVxuXG4vLyBDaHJvbWUgaGFzIHlldCB0byBpbXBsZW1lbnQgb25SZWplY3RlZFByb21pc2UsIHNvIHRyaWdnZXIgb25lcnJvciBpbnN0ZWFkXG5leHBvcnQgZnVuY3Rpb24gaW50ZXJjZXB0UmVqZWN0ZWRQcm9taXNlcygpIHtcbiAgY29uc29sZS5lcnJvciA9IChtc2csIGVycm9yLCAuLi5wYXJhbXMpID0+IHtcbiAgICBpZiAodHlwZW9mIG1zZyA9PT0gJ3N0cmluZycgJiZcbiAgICAgIG1zZy5pbmRleE9mKCdVbmhhbmRsZWQgcHJvbWlzZSByZWplY3Rpb24nKSA9PT0gMCkge1xuICAgICAgZXJyb3IudW5oYW5kbGVkUHJvbWlzZSA9IHRydWU7XG4gICAgICAvLyBVc2UgZGlmZmVyZW50IG1lc3NhZ2UgdG8gYXZvaWQgdHJpZ2dlcmluZyBhZ2FpblxuICAgICAgY29uc29sZS5uYXRpdmUuZXJyb3IoJ1JlamVjdGVkIHByb21pc2UnLCBlcnJvciwgLi4ucGFyYW1zKTtcbiAgICAgIHRocm93IGVycm9yO1xuICAgIH1cbiAgICBjb25zb2xlLm5hdGl2ZS5lcnJvciguLi5hcmd1bWVudHMpO1xuICB9O1xufVxuIl19