'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkRendererVendor = checkRendererVendor;
exports.getPlatformShaderDefines = getPlatformShaderDefines;
exports.assembleShaders = assembleShaders;

var _registerShaders = require('./register-shaders');

var _webgl = require('../../webgl');

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var VERTEX_SHADER = 'vertexShader';
var FRAGMENT_SHADER = 'fragmentShader';

function checkRendererVendor(debugInfo, gpuVendor) {
  var vendor = debugInfo.vendor,
      renderer = debugInfo.renderer;

  var result = void 0;
  switch (gpuVendor) {
    case 'nvidia':
      result = vendor.match(/NVIDIA/i) || renderer.match(/NVIDIA/i);
      break;
    case 'intel':
      result = vendor.match(/INTEL/i) || renderer.match(/INTEL/i);
      break;
    case 'amd':
      result = vendor.match(/AMD/i) || renderer.match(/AMD/i) || vendor.match(/ATI/i) || renderer.match(/ATI/i);
      break;
    default:
      result = false;
  }
  return result;
}

function getPlatformShaderDefines(gl) {
  /* eslint-disable */
  var platformDefines = '';
  var debugInfo = (0, _webgl.glGetDebugInfo)(gl);

  if (checkRendererVendor(debugInfo, 'nvidia')) {
    platformDefines += '#define NVIDIA_GPU\n#define NVIDIA_FP64_WORKAROUND 1\n#define NVIDIA_EQUATION_WORKAROUND 1\n';
  } else if (checkRendererVendor(debugInfo, 'intel')) {
    platformDefines += '#define INTEL_GPU\n#define INTEL_FP64_WORKAROUND 1\n#define NVIDIA_EQUATION_WORKAROUND 1\n #define INTEL_TAN_WORKAROUND 1\n';
  } else if (checkRendererVendor(debugInfo, 'amd')) {
    platformDefines += '#define AMD_GPU\n';
  } else {
    platformDefines += '#define DEFAULT_GPU\n';
  }

  return platformDefines;
}

function assembleShader(gl, _ref) {
  var source = _ref.source,
      type = _ref.type,
      _ref$modules = _ref.modules,
      modules = _ref$modules === undefined ? [] : _ref$modules;

  (0, _assert2.default)(typeof source === 'string', 'shader source must be a string');

  // Add platform defines
  var assembledSource = getPlatformShaderDefines(gl) + '\n';

  // Add dependent modules in resolved order
  var _iteratorNormalCompletion = true;
  var _didIteratorError = false;
  var _iteratorError = undefined;

  try {
    for (var _iterator = modules[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
      var moduleName = _step.value;

      var shaderModule = (0, _registerShaders.getShaderModule)(moduleName);
      if (!shaderModule) {
        (0, _assert2.default)(shaderModule, 'shader module is not defined');
      }
      var moduleSource = shaderModule[type];
      assembledSource += '// BEGIN SHADER MODULE ' + moduleName + '\n#define MODULE_' + moduleName.toUpperCase() + '\n' + moduleSource + '\n// END SHADER MODULE ' + moduleName;
    }

    // Add actual source of shader
  } catch (err) {
    _didIteratorError = true;
    _iteratorError = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion && _iterator.return) {
        _iterator.return();
      }
    } finally {
      if (_didIteratorError) {
        throw _iteratorError;
      }
    }
  }

  assembledSource += source;

  return assembledSource;
}

/**
 * Apply set of modules
 */
function assembleShaders(gl) {
  var opts = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  var vs = opts.vs,
      fs = opts.fs;

  var modules = (0, _registerShaders.getShaderDependencies)(opts.modules || []);
  return {
    gl: gl,
    vs: assembleShader(gl, Object.assign({}, opts, { source: vs, type: VERTEX_SHADER, modules: modules })),
    fs: assembleShader(gl, Object.assign({}, opts, { source: fs, type: FRAGMENT_SHADER, modules: modules }))
  };
}
//# sourceMappingURL=data:application/json;base64,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