'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerShaderModules = registerShaderModules;
exports.getShaderModule = getShaderModule;
exports.getShaderDependencies = getShaderDependencies;

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var shaderModules = {};

/**
 * Registers an array of shader modules
 * @param {Object[]} shaderModuleList - Array of shader modules
 */
function registerShaderModules(shaderModuleList) {
  var _iteratorNormalCompletion = true;
  var _didIteratorError = false;
  var _iteratorError = undefined;

  try {
    for (var _iterator = shaderModuleList[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
      var shaderModule = _step.value;

      (0, _assert2.default)(shaderModule.name, 'shader module has no name');
      if (shaderModules[shaderModule.name]) {
        throw new Error('shader module ' + shaderModule.name + ' already registered');
      }
      shaderModules[shaderModule.name] = shaderModule;
      shaderModule.dependencies = shaderModule.dependencies || [];
    }
  } catch (err) {
    _didIteratorError = true;
    _iteratorError = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion && _iterator.return) {
        _iterator.return();
      }
    } finally {
      if (_didIteratorError) {
        throw _iteratorError;
      }
    }
  }
}

function getShaderModule(moduleName) {
  var shaderModule = shaderModules[moduleName];
  if (!shaderModule) {
    // console.log(`${moduleName} not in registered modules:`, shaderModules);
    (0, _assert2.default)(false, 'Unknown shader module ' + moduleName);
  }
  return shaderModule;
}

/**
 * Takes a list of shader module names and returns a new list of
 * shader module names that includes all dependencies, sorted so
 * that modules that are dependencies of other modules come first.
 *
 * If the shader glsl code from the returned modules is concatenated
 * in the reverse order, it is guaranteed that all functions be resolved and
 * that all function and variable definitions come before use.
 *
 * @param {String[]} moduleNames - Array of module names
 * @return {String[]} - Array of modules
 */
function getShaderDependencies(moduleNames) {
  var result = {};
  getDependencyGraph({
    moduleNames: moduleNames,
    level: 0,
    result: result
  });
  return Object.keys(result).sort(function (a, b) {
    return result[a] - result[b];
  });
}

// Adds another level of dependencies to the result map
function getDependencyGraph(_ref) {
  var moduleNames = _ref.moduleNames,
      level = _ref.level,
      result = _ref.result;

  if (level >= 5) {
    throw new Error('Possible loop in shader dependency graph');
  }
  var _iteratorNormalCompletion2 = true;
  var _didIteratorError2 = false;
  var _iteratorError2 = undefined;

  try {
    for (var _iterator2 = moduleNames[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
      var moduleName = _step2.value;

      result[moduleName] = level;
    }
  } catch (err) {
    _didIteratorError2 = true;
    _iteratorError2 = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion2 && _iterator2.return) {
        _iterator2.return();
      }
    } finally {
      if (_didIteratorError2) {
        throw _iteratorError2;
      }
    }
  }

  var _iteratorNormalCompletion3 = true;
  var _didIteratorError3 = false;
  var _iteratorError3 = undefined;

  try {
    for (var _iterator3 = moduleNames[Symbol.iterator](), _step3; !(_iteratorNormalCompletion3 = (_step3 = _iterator3.next()).done); _iteratorNormalCompletion3 = true) {
      var _moduleName = _step3.value;

      var shaderModule = getShaderModule(_moduleName);

      getDependencyGraph({
        moduleNames: shaderModule.dependencies,
        level: level,
        result: result
      });
    }
  } catch (err) {
    _didIteratorError3 = true;
    _iteratorError3 = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion3 && _iterator3.return) {
        _iterator3.return();
      }
    } finally {
      if (_didIteratorError3) {
        throw _iteratorError3;
      }
    }
  }

  return result;
}
//# sourceMappingURL=data:application/json;base64,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