'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.BufferLayout = undefined;

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

var _webgl = require('./webgl');

var _webglChecks = require('./webgl-checks');

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var BufferLayout =
/**
 * @classdesc
 * Store characteristics of a data layout
 * This data can be used when updating vertex attributes with
 * the associated buffer, freeing the application from keeping
 * track of this metadata.
 *
 * @class
 * @param {GLuint} size - number of values per element (1-4)
 * @param {GLuint} type - type of values (e.g. gl.FLOAT)
 * @param {GLbool} normalized=false - normalize integers to [-1,1] or [0,1]
 * @param {GLuint} integer=false - WebGL2 only, int-to-float conversion
 * @param {GLuint} stride=0 - supports strided arrays
 * @param {GLuint} offset=0 - supports strided arrays
 */
exports.BufferLayout = function BufferLayout() {
  var _ref = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
      type = _ref.type,
      _ref$size = _ref.size,
      size = _ref$size === undefined ? 1 : _ref$size,
      _ref$offset = _ref.offset,
      offset = _ref$offset === undefined ? 0 : _ref$offset,
      _ref$stride = _ref.stride,
      stride = _ref$stride === undefined ? 0 : _ref$stride,
      _ref$normalized = _ref.normalized,
      normalized = _ref$normalized === undefined ? false : _ref$normalized,
      _ref$integer = _ref.integer,
      integer = _ref$integer === undefined ? false : _ref$integer,
      _ref$instanced = _ref.instanced,
      instanced = _ref$instanced === undefined ? 0 : _ref$instanced;

  _classCallCheck(this, BufferLayout);

  this.type = type;
  this.size = size;
  this.offset = offset;
  this.stride = stride;
  this.normalized = normalized;
  this.integer = integer;
  this.instanced = instanced;
};

// Encapsulates a WebGLBuffer object

var Buffer = function () {
  _createClass(Buffer, null, [{
    key: 'makeFrom',


    /**
     * Returns a Buffer wrapped WebGLBuffer from a variety of inputs.
     * Allows other functions to transparently accept raw WebGLBuffers etc
     * and manipulate them using the methods in the `Buffer` class.
     * Checks for ".handle" (allows use of stack.gl's gl-buffer)
     *
     * @param {WebGLRenderingContext} gl - if a new buffer needs to be initialized
     * @param {*} object - candidate that will be coerced to a buffer
     * @returns {Buffer} - Buffer object that wraps the buffer parameter
     */
    value: function makeFrom(gl) {
      var object = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      return object instanceof Buffer ? object :
      // Use .handle (e.g from stack.gl's gl-buffer), else use buffer directly
      new Buffer(gl).setData({ handle: object.handle || object });
    }

    /*
     * @classdesc
     * Can be used to store vertex data, pixel data retrieved from images
     * or the framebuffer, and a variety of other things.
     *
     * Mainly used for uploading VertexAttributes to GPU
     * Setting data on a buffers (arrays) uploads it to the GPU.
     *
     * Holds an attribute name as a convenience
     * setData - Initializes size of buffer and sets
     *
     * @param {WebGLRenderingContext} gl - gl context
     * @param {string} opt.id - id for debugging
     */

  }]);

  function Buffer() {
    var gl = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

    var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
        id = _ref2.id,
        handle = _ref2.handle;

    _classCallCheck(this, Buffer);

    (0, _webglChecks.assertWebGLContext)(gl);

    handle = handle || gl.createBuffer();
    if (!(handle instanceof _webgl.WebGLBuffer)) {
      throw new Error('Failed to create WebGLBuffer');
    }

    this.gl = gl;
    this.handle = handle;
    this.id = id;
    this.bytes = undefined;
    this.data = null;
    this.target = _webgl.GL.ARRAY_BUFFER;
    this.layout = null;

    this.userData = {};
    Object.seal(this);
  }

  _createClass(Buffer, [{
    key: 'delete',
    value: function _delete() {
      var gl = this.gl;

      if (this.handle) {
        gl.deleteBuffer(this.handle);
        this.handle = null;
      }
      return this;
    }

    /**
     * Creates and initializes the buffer object's data store.
     *
     * @param {ArrayBufferView} opt.data - contents
     * @param {GLsizeiptr} opt.bytes - the size of the buffer object's data store.
     * @param {GLenum} opt.usage=gl.STATIC_DRAW - Allocation hint for GPU driver
     *
     * Characteristics of stored data, hints for vertex attribute
     *
     * @param {GLenum} opt.dataType=gl.FLOAT - type of data stored in buffer
     * @param {GLuint} opt.size=1 - number of values per vertex
     * @returns {Buffer} Returns itself for chaining.
     */

  }, {
    key: 'setData',
    value: function setData() {
      var _ref3 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          data = _ref3.data,
          bytes = _ref3.bytes,
          _ref3$target = _ref3.target,
          target = _ref3$target === undefined ? _webgl.GL.ARRAY_BUFFER : _ref3$target,
          _ref3$usage = _ref3.usage,
          usage = _ref3$usage === undefined ? _webgl.GL.STATIC_DRAW : _ref3$usage,
          layout = _ref3.layout,
          type = _ref3.type,
          _ref3$size = _ref3.size,
          size = _ref3$size === undefined ? 1 : _ref3$size,
          _ref3$offset = _ref3.offset,
          offset = _ref3$offset === undefined ? 0 : _ref3$offset,
          _ref3$stride = _ref3.stride,
          stride = _ref3$stride === undefined ? 0 : _ref3$stride,
          _ref3$normalized = _ref3.normalized,
          normalized = _ref3$normalized === undefined ? false : _ref3$normalized,
          _ref3$integer = _ref3.integer,
          integer = _ref3$integer === undefined ? false : _ref3$integer,
          _ref3$instanced = _ref3.instanced,
          instanced = _ref3$instanced === undefined ? 0 : _ref3$instanced;

      var gl = this.gl;

      (0, _assert2.default)(data || bytes >= 0, 'Buffer.setData needs data or bytes');
      type = type || (0, _webgl.glTypeFromArray)(data);

      if (data) {
        (0, _webglChecks.assertArrayTypeMatch)(data, type, 'in Buffer.setData');
      }

      this.bytes = bytes;
      this.data = data;
      this.target = target;
      this.layout = layout || new BufferLayout({
        type: type,
        size: size,
        offset: offset,
        stride: stride,
        normalized: normalized,
        integer: integer,
        instanced: instanced
      });

      // Note: When we are just creating and/or filling the buffer with data,
      // the target we use doesn't technically matter, so use ARRAY_BUFFER
      // https://www.opengl.org/wiki/Buffer_Object
      this.bind({ target: target });
      gl.bufferData(target, data || bytes, usage);
      this.unbind({ target: target });

      return this;
    }

    /**
     * Updates a subset of a buffer object's data store.
     * @param {ArrayBufferView} opt.data - contents
     * @returns {Buffer} Returns itself for chaining.
     */

  }, {
    key: 'subData',
    value: function subData() {
      var _ref4 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          data = _ref4.data,
          _ref4$offset = _ref4.offset,
          offset = _ref4$offset === undefined ? 0 : _ref4$offset;

      var gl = this.gl;

      (0, _assert2.default)(data, 'Buffer.updateData needs data');

      // Note: When we are just creating and/or filling the buffer with data,
      // the target we use doesn't technically matter, so use ARRAY_BUFFER
      // https://www.opengl.org/wiki/Buffer_Object
      this.bind({ target: _webgl.GL.ARRAY_BUFFER });
      gl.bufferSubData(_webgl.GL.ARRAY_BUFFER, offset, data);
      this.unbind({ target: _webgl.GL.ARRAY_BUFFER });

      return this;
    }

    /**
     * Binds a buffer to a given binding point (target).
     *
     * @param {Glenum} target - target for the bind operation.
     *  Possible values: gl.TRANSFORM_FEEDBACK_BUFFER and gl.UNIFORM_BUFFER
     * @param {GLuint} index - the index of the target.
     * @returns {Buffer} - Returns itself for chaining.
     */

  }, {
    key: 'bind',
    value: function bind() {
      var _ref5 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref5$target = _ref5.target,
          target = _ref5$target === undefined ? this.target : _ref5$target;

      this.gl.bindBuffer(target, this.handle);
      return this;
    }
  }, {
    key: 'unbind',
    value: function unbind() {
      var _ref6 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref6$target = _ref6.target,
          target = _ref6$target === undefined ? this.target : _ref6$target;

      // this.gl.bindBuffer(target, null);
      return this;
    }

    /**
     * Note: WEBGL2
     * Binds a buffer to a given binding point (target) at a given index.
     *
     * @param {Glenum} target - target for the bind operation.
     *  Possible values: gl.TRANSFORM_FEEDBACK_BUFFER and gl.UNIFORM_BUFFER
     * @param {GLuint} index - the index of the target.
     * @returns {Buffer} - Returns itself for chaining.
     */

  }, {
    key: 'bindBase',
    value: function bindBase() {
      var _ref7 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref7$target = _ref7.target,
          target = _ref7$target === undefined ? this.target : _ref7$target,
          index = _ref7.index;

      (0, _webglChecks.assertWebGL2Context)(this.gl);
      this.gl.bindBufferBase(target, index, this.handle);
      return this;
    }
  }, {
    key: 'unbindBase',
    value: function unbindBase() {
      var _ref8 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref8$target = _ref8.target,
          target = _ref8$target === undefined ? this.target : _ref8$target,
          index = _ref8.index;

      (0, _webglChecks.assertWebGL2Context)(this.gl);
      this.gl.bindBufferBase(target, index, null);
      return this;
    }

    /**
     * Note: WEBGL2
     * binds a range of a given WebGLBuffer to a given binding point (target)
     * at a given index.
     *
     * @param {Glenum} target - target for the bind operation.
     *  Possible values: gl.TRANSFORM_FEEDBACK_BUFFER and gl.UNIFORM_BUFFER
     * @param {GLuint} index - the index of the target.
     * @returns {Buffer} - Returns itself for chaining.
     */

  }, {
    key: 'bindRange',
    value: function bindRange() {
      var _ref9 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref9$target = _ref9.target,
          target = _ref9$target === undefined ? this.target : _ref9$target,
          index = _ref9.index,
          _ref9$offset = _ref9.offset,
          offset = _ref9$offset === undefined ? 0 : _ref9$offset,
          size = _ref9.size;

      (0, _webglChecks.assertWebGL2Context)(this.gl);
      this.gl.bindBufferRange(target, index, this.handle, offset, size);
      return this;
    }
  }, {
    key: 'unbindRange',
    value: function unbindRange() {
      var _ref10 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref10$target = _ref10.target,
          target = _ref10$target === undefined ? this.target : _ref10$target,
          index = _ref10.index;

      (0, _webglChecks.assertWebGL2Context)(this.gl);
      this.gl.bindBufferBase(target, index, null);
      return this;
    }
  }]);

  return Buffer;
}();

exports.default = Buffer;
//# sourceMappingURL=data:application/json;base64,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