'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createGLContext = createGLContext;
exports.getGLExtension = getGLExtension;
exports.poll = poll;
exports.glContextWithState = glContextWithState;
exports.glGetDebugInfo = glGetDebugInfo;

var _webglDebug = require('webgl-debug');

var _webglDebug2 = _interopRequireDefault(_webglDebug);

var _webglTypes = require('./webgl-types');

var _webglChecks = require('./webgl-checks');

var _queryManager = require('./helpers/query-manager');

var _queryManager2 = _interopRequireDefault(_queryManager);

var _utils = require('../utils');

var _globals = require('../utils/globals');

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } } // WebGLRenderingContext related methods
/* eslint-disable no-try-catch, no-loop-func */


/* global document */

var luma = _globals.global.luma;


var ERR_WEBGL_MISSING_BROWSER = 'WebGL API is missing. Check your if your browser supports WebGL or\ninstall a recent version of a major browser.';

var ERR_WEBGL_MISSING_NODE = 'WebGL API is missing. To run luma.gl under Node.js, please "npm install gl"\nand import \'luma.gl/headless\' before importing \'luma.gl\'.';

var ERR_HEADLESSGL_NOT_AVAILABLE = 'Cannot create headless WebGL context, headlessGL not available';

var ERR_HEADLESSGL_FAILED = 'headlessGL failed to create headless WebGL context';

// Checks if WebGL is enabled and creates a context for using WebGL.
/* eslint-disable complexity, max-statements */
function createGLContext() {
  var opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
  var canvas = opts.canvas;
  var _opts$width = opts.width,
      width = _opts$width === undefined ? 800 : _opts$width,
      _opts$height = opts.height,
      height = _opts$height === undefined ? 600 : _opts$height,
      _opts$webgl = opts.webgl2,
      webgl2 = _opts$webgl === undefined ? false : _opts$webgl,
      _opts$debug = opts.debug,
      debug = _opts$debug === undefined ? true : _opts$debug;


  var gl = void 0;

  if (!_utils.isBrowser) {
    gl = _createHeadlessContext(width, height, opts);
  } else {
    // Create browser gl context
    if (!_webglTypes.webGLTypesAvailable) {
      throw new Error(ERR_WEBGL_MISSING_BROWSER);
    }
    // Make sure we have a canvas
    canvas = canvas;
    if (typeof canvas === 'string') {
      if (!_utils.isPageLoaded) {
        throw new Error('createGLContext called on canvas \'' + canvas + '\' before page was loaded');
      }
      canvas = document.getElementById(canvas);
    }
    if (!canvas) {
      canvas = _createCanvas();
    }

    canvas.addEventListener('webglcontextcreationerror', function (e) {
      _utils.log.log(0, e.statusMessage || 'Unknown error');
    }, false);

    // Prefer webgl2 over webgl1, prefer conformant over experimental
    if (webgl2) {
      gl = gl || canvas.getContext('webgl2', opts);
      gl = gl || canvas.getContext('experimental-webgl2', opts);
    } else {
      gl = gl || canvas.getContext('webgl', opts);
      gl = gl || canvas.getContext('experimental-webgl', opts);
    }

    (0, _assert2.default)(gl, 'Failed to create WebGLRenderingContext');
  }

  if (_utils.isBrowser && debug) {
    var debugGL = _webglDebug2.default.makeDebugContext(gl, throwOnError, validateArgsAndLog);

    var WebGLDebugContext = function WebGLDebugContext() {
      _classCallCheck(this, WebGLDebugContext);
    };

    Object.assign(WebGLDebugContext.prototype, debugGL);
    gl = debugGL;
    gl.debug = true;
    _utils.log.priority = _utils.log.priority < 1 ? 1 : _utils.log.priority;

    logInfo(gl);
  }

  return gl;
}

// Create a canvas set to 100%
// TODO - remove
function _createCanvas() {
  var canvas = document.createElement('canvas');
  canvas.id = 'lumagl-canvas';
  canvas.style.width = '100%';
  canvas.style.height = '100%';
  // adds the canvas to the body element
  _utils.pageLoadPromise.then(function (document) {
    var body = document.body;
    body.insertBefore(canvas, body.firstChild);
  });
  return canvas;
}

function _createHeadlessContext(width, height, opts) {
  // Create headless gl context
  if (!_webglTypes.webGLTypesAvailable) {
    throw new Error(ERR_WEBGL_MISSING_NODE);
  }
  if (!luma.globals.headlessGL) {
    throw new Error(ERR_HEADLESSGL_NOT_AVAILABLE);
  }
  var gl = luma.globals.headlessGL(width, height, opts);
  if (!gl) {
    throw new Error(ERR_HEADLESSGL_FAILED);
  }
  return gl;
}

// Returns the extension or throws an error
function getGLExtension(gl, extensionName) {
  // assertWebGLContext(gl);
  var ERROR = 'Illegal arg to getExtension';
  (0, _assert2.default)(gl instanceof _webglTypes.WebGLRenderingContext, ERROR);
  (0, _assert2.default)(typeof extensionName === 'string', ERROR);
  var extension = gl.getExtension(extensionName);
  (0, _assert2.default)(extension, extensionName + ' not supported!');
  return extension;
}

// POLLING FOR PENDING QUERIES

// Calling this function checks all pending queries for completion
function poll(gl) {
  (0, _webglChecks.assertWebGLContext)(gl);
  _queryManager2.default.poll(gl);
}

// VERY LIMITED / BASIC GL STATE MANAGEMENT

// Executes a function with gl states temporarily set, exception safe
// Currently support scissor test and framebuffer binding
function glContextWithState(gl, _ref, func) {
  var scissorTest = _ref.scissorTest,
      framebuffer = _ref.framebuffer;

  // assertWebGLContext(gl);

  var scissorTestWasEnabled = void 0;
  if (scissorTest) {
    scissorTestWasEnabled = gl.isEnabled(gl.SCISSOR_TEST);
    var x = scissorTest.x,
        y = scissorTest.y,
        w = scissorTest.w,
        h = scissorTest.h;

    gl.enable(gl.SCISSOR_TEST);
    gl.scissor(x, y, w, h);
  }

  if (framebuffer) {
    // TODO - was there any previously set frame buffer we need to remember?
    framebuffer.bind();
  }

  var value = void 0;
  try {
    value = func(gl);
  } finally {
    if (!scissorTestWasEnabled) {
      gl.disable(gl.SCISSOR_TEST);
    }
    if (framebuffer) {
      // TODO - was there any previously set frame buffer?
      // TODO - delegate "unbind" to Framebuffer object?
      gl.bindFramebuffer(gl.FRAMEBUFFER, null);
    }
  }

  return value;
}

// DEBUG INFO

/**
 * Provides strings identifying the GPU vendor and driver.
 * https://www.khronos.org/registry/webgl/extensions/WEBGL_debug_renderer_info/
 * @param {WebGLRenderingContext} gl - context
 * @return {Object} - 'vendor' and 'renderer' string fields.
 */
function glGetDebugInfo(gl) {
  var info = gl.getExtension('WEBGL_debug_renderer_info');
  // We can't determine if 'WEBGL_debug_renderer_info' is supported by
  // checking whether info is null here. Firefox doesn't follow the
  // specs by returning null for unsupported extension. Instead,
  // it returns an object without GL_UNMASKED_VENDOR_WEBGL and GL_UNMASKED_RENDERER_WEBGL.
  return {
    vendor: info && info.UNMASKED_VENDOR_WEBGL ? gl.getParameter(info.UNMASKED_VENDOR_WEBGL) : 'unknown',
    renderer: info && info.UNMASKED_RENDERER_WEBGL ? gl.getParameter(info.UNMASKED_RENDERER_WEBGL) : 'unknown'
  };
}

function logInfo(gl) {
  var webGL = (0, _webglChecks.isWebGL2Context)(gl) ? 'WebGL2' : 'WebGL1';
  var info = glGetDebugInfo(gl);
  var driver = info ? '(' + info.vendor + ' ' + info.renderer + ')' : '';
  var debug = gl.debug ? 'debug' : '';
  _utils.log.log(0, 'luma.gl: Created ' + webGL + ' ' + debug + ' context ' + driver, gl);

  // const extensions = gl.getSupportedExtensions();
  // log.log(0, `Supported extensions: [${extensions.join(', ')}]`);
}

// DEBUG TRACING

function getFunctionString(functionName, functionArgs) {
  var args = _webglDebug2.default.glFunctionArgsToString(functionName, functionArgs);
  args = '' + args.slice(0, 100) + (args.length > 100 ? '...' : '');
  return 'gl.' + functionName + '(' + args + ')';
}

function throwOnError(err, functionName, args) {
  var errorMessage = _webglDebug2.default.glEnumToString(err);
  var functionArgs = _webglDebug2.default.glFunctionArgsToString(functionName, args);
  throw new Error(errorMessage + ' was caused by call to: ' + ('gl.' + functionName + '(' + functionArgs + ')'));
}

// Don't generate function string until it is needed
function validateArgsAndLog(functionName, functionArgs) {
  var functionString = void 0;
  if (_utils.log.priority >= 4) {
    functionString = getFunctionString(functionName, functionArgs);
    _utils.log.info(4, '' + functionString);
  }

  var _iteratorNormalCompletion = true;
  var _didIteratorError = false;
  var _iteratorError = undefined;

  try {
    for (var _iterator = functionArgs[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
      var arg = _step.value;

      if (arg === undefined) {
        functionString = functionString || getFunctionString(functionName, functionArgs);
        throw new Error('Undefined argument: ' + functionString);
      }
    }
  } catch (err) {
    _didIteratorError = true;
    _iteratorError = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion && _iterator.return) {
        _iterator.return();
      }
    } finally {
      if (_didIteratorError) {
        throw _iteratorError;
      }
    }
  }

  if (_utils.log.break) {
    functionString = functionString || getFunctionString(functionName, functionArgs);
    var isBreakpoint = _utils.log.break && _utils.log.break.every(function (breakString) {
      return functionString.indexOf(breakString) !== -1;
    });

    /* eslint-disable no-debugger */
    if (isBreakpoint) {
      debugger;
    }
    /* eslint-enable no-debugger */
  }
}
//# sourceMappingURL=data:application/json;base64,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