'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

var _webgl = require('./webgl');

var _webglChecks = require('./webgl-checks');

var _texture2d = require('./texture-2d');

var _texture2d2 = _interopRequireDefault(_texture2d);

var _renderbuffer = require('./renderbuffer');

var _renderbuffer2 = _interopRequireDefault(_renderbuffer);

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

var _utils = require('../utils');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

// Returns number of components in a specific WebGL format
function glFormatToComponents(format) {
  switch (format) {
    case _webgl.GL.ALPHA:
      return 1;
    case _webgl.GL.RGB:
      return 3;
    case _webgl.GL.RGBA:
      return 4;
    default:
      throw new Error('Unknown format');
  }
}

var Framebuffer = function () {
  _createClass(Framebuffer, null, [{
    key: 'makeFrom',
    value: function makeFrom(gl) {
      var object = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      return object instanceof Framebuffer ? object :
      // Use .handle (e.g from stack.gl's gl-buffer), else use buffer directly
      new Framebuffer(gl, { handle: object.handle || object });
    }

    /* eslint-disable max-statements */

  }]);

  function Framebuffer(gl) {
    var opts = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    _classCallCheck(this, Framebuffer);

    (0, _webglChecks.assertWebGLContext)(gl);

    var id = opts.id;


    var handle = gl.createFramebuffer();
    if (!handle) {
      throw new Error('Failed to create WebGL Framebuffer');
    }

    this.gl = gl;
    this.id = (0, _utils.uid)(id);
    this.handle = handle;
    this.colorBuffer = null;
    this.depthBuffer = null;
    this.stencilBuffer = null;
    this.texture = null;
    this.userData = {};
    this.width = 0;
    this.height = 0;
    Object.seal(this);

    this.resize(opts);
  }
  /* eslint-enable max-statements */

  _createClass(Framebuffer, [{
    key: 'delete',
    value: function _delete() {
      var gl = this.gl;

      gl.deleteFramebuffer(this.handle);
    }

    // SIMPLIFIED INTERFACE

  }, {
    key: 'resize',
    value: function resize(_ref) {
      var width = _ref.width,
          height = _ref.height;

      this.update({ width: width, height: height });
    }

    /* eslint-disable max-statements */

  }, {
    key: 'update',
    value: function update(_ref2) {
      var _ref2$width = _ref2.width,
          width = _ref2$width === undefined ? 1 : _ref2$width,
          _ref2$height = _ref2.height,
          height = _ref2$height === undefined ? 1 : _ref2$height,
          _ref2$depth = _ref2.depth,
          depth = _ref2$depth === undefined ? true : _ref2$depth,
          _ref2$minFilter = _ref2.minFilter,
          minFilter = _ref2$minFilter === undefined ? _webgl.GL.NEAREST : _ref2$minFilter,
          _ref2$magFilter = _ref2.magFilter,
          magFilter = _ref2$magFilter === undefined ? _webgl.GL.NEAREST : _ref2$magFilter,
          _ref2$format = _ref2.format,
          format = _ref2$format === undefined ? _webgl.GL.RGBA : _ref2$format,
          _ref2$type = _ref2.type,
          type = _ref2$type === undefined ? _webgl.GL.UNSIGNED_BYTE : _ref2$type;

      (0, _assert2.default)(width >= 0 && height >= 0, 'Width and height need to be integers');
      if (width === this.width && height === this.height) {
        return;
      }

      _utils.log.log(2, 'Resizing framebuffer ' + this.id + ' to ' + width + 'x' + height);

      var gl = this.gl;

      // TODO - do we need to reallocate the framebuffer?

      var colorBuffer = new _texture2d2.default(gl, {
        minFilter: this.minFilter,
        magFilter: this.magFilter
      })
      // TODO - should be handled by Texture2D constructor?
      .setImageData({
        data: null,
        width: width,
        height: height,
        type: type,
        format: format
      });

      this.attachTexture({
        attachment: _webgl.GL.COLOR_ATTACHMENT0,
        texture: colorBuffer
      });

      if (this.colorBuffer) {
        this.colorBuffer.delete();
      }
      this.colorBuffer = colorBuffer;
      this.texture = colorBuffer;

      // Add a depth buffer if requested
      if (depth) {
        var depthBuffer = new _renderbuffer2.default(gl).storage({
          internalFormat: _webgl.GL.DEPTH_COMPONENT16,
          width: width,
          height: height
        });
        this.attachRenderbuffer({
          attachment: _webgl.GL.DEPTH_ATTACHMENT,
          renderbuffer: depthBuffer
        });

        if (this.depthBuffer) {
          this.depthBuffer.delete();
        }
        this.depthBuffer = depthBuffer;
      }

      // Checks that framebuffer was properly set up,
      // if not, throws an explanatory error
      this.checkStatus();

      this.width = width;
      this.height = height;
    }

    // WEBGL INTERFACE

  }, {
    key: 'bind',
    value: function bind() {
      var _ref3 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref3$target = _ref3.target,
          target = _ref3$target === undefined ? _webgl.GL.FRAMEBUFFER : _ref3$target;

      var gl = this.gl;

      gl.bindFramebuffer(target, this.handle);
      return this;
    }
  }, {
    key: 'unbind',
    value: function unbind() {
      var _ref4 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref4$target = _ref4.target,
          target = _ref4$target === undefined ? _webgl.GL.FRAMEBUFFER : _ref4$target;

      var gl = this.gl;

      gl.bindFramebuffer(target, null);
      return this;
    }

    //
    // NOTE: Slow requires roundtrip to GPU
    // App can provide pixelArray or have it auto allocated by this method
    // @returns {Uint8Array|Uint16Array|FloatArray} - pixel array,
    //  newly allocated by this method unless provided by app.

  }, {
    key: 'readPixels',
    value: function readPixels(_ref5) {
      var _ref5$x = _ref5.x,
          x = _ref5$x === undefined ? 0 : _ref5$x,
          _ref5$y = _ref5.y,
          y = _ref5$y === undefined ? 0 : _ref5$y,
          width = _ref5.width,
          height = _ref5.height,
          _ref5$format = _ref5.format,
          format = _ref5$format === undefined ? _webgl.GL.RGBA : _ref5$format,
          type = _ref5.type,
          _ref5$pixelArray = _ref5.pixelArray,
          pixelArray = _ref5$pixelArray === undefined ? null : _ref5$pixelArray;
      var gl = this.gl;

      // Deduce type and allocated pixelArray if needed

      if (!pixelArray) {
        // Allocate pixel array if not already available, using supplied type
        type = type || _webgl.GL.UNSIGNED_BYTE;
        var ArrayType = (0, _webgl.glTypeToArray)(type);
        var components = glFormatToComponents(format);
        // TODO - check for composite type (components = 1).
        pixelArray = pixelArray || new ArrayType(width * height * components);
      }

      // Pixel array available, if necessary, deduce type from it.
      type = type || (0, _webgl.glTypeFromArray)(pixelArray);

      this.bind();
      gl.readPixels(x, y, width, height, format, type, pixelArray);
      this.unbind();

      return pixelArray;
    }

    /**
     * Used to attach textures to a framebuffer, the textures will store
     * the various buffers.
     *
     *  The set of available attachments is larger in WebGL2, and also the
     *  extensions WEBGL_draw_buffers and WEBGL_depth_texture provide additional
     *  attachments that match or exceed the WebGL2 set.
     *
     * @param {Texture2D|TextureCube|WebGLTexture|null} opt.texture=null -
     *    default is null which unbinds the texture for the attachment
     * @param {String|Number} opt.attachment= - which attachment to bind
     *    defaults to gl.COLOR_ATTACHMENT0.
     * @param {String|Number} opt.target= - bind point, normally gl.FRAMEBUFFER
     *    (WebGL2 support separating bet)
     * @param {String|Number} opt.textureTarget= - can be used to specify
     *    faces of a cube map.
     * @returns {FrameBuffer} returns itself to enable chaining
     */

  }, {
    key: 'attachTexture',
    value: function attachTexture() {
      var _ref6 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref6$texture = _ref6.texture,
          texture = _ref6$texture === undefined ? null : _ref6$texture,
          _ref6$target = _ref6.target,
          target = _ref6$target === undefined ? _webgl.GL.FRAMEBUFFER : _ref6$target,
          _ref6$attachment = _ref6.attachment,
          attachment = _ref6$attachment === undefined ? _webgl.GL.COLOR_ATTACHMENT0 : _ref6$attachment,
          _ref6$textureTarget = _ref6.textureTarget,
          textureTarget = _ref6$textureTarget === undefined ? _webgl.GL.TEXTURE_2D : _ref6$textureTarget,
          _ref6$mipmapLevel = _ref6.mipmapLevel,
          mipmapLevel = _ref6$mipmapLevel === undefined ? 0 : _ref6$mipmapLevel;

      var gl = this.gl;


      texture = texture && _texture2d2.default.makeFrom(gl, texture);

      this.bind({ target: target });

      gl.framebufferTexture2D(target, (0, _webgl.glGet)(attachment), (0, _webgl.glGet)(textureTarget), texture.handle, mipmapLevel);

      this.unbind();
      return this;
    }

    /**
     * Used to attach a framebuffer to a framebuffer, the textures will store
     * the various buffers.
     * @param {Object} opts= - named parameters
     * @param {RenderBuffer|WebGLRenderBuffer|null} opts.renderbuffer=null -
     *    renderbuffer to bind
     *    default is null which unbinds the renderbuffer for the attachment
     * @param {String|Number} opts.attachment= - which buffer to bind
     * @returns {FrameBuffer} returns itself to enable chaining
     */

  }, {
    key: 'attachRenderbuffer',
    value: function attachRenderbuffer() {
      var _ref7 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref7$renderbuffer = _ref7.renderbuffer,
          renderbuffer = _ref7$renderbuffer === undefined ? null : _ref7$renderbuffer,
          _ref7$attachment = _ref7.attachment,
          attachment = _ref7$attachment === undefined ? _webgl.GL.COLOR_ATTACHMENT0 : _ref7$attachment,
          _ref7$target = _ref7.target,
          target = _ref7$target === undefined ? _webgl.GL.FRAMEBUFFER : _ref7$target,
          _ref7$renderbufferTar = _ref7.renderbufferTarget,
          renderbufferTarget = _ref7$renderbufferTar === undefined ? _webgl.GL.RENDERBUFFER : _ref7$renderbufferTar;

      var gl = this.gl;

      renderbuffer = renderbuffer && _renderbuffer2.default.makeFrom(gl, renderbuffer);

      this.bind({ target: target });

      gl.framebufferRenderbuffer(target, (0, _webgl.glGet)(attachment), (0, _webgl.glGet)(renderbufferTarget), renderbuffer.handle);

      this.unbind({ target: target });

      return this;
    }
  }, {
    key: 'checkStatus',
    value: function checkStatus() {
      var _ref8 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref8$target = _ref8.target,
          target = _ref8$target === undefined ? _webgl.GL.FRAMEBUFFER : _ref8$target;

      var gl = this.gl;


      this.bind({ target: target });

      var status = gl.checkFramebufferStatus(target);

      this.unbind({ target: target });

      if (status !== gl.FRAMEBUFFER_COMPLETE) {
        throw new Error(this._getFrameBufferStatus(status));
      }

      return this;
    }

    // WEBGL2 INTERFACE

  }, {
    key: 'blit',
    value: function blit(_ref9) {
      var srcX0 = _ref9.srcX0,
          srcY0 = _ref9.srcY0,
          srcX1 = _ref9.srcX1,
          srcY1 = _ref9.srcY1,
          dstX0 = _ref9.dstX0,
          dstY0 = _ref9.dstY0,
          dstX1 = _ref9.dstX1,
          dstY1 = _ref9.dstY1,
          mask = _ref9.mask,
          _ref9$filter = _ref9.filter,
          filter = _ref9$filter === undefined ? _webgl.GL.NEAREST : _ref9$filter;
      var gl = this.gl;

      (0, _webglChecks.assertWebGL2Context)(gl);
      gl.blitFramebuffer(srcX0, srcY0, srcX1, srcY1, dstX0, dstY0, dstX1, dstY1, mask, filter);
      return this;
    }
  }, {
    key: 'textureLayer',
    value: function textureLayer() {
      var _ref10 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref10$target = _ref10.target,
          target = _ref10$target === undefined ? _webgl.GL.FRAMEBUFFER : _ref10$target,
          attachment = _ref10.attachment,
          texture = _ref10.texture,
          level = _ref10.level,
          layer = _ref10.layer;

      var gl = this.gl;

      (0, _webglChecks.assertWebGL2Context)(gl);
      gl.framebufferTextureLayer(target, attachment, texture, level, layer);
      return this;
    }
  }, {
    key: 'invalidate',
    value: function invalidate(_ref11) {
      var _ref11$target = _ref11.target,
          target = _ref11$target === undefined ? _webgl.GL.FRAMEBUFFER : _ref11$target,
          _ref11$attachments = _ref11.attachments,
          attachments = _ref11$attachments === undefined ? [] : _ref11$attachments;
      var gl = this.gl;

      (0, _webglChecks.assertWebGL2Context)(gl);
      gl.invalidateFramebuffer(target, attachments);
      return this;
    }
  }, {
    key: 'invalidateSub',
    value: function invalidateSub(_ref12) {
      var _ref12$target = _ref12.target,
          target = _ref12$target === undefined ? _webgl.GL.FRAMEBUFFER : _ref12$target,
          _ref12$attachments = _ref12.attachments,
          attachments = _ref12$attachments === undefined ? [] : _ref12$attachments,
          _ref12$x = _ref12.x,
          x = _ref12$x === undefined ? 0 : _ref12$x,
          _ref12$y = _ref12.y,
          y = _ref12$y === undefined ? 0 : _ref12$y,
          width = _ref12.width,
          height = _ref12.height;
      var gl = this.gl;

      (0, _webglChecks.assertWebGL2Context)(gl);
      gl.invalidateFramebuffer(target, attachments, x, y, width, height);
      return this;
    }

    // Selects a color buffer as the source for pixels for subsequent calls to
    // copyTexImage2D, copyTexSubImage2D, copyTexSubImage3D or readPixels.
    // src
    //  gl.BACK: Reads from the back color buffer.
    //  gl.NONE: Reads from no color buffer.
    //  gl.COLOR_ATTACHMENT{0-15}: Reads from one of 16 color attachment buffers.

  }, {
    key: 'readBuffer',
    value: function readBuffer(_ref13) {
      var src = _ref13.src;
      var gl = this.gl;

      (0, _webglChecks.assertWebGL2Context)(gl);
      gl.readBuffer(src);
      return this;
    }

    // @returns {GLint}

  }, {
    key: 'alphaSize',
    value: function alphaSize() {
      return this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_ALPHA_SIZE);
    }

    // @returns {GLint}

  }, {
    key: 'blueSize',
    value: function blueSize() {
      return this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_BLUE_SIZE);
    }

    // @returns {GLenum}

  }, {
    key: 'colorEncoding',
    value: function colorEncoding() {
      return this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING);
    }

    // @returns {GLenum}

  }, {
    key: 'componentType',
    value: function componentType() {
      return this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE);
    }

    // @returns {GLint}

  }, {
    key: 'depthSize',
    value: function depthSize() {
      return this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_DEPTH_SIZE);
    }

    // @returns {GLint}

  }, {
    key: 'greenSize',
    value: function greenSize() {
      return this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_GREEN_SIZE);
    }

    // @returns {WebGLRenderbuffer|WebGLTexture}

  }, {
    key: 'objectName',
    value: function objectName() {
      return this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_OBJECT_NAME);
    }

    // @returns {GLenum}

  }, {
    key: 'objectType',
    value: function objectType() {
      return this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE);
    }

    // @returns {GLint}

  }, {
    key: 'redSize',
    value: function redSize() {
      return this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_RED_SIZE);
    }

    // @returns {GLint}

  }, {
    key: 'stencilSize',
    value: function stencilSize() {
      return this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_STENCIL_SIZE);
    }

    // @returns {GLint}

  }, {
    key: 'cubeMapFace',
    value: function cubeMapFace() {
      return this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE);
    }

    // @returns {GLint}

  }, {
    key: 'layer',
    value: function layer() {
      return this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER);
    }

    // @returns {GLint}

  }, {
    key: 'level',
    value: function level() {
      return this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL);
    }
  }, {
    key: 'getParameters',
    value: function getParameters() {
      return {
        redSize: this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_RED_SIZE),
        greenSize: this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_GREEN_SIZE),
        blueSize: this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_BLUE_SIZE),
        alphaSize: this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_ALPHA_SIZE),

        depthSize: this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_DEPTH_SIZE),
        stencilSize: this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_STENCIL_SIZE),

        colorEncoding: this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING),
        componentType: this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE),
        objectName: this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_OBJECT_NAME),
        objectType: this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE),
        cubeMapFace: this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE),
        layer: this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER),
        level: this.getAttachmentParameter(_webgl.GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL)
      };
    }

    // (OpenGL ES 3.0.4 §6.1.13, similar to glGetFramebufferAttachmentParameteriv)
    // Return the value for the passed pname given the target and attachment.
    // The type returned is the natural type for the requested pname:
    // pname returned type
    // FRAMEBUFFER_ATTACHMENT_ALPHA_SIZE GLint
    // FRAMEBUFFER_ATTACHMENT_BLUE_SIZE  GLint
    // FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING GLenum
    // FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE GLenum
    // FRAMEBUFFER_ATTACHMENT_DEPTH_SIZE GLint
    // FRAMEBUFFER_ATTACHMENT_GREEN_SIZE GLint
    // FRAMEBUFFER_ATTACHMENT_OBJECT_NAME  WebGLRenderbuffer or WebGLTexture
    // FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE  GLenum
    // FRAMEBUFFER_ATTACHMENT_RED_SIZE GLint
    // FRAMEBUFFER_ATTACHMENT_STENCIL_SIZE GLint
    // FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE  GLint
    // FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER  GLint
    // FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL  GLint
    // If pname is not in the table above, generates an INVALID_ENUM error.
    // If an OpenGL error is generated, returns null.

  }, {
    key: 'getAttachmentParameter',
    value: function getAttachmentParameter() {
      var _ref14 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          pname = _ref14.pname,
          target = _ref14.target,
          attachment = _ref14.attachment;

      var gl = this.gl;

      (0, _webglChecks.assertWebGL2Context)(gl);
      var value = gl.getFramebufferAttachmentParameter(target, attachment, pname);
      return value;
    }

    /* eslint-disable max-len */

  }, {
    key: '_getFrameBufferStatus',
    value: function _getFrameBufferStatus(status) {
      var error = void 0;
      switch (status) {
        case _webgl.GL.FRAMEBUFFER_COMPLETE:
          error = 'Success. Framebuffer is correctly set up';
          break;
        case _webgl.GL.FRAMEBUFFER_INCOMPLETE_ATTACHMENT:
          error = 'The attachment types are mismatched or not all framebuffer attachment points are framebuffer attachment complete.';
          break;
        case _webgl.GL.FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT:
          error = 'There is no attachment.';
          break;
        case _webgl.GL.FRAMEBUFFER_INCOMPLETE_DIMENSIONS:
          error = 'Height and width of the attachment are not the same.';
          break;
        case _webgl.GL.FRAMEBUFFER_UNSUPPORTED:
          error = 'The format of the attachment is not supported or if depth and stencil attachments are not the same renderbuffer.';
          break;
        // When using a WebGL 2 context, the following values can be returned
        case _webgl.GL.FRAMEBUFFER_INCOMPLETE_MULTISAMPLE:
          error = 'The values of GL.RENDERBUFFER_SAMPLES are different among attached renderbuffers, or are non-zero if the attached images are a mix of renderbuffers and textures.';
          break;
        default:
          error = 'Framebuffer error ' + status;
          break;
      }
      return error;
    }
    /* eslint-enable max-len */

  }]);

  return Framebuffer;
}();

exports.default = Framebuffer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy93ZWJnbC9mcmFtZWJ1ZmZlci5qcyJdLCJuYW1lcyI6WyJnbEZvcm1hdFRvQ29tcG9uZW50cyIsImZvcm1hdCIsIkFMUEhBIiwiUkdCIiwiUkdCQSIsIkVycm9yIiwiRnJhbWVidWZmZXIiLCJnbCIsIm9iamVjdCIsImhhbmRsZSIsIm9wdHMiLCJpZCIsImNyZWF0ZUZyYW1lYnVmZmVyIiwiY29sb3JCdWZmZXIiLCJkZXB0aEJ1ZmZlciIsInN0ZW5jaWxCdWZmZXIiLCJ0ZXh0dXJlIiwidXNlckRhdGEiLCJ3aWR0aCIsImhlaWdodCIsIk9iamVjdCIsInNlYWwiLCJyZXNpemUiLCJkZWxldGVGcmFtZWJ1ZmZlciIsInVwZGF0ZSIsImRlcHRoIiwibWluRmlsdGVyIiwiTkVBUkVTVCIsIm1hZ0ZpbHRlciIsInR5cGUiLCJVTlNJR05FRF9CWVRFIiwibG9nIiwic2V0SW1hZ2VEYXRhIiwiZGF0YSIsImF0dGFjaFRleHR1cmUiLCJhdHRhY2htZW50IiwiQ09MT1JfQVRUQUNITUVOVDAiLCJkZWxldGUiLCJzdG9yYWdlIiwiaW50ZXJuYWxGb3JtYXQiLCJERVBUSF9DT01QT05FTlQxNiIsImF0dGFjaFJlbmRlcmJ1ZmZlciIsIkRFUFRIX0FUVEFDSE1FTlQiLCJyZW5kZXJidWZmZXIiLCJjaGVja1N0YXR1cyIsInRhcmdldCIsIkZSQU1FQlVGRkVSIiwiYmluZEZyYW1lYnVmZmVyIiwieCIsInkiLCJwaXhlbEFycmF5IiwiQXJyYXlUeXBlIiwiY29tcG9uZW50cyIsImJpbmQiLCJyZWFkUGl4ZWxzIiwidW5iaW5kIiwidGV4dHVyZVRhcmdldCIsIlRFWFRVUkVfMkQiLCJtaXBtYXBMZXZlbCIsIm1ha2VGcm9tIiwiZnJhbWVidWZmZXJUZXh0dXJlMkQiLCJyZW5kZXJidWZmZXJUYXJnZXQiLCJSRU5ERVJCVUZGRVIiLCJmcmFtZWJ1ZmZlclJlbmRlcmJ1ZmZlciIsInN0YXR1cyIsImNoZWNrRnJhbWVidWZmZXJTdGF0dXMiLCJGUkFNRUJVRkZFUl9DT01QTEVURSIsIl9nZXRGcmFtZUJ1ZmZlclN0YXR1cyIsInNyY1gwIiwic3JjWTAiLCJzcmNYMSIsInNyY1kxIiwiZHN0WDAiLCJkc3RZMCIsImRzdFgxIiwiZHN0WTEiLCJtYXNrIiwiZmlsdGVyIiwiYmxpdEZyYW1lYnVmZmVyIiwibGV2ZWwiLCJsYXllciIsImZyYW1lYnVmZmVyVGV4dHVyZUxheWVyIiwiYXR0YWNobWVudHMiLCJpbnZhbGlkYXRlRnJhbWVidWZmZXIiLCJzcmMiLCJyZWFkQnVmZmVyIiwiZ2V0QXR0YWNobWVudFBhcmFtZXRlciIsIkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfQUxQSEFfU0laRSIsIkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfQkxVRV9TSVpFIiwiRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9DT0xPUl9FTkNPRElORyIsIkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfQ09NUE9ORU5UX1RZUEUiLCJGUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX0RFUFRIX1NJWkUiLCJGUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX0dSRUVOX1NJWkUiLCJGUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX09CSkVDVF9OQU1FIiwiRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9PQkpFQ1RfVFlQRSIsIkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfUkVEX1NJWkUiLCJGUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX1NURU5DSUxfU0laRSIsIkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfVEVYVFVSRV9DVUJFX01BUF9GQUNFIiwiRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9URVhUVVJFX0xBWUVSIiwiRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9URVhUVVJFX0xFVkVMIiwicmVkU2l6ZSIsImdyZWVuU2l6ZSIsImJsdWVTaXplIiwiYWxwaGFTaXplIiwiZGVwdGhTaXplIiwic3RlbmNpbFNpemUiLCJjb2xvckVuY29kaW5nIiwiY29tcG9uZW50VHlwZSIsIm9iamVjdE5hbWUiLCJvYmplY3RUeXBlIiwiY3ViZU1hcEZhY2UiLCJwbmFtZSIsInZhbHVlIiwiZ2V0RnJhbWVidWZmZXJBdHRhY2htZW50UGFyYW1ldGVyIiwiZXJyb3IiLCJGUkFNRUJVRkZFUl9JTkNPTVBMRVRFX0FUVEFDSE1FTlQiLCJGUkFNRUJVRkZFUl9JTkNPTVBMRVRFX01JU1NJTkdfQVRUQUNITUVOVCIsIkZSQU1FQlVGRkVSX0lOQ09NUExFVEVfRElNRU5TSU9OUyIsIkZSQU1FQlVGRkVSX1VOU1VQUE9SVEVEIiwiRlJBTUVCVUZGRVJfSU5DT01QTEVURV9NVUxUSVNBTVBMRSJdLCJtYXBwaW5ncyI6Ijs7Ozs7Ozs7O0FBQUE7O0FBQ0E7O0FBQ0E7Ozs7QUFDQTs7OztBQUNBOzs7O0FBQ0E7Ozs7OztBQUVBO0FBQ0EsU0FBU0Esb0JBQVQsQ0FBOEJDLE1BQTlCLEVBQXNDO0FBQ3BDLFVBQVFBLE1BQVI7QUFDQSxTQUFLLFVBQUdDLEtBQVI7QUFBZSxhQUFPLENBQVA7QUFDZixTQUFLLFVBQUdDLEdBQVI7QUFBYSxhQUFPLENBQVA7QUFDYixTQUFLLFVBQUdDLElBQVI7QUFBYyxhQUFPLENBQVA7QUFDZDtBQUFTLFlBQU0sSUFBSUMsS0FBSixDQUFVLGdCQUFWLENBQU47QUFKVDtBQU1EOztJQUVvQkMsVzs7OzZCQUNIQyxFLEVBQWlCO0FBQUEsVUFBYkMsTUFBYSx1RUFBSixFQUFJOztBQUMvQixhQUFPQSxrQkFBa0JGLFdBQWxCLEdBQWdDRSxNQUFoQztBQUNMO0FBQ0EsVUFBSUYsV0FBSixDQUFnQkMsRUFBaEIsRUFBb0IsRUFBQ0UsUUFBUUQsT0FBT0MsTUFBUCxJQUFpQkQsTUFBMUIsRUFBcEIsQ0FGRjtBQUdEOztBQUVEOzs7O0FBQ0EsdUJBQVlELEVBQVosRUFBMkI7QUFBQSxRQUFYRyxJQUFXLHVFQUFKLEVBQUk7O0FBQUE7O0FBQ3pCLHlDQUFtQkgsRUFBbkI7O0FBRHlCLFFBR2xCSSxFQUhrQixHQUdaRCxJQUhZLENBR2xCQyxFQUhrQjs7O0FBS3pCLFFBQU1GLFNBQVNGLEdBQUdLLGlCQUFILEVBQWY7QUFDQSxRQUFJLENBQUNILE1BQUwsRUFBYTtBQUNYLFlBQU0sSUFBSUosS0FBSixDQUFVLG9DQUFWLENBQU47QUFDRDs7QUFFRCxTQUFLRSxFQUFMLEdBQVVBLEVBQVY7QUFDQSxTQUFLSSxFQUFMLEdBQVUsZ0JBQUlBLEVBQUosQ0FBVjtBQUNBLFNBQUtGLE1BQUwsR0FBY0EsTUFBZDtBQUNBLFNBQUtJLFdBQUwsR0FBbUIsSUFBbkI7QUFDQSxTQUFLQyxXQUFMLEdBQW1CLElBQW5CO0FBQ0EsU0FBS0MsYUFBTCxHQUFxQixJQUFyQjtBQUNBLFNBQUtDLE9BQUwsR0FBZSxJQUFmO0FBQ0EsU0FBS0MsUUFBTCxHQUFnQixFQUFoQjtBQUNBLFNBQUtDLEtBQUwsR0FBYSxDQUFiO0FBQ0EsU0FBS0MsTUFBTCxHQUFjLENBQWQ7QUFDQUMsV0FBT0MsSUFBUCxDQUFZLElBQVo7O0FBRUEsU0FBS0MsTUFBTCxDQUFZWixJQUFaO0FBQ0Q7QUFDRDs7Ozs4QkFFUztBQUFBLFVBQ0FILEVBREEsR0FDTSxJQUROLENBQ0FBLEVBREE7O0FBRVBBLFNBQUdnQixpQkFBSCxDQUFxQixLQUFLZCxNQUExQjtBQUNEOztBQUVEOzs7O2lDQUV3QjtBQUFBLFVBQWhCUyxLQUFnQixRQUFoQkEsS0FBZ0I7QUFBQSxVQUFUQyxNQUFTLFFBQVRBLE1BQVM7O0FBQ3RCLFdBQUtLLE1BQUwsQ0FBWSxFQUFDTixZQUFELEVBQVFDLGNBQVIsRUFBWjtBQUNEOztBQUVEOzs7O2tDQVNHO0FBQUEsOEJBUERELEtBT0M7QUFBQSxVQVBEQSxLQU9DLCtCQVBPLENBT1A7QUFBQSwrQkFOREMsTUFNQztBQUFBLFVBTkRBLE1BTUMsZ0NBTlEsQ0FNUjtBQUFBLDhCQUxETSxLQUtDO0FBQUEsVUFMREEsS0FLQywrQkFMTyxJQUtQO0FBQUEsa0NBSkRDLFNBSUM7QUFBQSxVQUpEQSxTQUlDLG1DQUpXLFVBQUdDLE9BSWQ7QUFBQSxrQ0FIREMsU0FHQztBQUFBLFVBSERBLFNBR0MsbUNBSFcsVUFBR0QsT0FHZDtBQUFBLCtCQUZEMUIsTUFFQztBQUFBLFVBRkRBLE1BRUMsZ0NBRlEsVUFBR0csSUFFWDtBQUFBLDZCQUREeUIsSUFDQztBQUFBLFVBRERBLElBQ0MsOEJBRE0sVUFBR0MsYUFDVDs7QUFDRCw0QkFBT1osU0FBUyxDQUFULElBQWNDLFVBQVUsQ0FBL0IsRUFBa0Msc0NBQWxDO0FBQ0EsVUFBSUQsVUFBVSxLQUFLQSxLQUFmLElBQXdCQyxXQUFXLEtBQUtBLE1BQTVDLEVBQW9EO0FBQ2xEO0FBQ0Q7O0FBRUQsaUJBQUlZLEdBQUosQ0FBUSxDQUFSLDRCQUFtQyxLQUFLcEIsRUFBeEMsWUFBaURPLEtBQWpELFNBQTBEQyxNQUExRDs7QUFOQyxVQVFNWixFQVJOLEdBUVksSUFSWixDQVFNQSxFQVJOOztBQVVEOztBQUNBLFVBQU1NLGNBQWMsd0JBQWNOLEVBQWQsRUFBa0I7QUFDcENtQixtQkFBVyxLQUFLQSxTQURvQjtBQUVwQ0UsbUJBQVcsS0FBS0E7QUFGb0IsT0FBbEI7QUFJcEI7QUFKb0IsT0FLbkJJLFlBTG1CLENBS047QUFDWkMsY0FBTSxJQURNO0FBRVpmLG9CQUZZO0FBR1pDLHNCQUhZO0FBSVpVLGtCQUpZO0FBS1o1QjtBQUxZLE9BTE0sQ0FBcEI7O0FBYUEsV0FBS2lDLGFBQUwsQ0FBbUI7QUFDakJDLG9CQUFZLFVBQUdDLGlCQURFO0FBRWpCcEIsaUJBQVNIO0FBRlEsT0FBbkI7O0FBS0EsVUFBSSxLQUFLQSxXQUFULEVBQXNCO0FBQ3BCLGFBQUtBLFdBQUwsQ0FBaUJ3QixNQUFqQjtBQUNEO0FBQ0QsV0FBS3hCLFdBQUwsR0FBbUJBLFdBQW5CO0FBQ0EsV0FBS0csT0FBTCxHQUFlSCxXQUFmOztBQUVBO0FBQ0EsVUFBSVksS0FBSixFQUFXO0FBQ1QsWUFBTVgsY0FBYywyQkFBaUJQLEVBQWpCLEVBQXFCK0IsT0FBckIsQ0FBNkI7QUFDL0NDLDBCQUFnQixVQUFHQyxpQkFENEI7QUFFL0N0QixzQkFGK0M7QUFHL0NDO0FBSCtDLFNBQTdCLENBQXBCO0FBS0EsYUFBS3NCLGtCQUFMLENBQXdCO0FBQ3RCTixzQkFBWSxVQUFHTyxnQkFETztBQUV0QkMsd0JBQWM3QjtBQUZRLFNBQXhCOztBQUtBLFlBQUksS0FBS0EsV0FBVCxFQUFzQjtBQUNwQixlQUFLQSxXQUFMLENBQWlCdUIsTUFBakI7QUFDRDtBQUNELGFBQUt2QixXQUFMLEdBQW1CQSxXQUFuQjtBQUNEOztBQUVEO0FBQ0E7QUFDQSxXQUFLOEIsV0FBTDs7QUFFQSxXQUFLMUIsS0FBTCxHQUFhQSxLQUFiO0FBQ0EsV0FBS0MsTUFBTCxHQUFjQSxNQUFkO0FBQ0Q7O0FBRUQ7Ozs7MkJBRXFDO0FBQUEsc0ZBQUosRUFBSTtBQUFBLCtCQUEvQjBCLE1BQStCO0FBQUEsVUFBL0JBLE1BQStCLGdDQUF0QixVQUFHQyxXQUFtQjs7QUFBQSxVQUM1QnZDLEVBRDRCLEdBQ3RCLElBRHNCLENBQzVCQSxFQUQ0Qjs7QUFFbkNBLFNBQUd3QyxlQUFILENBQW1CRixNQUFuQixFQUEyQixLQUFLcEMsTUFBaEM7QUFDQSxhQUFPLElBQVA7QUFDRDs7OzZCQUVzQztBQUFBLHNGQUFKLEVBQUk7QUFBQSwrQkFBL0JvQyxNQUErQjtBQUFBLFVBQS9CQSxNQUErQixnQ0FBdEIsVUFBR0MsV0FBbUI7O0FBQUEsVUFDOUJ2QyxFQUQ4QixHQUN4QixJQUR3QixDQUM5QkEsRUFEOEI7O0FBRXJDQSxTQUFHd0MsZUFBSCxDQUFtQkYsTUFBbkIsRUFBMkIsSUFBM0I7QUFDQSxhQUFPLElBQVA7QUFDRDs7QUFFRDtBQUNBO0FBQ0E7QUFDQTtBQUNBOzs7O3NDQVNHO0FBQUEsMEJBUERHLENBT0M7QUFBQSxVQVBEQSxDQU9DLDJCQVBHLENBT0g7QUFBQSwwQkFOREMsQ0FNQztBQUFBLFVBTkRBLENBTUMsMkJBTkcsQ0FNSDtBQUFBLFVBTEQvQixLQUtDLFNBTERBLEtBS0M7QUFBQSxVQUpEQyxNQUlDLFNBSkRBLE1BSUM7QUFBQSwrQkFIRGxCLE1BR0M7QUFBQSxVQUhEQSxNQUdDLGdDQUhRLFVBQUdHLElBR1g7QUFBQSxVQUZEeUIsSUFFQyxTQUZEQSxJQUVDO0FBQUEsbUNBRERxQixVQUNDO0FBQUEsVUFEREEsVUFDQyxvQ0FEWSxJQUNaO0FBQUEsVUFDTTNDLEVBRE4sR0FDWSxJQURaLENBQ01BLEVBRE47O0FBR0Q7O0FBQ0EsVUFBSSxDQUFDMkMsVUFBTCxFQUFpQjtBQUNmO0FBQ0FyQixlQUFPQSxRQUFRLFVBQUdDLGFBQWxCO0FBQ0EsWUFBTXFCLFlBQVksMEJBQWN0QixJQUFkLENBQWxCO0FBQ0EsWUFBTXVCLGFBQWFwRCxxQkFBcUJDLE1BQXJCLENBQW5CO0FBQ0E7QUFDQWlELHFCQUFhQSxjQUFjLElBQUlDLFNBQUosQ0FBY2pDLFFBQVFDLE1BQVIsR0FBaUJpQyxVQUEvQixDQUEzQjtBQUNEOztBQUVEO0FBQ0F2QixhQUFPQSxRQUFRLDRCQUFnQnFCLFVBQWhCLENBQWY7O0FBRUEsV0FBS0csSUFBTDtBQUNBOUMsU0FBRytDLFVBQUgsQ0FBY04sQ0FBZCxFQUFpQkMsQ0FBakIsRUFBb0IvQixLQUFwQixFQUEyQkMsTUFBM0IsRUFBbUNsQixNQUFuQyxFQUEyQzRCLElBQTNDLEVBQWlEcUIsVUFBakQ7QUFDQSxXQUFLSyxNQUFMOztBQUVBLGFBQU9MLFVBQVA7QUFDRDs7QUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O29DQXlCUTtBQUFBLHNGQUFKLEVBQUk7QUFBQSxnQ0FOTmxDLE9BTU07QUFBQSxVQU5OQSxPQU1NLGlDQU5JLElBTUo7QUFBQSwrQkFMTjZCLE1BS007QUFBQSxVQUxOQSxNQUtNLGdDQUxHLFVBQUdDLFdBS047QUFBQSxtQ0FKTlgsVUFJTTtBQUFBLFVBSk5BLFVBSU0sb0NBSk8sVUFBR0MsaUJBSVY7QUFBQSxzQ0FITm9CLGFBR007QUFBQSxVQUhOQSxhQUdNLHVDQUhVLFVBQUdDLFVBR2I7QUFBQSxvQ0FETkMsV0FDTTtBQUFBLFVBRE5BLFdBQ00scUNBRFEsQ0FDUjs7QUFBQSxVQUNDbkQsRUFERCxHQUNPLElBRFAsQ0FDQ0EsRUFERDs7O0FBR05TLGdCQUFVQSxXQUFXLG9CQUFVMkMsUUFBVixDQUFtQnBELEVBQW5CLEVBQXVCUyxPQUF2QixDQUFyQjs7QUFFQSxXQUFLcUMsSUFBTCxDQUFVLEVBQUNSLGNBQUQsRUFBVjs7QUFFQXRDLFNBQUdxRCxvQkFBSCxDQUNFZixNQURGLEVBRUUsa0JBQU1WLFVBQU4sQ0FGRixFQUdFLGtCQUFNcUIsYUFBTixDQUhGLEVBSUV4QyxRQUFRUCxNQUpWLEVBS0VpRCxXQUxGOztBQVFBLFdBQUtILE1BQUw7QUFDQSxhQUFPLElBQVA7QUFDRDs7QUFFRDs7Ozs7Ozs7Ozs7Ozt5Q0FlUTtBQUFBLHNGQUFKLEVBQUk7QUFBQSxxQ0FKTlosWUFJTTtBQUFBLFVBSk5BLFlBSU0sc0NBSlMsSUFJVDtBQUFBLG1DQUhOUixVQUdNO0FBQUEsVUFITkEsVUFHTSxvQ0FITyxVQUFHQyxpQkFHVjtBQUFBLCtCQUZOUyxNQUVNO0FBQUEsVUFGTkEsTUFFTSxnQ0FGRyxVQUFHQyxXQUVOO0FBQUEsd0NBRE5lLGtCQUNNO0FBQUEsVUFETkEsa0JBQ00seUNBRGUsVUFBR0MsWUFDbEI7O0FBQUEsVUFDQ3ZELEVBREQsR0FDTyxJQURQLENBQ0NBLEVBREQ7O0FBRU5vQyxxQkFBZUEsZ0JBQWdCLHVCQUFhZ0IsUUFBYixDQUFzQnBELEVBQXRCLEVBQTBCb0MsWUFBMUIsQ0FBL0I7O0FBRUEsV0FBS1UsSUFBTCxDQUFVLEVBQUNSLGNBQUQsRUFBVjs7QUFFQXRDLFNBQUd3RCx1QkFBSCxDQUNFbEIsTUFERixFQUVFLGtCQUFNVixVQUFOLENBRkYsRUFHRSxrQkFBTTBCLGtCQUFOLENBSEYsRUFJRWxCLGFBQWFsQyxNQUpmOztBQU9BLFdBQUs4QyxNQUFMLENBQVksRUFBQ1YsY0FBRCxFQUFaOztBQUVBLGFBQU8sSUFBUDtBQUNEOzs7a0NBRTJDO0FBQUEsc0ZBQUosRUFBSTtBQUFBLCtCQUEvQkEsTUFBK0I7QUFBQSxVQUEvQkEsTUFBK0IsZ0NBQXRCLFVBQUdDLFdBQW1COztBQUFBLFVBQ25DdkMsRUFEbUMsR0FDN0IsSUFENkIsQ0FDbkNBLEVBRG1DOzs7QUFHMUMsV0FBSzhDLElBQUwsQ0FBVSxFQUFDUixjQUFELEVBQVY7O0FBRUEsVUFBTW1CLFNBQVN6RCxHQUFHMEQsc0JBQUgsQ0FBMEJwQixNQUExQixDQUFmOztBQUVBLFdBQUtVLE1BQUwsQ0FBWSxFQUFDVixjQUFELEVBQVo7O0FBRUEsVUFBSW1CLFdBQVd6RCxHQUFHMkQsb0JBQWxCLEVBQXdDO0FBQ3RDLGNBQU0sSUFBSTdELEtBQUosQ0FBVSxLQUFLOEQscUJBQUwsQ0FBMkJILE1BQTNCLENBQVYsQ0FBTjtBQUNEOztBQUVELGFBQU8sSUFBUDtBQUNEOztBQUVEOzs7O2dDQU9HO0FBQUEsVUFKREksS0FJQyxTQUpEQSxLQUlDO0FBQUEsVUFKTUMsS0FJTixTQUpNQSxLQUlOO0FBQUEsVUFKYUMsS0FJYixTQUphQSxLQUliO0FBQUEsVUFKb0JDLEtBSXBCLFNBSm9CQSxLQUlwQjtBQUFBLFVBSERDLEtBR0MsU0FIREEsS0FHQztBQUFBLFVBSE1DLEtBR04sU0FITUEsS0FHTjtBQUFBLFVBSGFDLEtBR2IsU0FIYUEsS0FHYjtBQUFBLFVBSG9CQyxLQUdwQixTQUhvQkEsS0FHcEI7QUFBQSxVQUZEQyxJQUVDLFNBRkRBLElBRUM7QUFBQSwrQkFEREMsTUFDQztBQUFBLFVBRERBLE1BQ0MsZ0NBRFEsVUFBR2xELE9BQ1g7QUFBQSxVQUNNcEIsRUFETixHQUNZLElBRFosQ0FDTUEsRUFETjs7QUFFRCw0Q0FBb0JBLEVBQXBCO0FBQ0FBLFNBQUd1RSxlQUFILENBQ0VWLEtBREYsRUFDU0MsS0FEVCxFQUNnQkMsS0FEaEIsRUFDdUJDLEtBRHZCLEVBRUVDLEtBRkYsRUFFU0MsS0FGVCxFQUVnQkMsS0FGaEIsRUFFdUJDLEtBRnZCLEVBR0VDLElBSEYsRUFJRUMsTUFKRjtBQU1BLGFBQU8sSUFBUDtBQUNEOzs7bUNBUU87QUFBQSx1RkFBSixFQUFJO0FBQUEsaUNBTE5oQyxNQUtNO0FBQUEsVUFMTkEsTUFLTSxpQ0FMRyxVQUFHQyxXQUtOO0FBQUEsVUFKTlgsVUFJTSxVQUpOQSxVQUlNO0FBQUEsVUFITm5CLE9BR00sVUFITkEsT0FHTTtBQUFBLFVBRk4rRCxLQUVNLFVBRk5BLEtBRU07QUFBQSxVQUROQyxLQUNNLFVBRE5BLEtBQ007O0FBQUEsVUFDQ3pFLEVBREQsR0FDTyxJQURQLENBQ0NBLEVBREQ7O0FBRU4sNENBQW9CQSxFQUFwQjtBQUNBQSxTQUFHMEUsdUJBQUgsQ0FBMkJwQyxNQUEzQixFQUFtQ1YsVUFBbkMsRUFBK0NuQixPQUEvQyxFQUF3RCtELEtBQXhELEVBQStEQyxLQUEvRDtBQUNBLGFBQU8sSUFBUDtBQUNEOzs7dUNBS0U7QUFBQSxpQ0FGRG5DLE1BRUM7QUFBQSxVQUZEQSxNQUVDLGlDQUZRLFVBQUdDLFdBRVg7QUFBQSxzQ0FERG9DLFdBQ0M7QUFBQSxVQUREQSxXQUNDLHNDQURhLEVBQ2I7QUFBQSxVQUNNM0UsRUFETixHQUNZLElBRFosQ0FDTUEsRUFETjs7QUFFRCw0Q0FBb0JBLEVBQXBCO0FBQ0FBLFNBQUc0RSxxQkFBSCxDQUF5QnRDLE1BQXpCLEVBQWlDcUMsV0FBakM7QUFDQSxhQUFPLElBQVA7QUFDRDs7OzBDQVNFO0FBQUEsaUNBTkRyQyxNQU1DO0FBQUEsVUFOREEsTUFNQyxpQ0FOUSxVQUFHQyxXQU1YO0FBQUEsc0NBTERvQyxXQUtDO0FBQUEsVUFMREEsV0FLQyxzQ0FMYSxFQUtiO0FBQUEsNEJBSkRsQyxDQUlDO0FBQUEsVUFKREEsQ0FJQyw0QkFKRyxDQUlIO0FBQUEsNEJBSERDLENBR0M7QUFBQSxVQUhEQSxDQUdDLDRCQUhHLENBR0g7QUFBQSxVQUZEL0IsS0FFQyxVQUZEQSxLQUVDO0FBQUEsVUFEREMsTUFDQyxVQUREQSxNQUNDO0FBQUEsVUFDTVosRUFETixHQUNZLElBRFosQ0FDTUEsRUFETjs7QUFFRCw0Q0FBb0JBLEVBQXBCO0FBQ0FBLFNBQUc0RSxxQkFBSCxDQUF5QnRDLE1BQXpCLEVBQWlDcUMsV0FBakMsRUFBOENsQyxDQUE5QyxFQUFpREMsQ0FBakQsRUFBb0QvQixLQUFwRCxFQUEyREMsTUFBM0Q7QUFDQSxhQUFPLElBQVA7QUFDRDs7QUFFRDtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7Ozs7dUNBQ2tCO0FBQUEsVUFBTmlFLEdBQU0sVUFBTkEsR0FBTTtBQUFBLFVBQ1Q3RSxFQURTLEdBQ0gsSUFERyxDQUNUQSxFQURTOztBQUVoQiw0Q0FBb0JBLEVBQXBCO0FBQ0FBLFNBQUc4RSxVQUFILENBQWNELEdBQWQ7QUFDQSxhQUFPLElBQVA7QUFDRDs7QUFFRDs7OztnQ0FDWTtBQUNWLGFBQU8sS0FBS0Usc0JBQUwsQ0FDTCxVQUFHQyxpQ0FERSxDQUFQO0FBRUQ7O0FBRUQ7Ozs7K0JBQ1c7QUFDVCxhQUFPLEtBQUtELHNCQUFMLENBQ0wsVUFBR0UsZ0NBREUsQ0FBUDtBQUVEOztBQUVEOzs7O29DQUNnQjtBQUNkLGFBQU8sS0FBS0Ysc0JBQUwsQ0FDTCxVQUFHRyxxQ0FERSxDQUFQO0FBRUQ7O0FBRUQ7Ozs7b0NBQ2dCO0FBQ2QsYUFBTyxLQUFLSCxzQkFBTCxDQUNMLFVBQUdJLHFDQURFLENBQVA7QUFFRDs7QUFFRDs7OztnQ0FDWTtBQUNWLGFBQU8sS0FBS0osc0JBQUwsQ0FDTCxVQUFHSyxpQ0FERSxDQUFQO0FBRUQ7O0FBRUQ7Ozs7Z0NBQ1k7QUFDVixhQUFPLEtBQUtMLHNCQUFMLENBQ0wsVUFBR00saUNBREUsQ0FBUDtBQUVEOztBQUVEOzs7O2lDQUNhO0FBQ1gsYUFBTyxLQUFLTixzQkFBTCxDQUNMLFVBQUdPLGtDQURFLENBQVA7QUFFRDs7QUFFRDs7OztpQ0FDYTtBQUNYLGFBQU8sS0FBS1Asc0JBQUwsQ0FDTCxVQUFHUSxrQ0FERSxDQUFQO0FBRUQ7O0FBRUQ7Ozs7OEJBQ1U7QUFDUixhQUFPLEtBQUtSLHNCQUFMLENBQ0wsVUFBR1MsK0JBREUsQ0FBUDtBQUVEOztBQUVEOzs7O2tDQUNjO0FBQ1osYUFBTyxLQUFLVCxzQkFBTCxDQUNMLFVBQUdVLG1DQURFLENBQVA7QUFFRDs7QUFFRDs7OztrQ0FDYztBQUNaLGFBQU8sS0FBS1Ysc0JBQUwsQ0FDTCxVQUFHVyw0Q0FERSxDQUFQO0FBRUQ7O0FBRUQ7Ozs7NEJBQ1E7QUFDTixhQUFPLEtBQUtYLHNCQUFMLENBQ0wsVUFBR1ksb0NBREUsQ0FBUDtBQUVEOztBQUVEOzs7OzRCQUNRO0FBQ04sYUFBTyxLQUFLWixzQkFBTCxDQUNMLFVBQUdhLG9DQURFLENBQVA7QUFFRDs7O29DQUVlO0FBQ2QsYUFBTztBQUNMQyxpQkFBUyxLQUFLZCxzQkFBTCxDQUNQLFVBQUdTLCtCQURJLENBREo7QUFHTE0sbUJBQVcsS0FBS2Ysc0JBQUwsQ0FDVCxVQUFHTSxpQ0FETSxDQUhOO0FBS0xVLGtCQUFVLEtBQUtoQixzQkFBTCxDQUNSLFVBQUdFLGdDQURLLENBTEw7QUFPTGUsbUJBQVcsS0FBS2pCLHNCQUFMLENBQ1QsVUFBR0MsaUNBRE0sQ0FQTjs7QUFVTGlCLG1CQUFXLEtBQUtsQixzQkFBTCxDQUNULFVBQUdLLGlDQURNLENBVk47QUFZTGMscUJBQWEsS0FBS25CLHNCQUFMLENBQ1gsVUFBR1UsbUNBRFEsQ0FaUjs7QUFlTFUsdUJBQWUsS0FBS3BCLHNCQUFMLENBQ2IsVUFBR0cscUNBRFUsQ0FmVjtBQWlCTGtCLHVCQUFlLEtBQUtyQixzQkFBTCxDQUNiLFVBQUdJLHFDQURVLENBakJWO0FBbUJMa0Isb0JBQVksS0FBS3RCLHNCQUFMLENBQ1YsVUFBR08sa0NBRE8sQ0FuQlA7QUFxQkxnQixvQkFBWSxLQUFLdkIsc0JBQUwsQ0FDVixVQUFHUSxrQ0FETyxDQXJCUDtBQXVCTGdCLHFCQUFhLEtBQUt4QixzQkFBTCxDQUNYLFVBQUdXLDRDQURRLENBdkJSO0FBeUJMakIsZUFBTyxLQUFLTSxzQkFBTCxDQUNMLFVBQUdZLG9DQURFLENBekJGO0FBMkJMbkIsZUFBTyxLQUFLTyxzQkFBTCxDQUNMLFVBQUdhLG9DQURFO0FBM0JGLE9BQVA7QUE4QkQ7O0FBRUQ7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7Ozs7NkNBS1E7QUFBQSx1RkFBSixFQUFJO0FBQUEsVUFITlksS0FHTSxVQUhOQSxLQUdNO0FBQUEsVUFGTmxFLE1BRU0sVUFGTkEsTUFFTTtBQUFBLFVBRE5WLFVBQ00sVUFETkEsVUFDTTs7QUFBQSxVQUNDNUIsRUFERCxHQUNPLElBRFAsQ0FDQ0EsRUFERDs7QUFFTiw0Q0FBb0JBLEVBQXBCO0FBQ0EsVUFBTXlHLFFBQVF6RyxHQUFHMEcsaUNBQUgsQ0FDWnBFLE1BRFksRUFDSlYsVUFESSxFQUNRNEUsS0FEUixDQUFkO0FBR0EsYUFBT0MsS0FBUDtBQUNEOztBQUVEOzs7OzBDQUNzQmhELE0sRUFBUTtBQUM1QixVQUFJa0QsY0FBSjtBQUNBLGNBQVFsRCxNQUFSO0FBQ0EsYUFBSyxVQUFHRSxvQkFBUjtBQUNFZ0Qsa0JBQVEsMENBQVI7QUFDQTtBQUNGLGFBQUssVUFBR0MsaUNBQVI7QUFDRUQsa0JBQVEsbUhBQVI7QUFDQTtBQUNGLGFBQUssVUFBR0UseUNBQVI7QUFDRUYsa0JBQVEseUJBQVI7QUFDQTtBQUNGLGFBQUssVUFBR0csaUNBQVI7QUFDRUgsa0JBQVEsc0RBQVI7QUFDQTtBQUNGLGFBQUssVUFBR0ksdUJBQVI7QUFDRUosa0JBQVEsa0hBQVI7QUFDQTtBQUNGO0FBQ0EsYUFBSyxVQUFHSyxrQ0FBUjtBQUNFTCxrQkFBUSxtS0FBUjtBQUNBO0FBQ0Y7QUFDRUEseUNBQTZCbEQsTUFBN0I7QUFDQTtBQXRCRjtBQXdCQSxhQUFPa0QsS0FBUDtBQUNEO0FBQ0Q7Ozs7Ozs7a0JBbmZtQjVHLFciLCJmaWxlIjoiZnJhbWVidWZmZXIuanMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge0dMLCBnbEdldCwgZ2xUeXBlVG9BcnJheSwgZ2xUeXBlRnJvbUFycmF5fSBmcm9tICcuL3dlYmdsJztcbmltcG9ydCB7YXNzZXJ0V2ViR0xDb250ZXh0LCBhc3NlcnRXZWJHTDJDb250ZXh0fSBmcm9tICcuL3dlYmdsLWNoZWNrcyc7XG5pbXBvcnQgVGV4dHVyZTJEIGZyb20gJy4vdGV4dHVyZS0yZCc7XG5pbXBvcnQgUmVuZGVyYnVmZmVyIGZyb20gJy4vcmVuZGVyYnVmZmVyJztcbmltcG9ydCBhc3NlcnQgZnJvbSAnYXNzZXJ0JztcbmltcG9ydCB7dWlkLCBsb2d9IGZyb20gJy4uL3V0aWxzJztcblxuLy8gUmV0dXJucyBudW1iZXIgb2YgY29tcG9uZW50cyBpbiBhIHNwZWNpZmljIFdlYkdMIGZvcm1hdFxuZnVuY3Rpb24gZ2xGb3JtYXRUb0NvbXBvbmVudHMoZm9ybWF0KSB7XG4gIHN3aXRjaCAoZm9ybWF0KSB7XG4gIGNhc2UgR0wuQUxQSEE6IHJldHVybiAxO1xuICBjYXNlIEdMLlJHQjogcmV0dXJuIDM7XG4gIGNhc2UgR0wuUkdCQTogcmV0dXJuIDQ7XG4gIGRlZmF1bHQ6IHRocm93IG5ldyBFcnJvcignVW5rbm93biBmb3JtYXQnKTtcbiAgfVxufVxuXG5leHBvcnQgZGVmYXVsdCBjbGFzcyBGcmFtZWJ1ZmZlciB7XG4gIHN0YXRpYyBtYWtlRnJvbShnbCwgb2JqZWN0ID0ge30pIHtcbiAgICByZXR1cm4gb2JqZWN0IGluc3RhbmNlb2YgRnJhbWVidWZmZXIgPyBvYmplY3QgOlxuICAgICAgLy8gVXNlIC5oYW5kbGUgKGUuZyBmcm9tIHN0YWNrLmdsJ3MgZ2wtYnVmZmVyKSwgZWxzZSB1c2UgYnVmZmVyIGRpcmVjdGx5XG4gICAgICBuZXcgRnJhbWVidWZmZXIoZ2wsIHtoYW5kbGU6IG9iamVjdC5oYW5kbGUgfHwgb2JqZWN0fSk7XG4gIH1cblxuICAvKiBlc2xpbnQtZGlzYWJsZSBtYXgtc3RhdGVtZW50cyAqL1xuICBjb25zdHJ1Y3RvcihnbCwgb3B0cyA9IHt9KSB7XG4gICAgYXNzZXJ0V2ViR0xDb250ZXh0KGdsKTtcblxuICAgIGNvbnN0IHtpZH0gPSBvcHRzO1xuXG4gICAgY29uc3QgaGFuZGxlID0gZ2wuY3JlYXRlRnJhbWVidWZmZXIoKTtcbiAgICBpZiAoIWhhbmRsZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdGYWlsZWQgdG8gY3JlYXRlIFdlYkdMIEZyYW1lYnVmZmVyJyk7XG4gICAgfVxuXG4gICAgdGhpcy5nbCA9IGdsO1xuICAgIHRoaXMuaWQgPSB1aWQoaWQpO1xuICAgIHRoaXMuaGFuZGxlID0gaGFuZGxlO1xuICAgIHRoaXMuY29sb3JCdWZmZXIgPSBudWxsO1xuICAgIHRoaXMuZGVwdGhCdWZmZXIgPSBudWxsO1xuICAgIHRoaXMuc3RlbmNpbEJ1ZmZlciA9IG51bGw7XG4gICAgdGhpcy50ZXh0dXJlID0gbnVsbDtcbiAgICB0aGlzLnVzZXJEYXRhID0ge307XG4gICAgdGhpcy53aWR0aCA9IDA7XG4gICAgdGhpcy5oZWlnaHQgPSAwO1xuICAgIE9iamVjdC5zZWFsKHRoaXMpO1xuXG4gICAgdGhpcy5yZXNpemUob3B0cyk7XG4gIH1cbiAgLyogZXNsaW50LWVuYWJsZSBtYXgtc3RhdGVtZW50cyAqL1xuXG4gIGRlbGV0ZSgpIHtcbiAgICBjb25zdCB7Z2x9ID0gdGhpcztcbiAgICBnbC5kZWxldGVGcmFtZWJ1ZmZlcih0aGlzLmhhbmRsZSk7XG4gIH1cblxuICAvLyBTSU1QTElGSUVEIElOVEVSRkFDRVxuXG4gIHJlc2l6ZSh7d2lkdGgsIGhlaWdodH0pIHtcbiAgICB0aGlzLnVwZGF0ZSh7d2lkdGgsIGhlaWdodH0pO1xuICB9XG5cbiAgLyogZXNsaW50LWRpc2FibGUgbWF4LXN0YXRlbWVudHMgKi9cbiAgdXBkYXRlKHtcbiAgICB3aWR0aCA9IDEsXG4gICAgaGVpZ2h0ID0gMSxcbiAgICBkZXB0aCA9IHRydWUsXG4gICAgbWluRmlsdGVyID0gR0wuTkVBUkVTVCxcbiAgICBtYWdGaWx0ZXIgPSBHTC5ORUFSRVNULFxuICAgIGZvcm1hdCA9IEdMLlJHQkEsXG4gICAgdHlwZSA9IEdMLlVOU0lHTkVEX0JZVEVcbiAgfSkge1xuICAgIGFzc2VydCh3aWR0aCA+PSAwICYmIGhlaWdodCA+PSAwLCAnV2lkdGggYW5kIGhlaWdodCBuZWVkIHRvIGJlIGludGVnZXJzJyk7XG4gICAgaWYgKHdpZHRoID09PSB0aGlzLndpZHRoICYmIGhlaWdodCA9PT0gdGhpcy5oZWlnaHQpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBsb2cubG9nKDIsIGBSZXNpemluZyBmcmFtZWJ1ZmZlciAke3RoaXMuaWR9IHRvICR7d2lkdGh9eCR7aGVpZ2h0fWApO1xuXG4gICAgY29uc3Qge2dsfSA9IHRoaXM7XG5cbiAgICAvLyBUT0RPIC0gZG8gd2UgbmVlZCB0byByZWFsbG9jYXRlIHRoZSBmcmFtZWJ1ZmZlcj9cbiAgICBjb25zdCBjb2xvckJ1ZmZlciA9IG5ldyBUZXh0dXJlMkQoZ2wsIHtcbiAgICAgIG1pbkZpbHRlcjogdGhpcy5taW5GaWx0ZXIsXG4gICAgICBtYWdGaWx0ZXI6IHRoaXMubWFnRmlsdGVyXG4gICAgfSlcbiAgICAvLyBUT0RPIC0gc2hvdWxkIGJlIGhhbmRsZWQgYnkgVGV4dHVyZTJEIGNvbnN0cnVjdG9yP1xuICAgIC5zZXRJbWFnZURhdGEoe1xuICAgICAgZGF0YTogbnVsbCxcbiAgICAgIHdpZHRoLFxuICAgICAgaGVpZ2h0LFxuICAgICAgdHlwZSxcbiAgICAgIGZvcm1hdFxuICAgIH0pO1xuXG4gICAgdGhpcy5hdHRhY2hUZXh0dXJlKHtcbiAgICAgIGF0dGFjaG1lbnQ6IEdMLkNPTE9SX0FUVEFDSE1FTlQwLFxuICAgICAgdGV4dHVyZTogY29sb3JCdWZmZXJcbiAgICB9KTtcblxuICAgIGlmICh0aGlzLmNvbG9yQnVmZmVyKSB7XG4gICAgICB0aGlzLmNvbG9yQnVmZmVyLmRlbGV0ZSgpO1xuICAgIH1cbiAgICB0aGlzLmNvbG9yQnVmZmVyID0gY29sb3JCdWZmZXI7XG4gICAgdGhpcy50ZXh0dXJlID0gY29sb3JCdWZmZXI7XG5cbiAgICAvLyBBZGQgYSBkZXB0aCBidWZmZXIgaWYgcmVxdWVzdGVkXG4gICAgaWYgKGRlcHRoKSB7XG4gICAgICBjb25zdCBkZXB0aEJ1ZmZlciA9IG5ldyBSZW5kZXJidWZmZXIoZ2wpLnN0b3JhZ2Uoe1xuICAgICAgICBpbnRlcm5hbEZvcm1hdDogR0wuREVQVEhfQ09NUE9ORU5UMTYsXG4gICAgICAgIHdpZHRoLFxuICAgICAgICBoZWlnaHRcbiAgICAgIH0pO1xuICAgICAgdGhpcy5hdHRhY2hSZW5kZXJidWZmZXIoe1xuICAgICAgICBhdHRhY2htZW50OiBHTC5ERVBUSF9BVFRBQ0hNRU5ULFxuICAgICAgICByZW5kZXJidWZmZXI6IGRlcHRoQnVmZmVyXG4gICAgICB9KTtcblxuICAgICAgaWYgKHRoaXMuZGVwdGhCdWZmZXIpIHtcbiAgICAgICAgdGhpcy5kZXB0aEJ1ZmZlci5kZWxldGUoKTtcbiAgICAgIH1cbiAgICAgIHRoaXMuZGVwdGhCdWZmZXIgPSBkZXB0aEJ1ZmZlcjtcbiAgICB9XG5cbiAgICAvLyBDaGVja3MgdGhhdCBmcmFtZWJ1ZmZlciB3YXMgcHJvcGVybHkgc2V0IHVwLFxuICAgIC8vIGlmIG5vdCwgdGhyb3dzIGFuIGV4cGxhbmF0b3J5IGVycm9yXG4gICAgdGhpcy5jaGVja1N0YXR1cygpO1xuXG4gICAgdGhpcy53aWR0aCA9IHdpZHRoO1xuICAgIHRoaXMuaGVpZ2h0ID0gaGVpZ2h0O1xuICB9XG5cbiAgLy8gV0VCR0wgSU5URVJGQUNFXG5cbiAgYmluZCh7dGFyZ2V0ID0gR0wuRlJBTUVCVUZGRVJ9ID0ge30pIHtcbiAgICBjb25zdCB7Z2x9ID0gdGhpcztcbiAgICBnbC5iaW5kRnJhbWVidWZmZXIodGFyZ2V0LCB0aGlzLmhhbmRsZSk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICB1bmJpbmQoe3RhcmdldCA9IEdMLkZSQU1FQlVGRkVSfSA9IHt9KSB7XG4gICAgY29uc3Qge2dsfSA9IHRoaXM7XG4gICAgZ2wuYmluZEZyYW1lYnVmZmVyKHRhcmdldCwgbnVsbCk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvL1xuICAvLyBOT1RFOiBTbG93IHJlcXVpcmVzIHJvdW5kdHJpcCB0byBHUFVcbiAgLy8gQXBwIGNhbiBwcm92aWRlIHBpeGVsQXJyYXkgb3IgaGF2ZSBpdCBhdXRvIGFsbG9jYXRlZCBieSB0aGlzIG1ldGhvZFxuICAvLyBAcmV0dXJucyB7VWludDhBcnJheXxVaW50MTZBcnJheXxGbG9hdEFycmF5fSAtIHBpeGVsIGFycmF5LFxuICAvLyAgbmV3bHkgYWxsb2NhdGVkIGJ5IHRoaXMgbWV0aG9kIHVubGVzcyBwcm92aWRlZCBieSBhcHAuXG4gIHJlYWRQaXhlbHMoe1xuICAgIHggPSAwLFxuICAgIHkgPSAwLFxuICAgIHdpZHRoLFxuICAgIGhlaWdodCxcbiAgICBmb3JtYXQgPSBHTC5SR0JBLFxuICAgIHR5cGUsXG4gICAgcGl4ZWxBcnJheSA9IG51bGxcbiAgfSkge1xuICAgIGNvbnN0IHtnbH0gPSB0aGlzO1xuXG4gICAgLy8gRGVkdWNlIHR5cGUgYW5kIGFsbG9jYXRlZCBwaXhlbEFycmF5IGlmIG5lZWRlZFxuICAgIGlmICghcGl4ZWxBcnJheSkge1xuICAgICAgLy8gQWxsb2NhdGUgcGl4ZWwgYXJyYXkgaWYgbm90IGFscmVhZHkgYXZhaWxhYmxlLCB1c2luZyBzdXBwbGllZCB0eXBlXG4gICAgICB0eXBlID0gdHlwZSB8fCBHTC5VTlNJR05FRF9CWVRFO1xuICAgICAgY29uc3QgQXJyYXlUeXBlID0gZ2xUeXBlVG9BcnJheSh0eXBlKTtcbiAgICAgIGNvbnN0IGNvbXBvbmVudHMgPSBnbEZvcm1hdFRvQ29tcG9uZW50cyhmb3JtYXQpO1xuICAgICAgLy8gVE9ETyAtIGNoZWNrIGZvciBjb21wb3NpdGUgdHlwZSAoY29tcG9uZW50cyA9IDEpLlxuICAgICAgcGl4ZWxBcnJheSA9IHBpeGVsQXJyYXkgfHwgbmV3IEFycmF5VHlwZSh3aWR0aCAqIGhlaWdodCAqIGNvbXBvbmVudHMpO1xuICAgIH1cblxuICAgIC8vIFBpeGVsIGFycmF5IGF2YWlsYWJsZSwgaWYgbmVjZXNzYXJ5LCBkZWR1Y2UgdHlwZSBmcm9tIGl0LlxuICAgIHR5cGUgPSB0eXBlIHx8IGdsVHlwZUZyb21BcnJheShwaXhlbEFycmF5KTtcblxuICAgIHRoaXMuYmluZCgpO1xuICAgIGdsLnJlYWRQaXhlbHMoeCwgeSwgd2lkdGgsIGhlaWdodCwgZm9ybWF0LCB0eXBlLCBwaXhlbEFycmF5KTtcbiAgICB0aGlzLnVuYmluZCgpO1xuXG4gICAgcmV0dXJuIHBpeGVsQXJyYXk7XG4gIH1cblxuICAvKipcbiAgICogVXNlZCB0byBhdHRhY2ggdGV4dHVyZXMgdG8gYSBmcmFtZWJ1ZmZlciwgdGhlIHRleHR1cmVzIHdpbGwgc3RvcmVcbiAgICogdGhlIHZhcmlvdXMgYnVmZmVycy5cbiAgICpcbiAgICogIFRoZSBzZXQgb2YgYXZhaWxhYmxlIGF0dGFjaG1lbnRzIGlzIGxhcmdlciBpbiBXZWJHTDIsIGFuZCBhbHNvIHRoZVxuICAgKiAgZXh0ZW5zaW9ucyBXRUJHTF9kcmF3X2J1ZmZlcnMgYW5kIFdFQkdMX2RlcHRoX3RleHR1cmUgcHJvdmlkZSBhZGRpdGlvbmFsXG4gICAqICBhdHRhY2htZW50cyB0aGF0IG1hdGNoIG9yIGV4Y2VlZCB0aGUgV2ViR0wyIHNldC5cbiAgICpcbiAgICogQHBhcmFtIHtUZXh0dXJlMkR8VGV4dHVyZUN1YmV8V2ViR0xUZXh0dXJlfG51bGx9IG9wdC50ZXh0dXJlPW51bGwgLVxuICAgKiAgICBkZWZhdWx0IGlzIG51bGwgd2hpY2ggdW5iaW5kcyB0aGUgdGV4dHVyZSBmb3IgdGhlIGF0dGFjaG1lbnRcbiAgICogQHBhcmFtIHtTdHJpbmd8TnVtYmVyfSBvcHQuYXR0YWNobWVudD0gLSB3aGljaCBhdHRhY2htZW50IHRvIGJpbmRcbiAgICogICAgZGVmYXVsdHMgdG8gZ2wuQ09MT1JfQVRUQUNITUVOVDAuXG4gICAqIEBwYXJhbSB7U3RyaW5nfE51bWJlcn0gb3B0LnRhcmdldD0gLSBiaW5kIHBvaW50LCBub3JtYWxseSBnbC5GUkFNRUJVRkZFUlxuICAgKiAgICAoV2ViR0wyIHN1cHBvcnQgc2VwYXJhdGluZyBiZXQpXG4gICAqIEBwYXJhbSB7U3RyaW5nfE51bWJlcn0gb3B0LnRleHR1cmVUYXJnZXQ9IC0gY2FuIGJlIHVzZWQgdG8gc3BlY2lmeVxuICAgKiAgICBmYWNlcyBvZiBhIGN1YmUgbWFwLlxuICAgKiBAcmV0dXJucyB7RnJhbWVCdWZmZXJ9IHJldHVybnMgaXRzZWxmIHRvIGVuYWJsZSBjaGFpbmluZ1xuICAgKi9cbiAgYXR0YWNoVGV4dHVyZSh7XG4gICAgdGV4dHVyZSA9IG51bGwsXG4gICAgdGFyZ2V0ID0gR0wuRlJBTUVCVUZGRVIsXG4gICAgYXR0YWNobWVudCA9IEdMLkNPTE9SX0FUVEFDSE1FTlQwLFxuICAgIHRleHR1cmVUYXJnZXQgPSBHTC5URVhUVVJFXzJELFxuICAgIC8vIG1pcG1hcExldmVsLCBjdXJyZW50bHkgb25seSAwIGlzIHN1cHBvcnRlZCBieSBXZWJHTFxuICAgIG1pcG1hcExldmVsID0gMFxuICB9ID0ge30pIHtcbiAgICBjb25zdCB7Z2x9ID0gdGhpcztcblxuICAgIHRleHR1cmUgPSB0ZXh0dXJlICYmIFRleHR1cmUyRC5tYWtlRnJvbShnbCwgdGV4dHVyZSk7XG5cbiAgICB0aGlzLmJpbmQoe3RhcmdldH0pO1xuXG4gICAgZ2wuZnJhbWVidWZmZXJUZXh0dXJlMkQoXG4gICAgICB0YXJnZXQsXG4gICAgICBnbEdldChhdHRhY2htZW50KSxcbiAgICAgIGdsR2V0KHRleHR1cmVUYXJnZXQpLFxuICAgICAgdGV4dHVyZS5oYW5kbGUsXG4gICAgICBtaXBtYXBMZXZlbFxuICAgICk7XG5cbiAgICB0aGlzLnVuYmluZCgpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVzZWQgdG8gYXR0YWNoIGEgZnJhbWVidWZmZXIgdG8gYSBmcmFtZWJ1ZmZlciwgdGhlIHRleHR1cmVzIHdpbGwgc3RvcmVcbiAgICogdGhlIHZhcmlvdXMgYnVmZmVycy5cbiAgICogQHBhcmFtIHtPYmplY3R9IG9wdHM9IC0gbmFtZWQgcGFyYW1ldGVyc1xuICAgKiBAcGFyYW0ge1JlbmRlckJ1ZmZlcnxXZWJHTFJlbmRlckJ1ZmZlcnxudWxsfSBvcHRzLnJlbmRlcmJ1ZmZlcj1udWxsIC1cbiAgICogICAgcmVuZGVyYnVmZmVyIHRvIGJpbmRcbiAgICogICAgZGVmYXVsdCBpcyBudWxsIHdoaWNoIHVuYmluZHMgdGhlIHJlbmRlcmJ1ZmZlciBmb3IgdGhlIGF0dGFjaG1lbnRcbiAgICogQHBhcmFtIHtTdHJpbmd8TnVtYmVyfSBvcHRzLmF0dGFjaG1lbnQ9IC0gd2hpY2ggYnVmZmVyIHRvIGJpbmRcbiAgICogQHJldHVybnMge0ZyYW1lQnVmZmVyfSByZXR1cm5zIGl0c2VsZiB0byBlbmFibGUgY2hhaW5pbmdcbiAgICovXG4gIGF0dGFjaFJlbmRlcmJ1ZmZlcih7XG4gICAgcmVuZGVyYnVmZmVyID0gbnVsbCxcbiAgICBhdHRhY2htZW50ID0gR0wuQ09MT1JfQVRUQUNITUVOVDAsXG4gICAgdGFyZ2V0ID0gR0wuRlJBTUVCVUZGRVIsXG4gICAgcmVuZGVyYnVmZmVyVGFyZ2V0ID0gR0wuUkVOREVSQlVGRkVSXG4gIH0gPSB7fSkge1xuICAgIGNvbnN0IHtnbH0gPSB0aGlzO1xuICAgIHJlbmRlcmJ1ZmZlciA9IHJlbmRlcmJ1ZmZlciAmJiBSZW5kZXJidWZmZXIubWFrZUZyb20oZ2wsIHJlbmRlcmJ1ZmZlcik7XG5cbiAgICB0aGlzLmJpbmQoe3RhcmdldH0pO1xuXG4gICAgZ2wuZnJhbWVidWZmZXJSZW5kZXJidWZmZXIoXG4gICAgICB0YXJnZXQsXG4gICAgICBnbEdldChhdHRhY2htZW50KSxcbiAgICAgIGdsR2V0KHJlbmRlcmJ1ZmZlclRhcmdldCksXG4gICAgICByZW5kZXJidWZmZXIuaGFuZGxlXG4gICAgKTtcblxuICAgIHRoaXMudW5iaW5kKHt0YXJnZXR9KTtcblxuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgY2hlY2tTdGF0dXMoe3RhcmdldCA9IEdMLkZSQU1FQlVGRkVSfSA9IHt9KSB7XG4gICAgY29uc3Qge2dsfSA9IHRoaXM7XG5cbiAgICB0aGlzLmJpbmQoe3RhcmdldH0pO1xuXG4gICAgY29uc3Qgc3RhdHVzID0gZ2wuY2hlY2tGcmFtZWJ1ZmZlclN0YXR1cyh0YXJnZXQpO1xuXG4gICAgdGhpcy51bmJpbmQoe3RhcmdldH0pO1xuXG4gICAgaWYgKHN0YXR1cyAhPT0gZ2wuRlJBTUVCVUZGRVJfQ09NUExFVEUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcih0aGlzLl9nZXRGcmFtZUJ1ZmZlclN0YXR1cyhzdGF0dXMpKTtcbiAgICB9XG5cbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8vIFdFQkdMMiBJTlRFUkZBQ0VcblxuICBibGl0KHtcbiAgICBzcmNYMCwgc3JjWTAsIHNyY1gxLCBzcmNZMSxcbiAgICBkc3RYMCwgZHN0WTAsIGRzdFgxLCBkc3RZMSxcbiAgICBtYXNrLFxuICAgIGZpbHRlciA9IEdMLk5FQVJFU1RcbiAgfSkge1xuICAgIGNvbnN0IHtnbH0gPSB0aGlzO1xuICAgIGFzc2VydFdlYkdMMkNvbnRleHQoZ2wpO1xuICAgIGdsLmJsaXRGcmFtZWJ1ZmZlcihcbiAgICAgIHNyY1gwLCBzcmNZMCwgc3JjWDEsIHNyY1kxLFxuICAgICAgZHN0WDAsIGRzdFkwLCBkc3RYMSwgZHN0WTEsXG4gICAgICBtYXNrLFxuICAgICAgZmlsdGVyXG4gICAgKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIHRleHR1cmVMYXllcih7XG4gICAgdGFyZ2V0ID0gR0wuRlJBTUVCVUZGRVIsXG4gICAgYXR0YWNobWVudCxcbiAgICB0ZXh0dXJlLFxuICAgIGxldmVsLFxuICAgIGxheWVyXG4gIH0gPSB7fSkge1xuICAgIGNvbnN0IHtnbH0gPSB0aGlzO1xuICAgIGFzc2VydFdlYkdMMkNvbnRleHQoZ2wpO1xuICAgIGdsLmZyYW1lYnVmZmVyVGV4dHVyZUxheWVyKHRhcmdldCwgYXR0YWNobWVudCwgdGV4dHVyZSwgbGV2ZWwsIGxheWVyKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIGludmFsaWRhdGUoe1xuICAgIHRhcmdldCA9IEdMLkZSQU1FQlVGRkVSLFxuICAgIGF0dGFjaG1lbnRzID0gW11cbiAgfSkge1xuICAgIGNvbnN0IHtnbH0gPSB0aGlzO1xuICAgIGFzc2VydFdlYkdMMkNvbnRleHQoZ2wpO1xuICAgIGdsLmludmFsaWRhdGVGcmFtZWJ1ZmZlcih0YXJnZXQsIGF0dGFjaG1lbnRzKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIGludmFsaWRhdGVTdWIoe1xuICAgIHRhcmdldCA9IEdMLkZSQU1FQlVGRkVSLFxuICAgIGF0dGFjaG1lbnRzID0gW10sXG4gICAgeCA9IDAsXG4gICAgeSA9IDAsXG4gICAgd2lkdGgsXG4gICAgaGVpZ2h0XG4gIH0pIHtcbiAgICBjb25zdCB7Z2x9ID0gdGhpcztcbiAgICBhc3NlcnRXZWJHTDJDb250ZXh0KGdsKTtcbiAgICBnbC5pbnZhbGlkYXRlRnJhbWVidWZmZXIodGFyZ2V0LCBhdHRhY2htZW50cywgeCwgeSwgd2lkdGgsIGhlaWdodCk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvLyBTZWxlY3RzIGEgY29sb3IgYnVmZmVyIGFzIHRoZSBzb3VyY2UgZm9yIHBpeGVscyBmb3Igc3Vic2VxdWVudCBjYWxscyB0b1xuICAvLyBjb3B5VGV4SW1hZ2UyRCwgY29weVRleFN1YkltYWdlMkQsIGNvcHlUZXhTdWJJbWFnZTNEIG9yIHJlYWRQaXhlbHMuXG4gIC8vIHNyY1xuICAvLyAgZ2wuQkFDSzogUmVhZHMgZnJvbSB0aGUgYmFjayBjb2xvciBidWZmZXIuXG4gIC8vICBnbC5OT05FOiBSZWFkcyBmcm9tIG5vIGNvbG9yIGJ1ZmZlci5cbiAgLy8gIGdsLkNPTE9SX0FUVEFDSE1FTlR7MC0xNX06IFJlYWRzIGZyb20gb25lIG9mIDE2IGNvbG9yIGF0dGFjaG1lbnQgYnVmZmVycy5cbiAgcmVhZEJ1ZmZlcih7c3JjfSkge1xuICAgIGNvbnN0IHtnbH0gPSB0aGlzO1xuICAgIGFzc2VydFdlYkdMMkNvbnRleHQoZ2wpO1xuICAgIGdsLnJlYWRCdWZmZXIoc3JjKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8vIEByZXR1cm5zIHtHTGludH1cbiAgYWxwaGFTaXplKCkge1xuICAgIHJldHVybiB0aGlzLmdldEF0dGFjaG1lbnRQYXJhbWV0ZXIoXG4gICAgICBHTC5GUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX0FMUEhBX1NJWkUpO1xuICB9XG5cbiAgLy8gQHJldHVybnMge0dMaW50fVxuICBibHVlU2l6ZSgpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRBdHRhY2htZW50UGFyYW1ldGVyKFxuICAgICAgR0wuRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9CTFVFX1NJWkUpO1xuICB9XG5cbiAgLy8gQHJldHVybnMge0dMZW51bX1cbiAgY29sb3JFbmNvZGluZygpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRBdHRhY2htZW50UGFyYW1ldGVyKFxuICAgICAgR0wuRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9DT0xPUl9FTkNPRElORyk7XG4gIH1cblxuICAvLyBAcmV0dXJucyB7R0xlbnVtfVxuICBjb21wb25lbnRUeXBlKCkge1xuICAgIHJldHVybiB0aGlzLmdldEF0dGFjaG1lbnRQYXJhbWV0ZXIoXG4gICAgICBHTC5GUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX0NPTVBPTkVOVF9UWVBFKTtcbiAgfVxuXG4gIC8vIEByZXR1cm5zIHtHTGludH1cbiAgZGVwdGhTaXplKCkge1xuICAgIHJldHVybiB0aGlzLmdldEF0dGFjaG1lbnRQYXJhbWV0ZXIoXG4gICAgICBHTC5GUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX0RFUFRIX1NJWkUpO1xuICB9XG5cbiAgLy8gQHJldHVybnMge0dMaW50fVxuICBncmVlblNpemUoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0QXR0YWNobWVudFBhcmFtZXRlcihcbiAgICAgIEdMLkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfR1JFRU5fU0laRSk7XG4gIH1cblxuICAvLyBAcmV0dXJucyB7V2ViR0xSZW5kZXJidWZmZXJ8V2ViR0xUZXh0dXJlfVxuICBvYmplY3ROYW1lKCkge1xuICAgIHJldHVybiB0aGlzLmdldEF0dGFjaG1lbnRQYXJhbWV0ZXIoXG4gICAgICBHTC5GUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX09CSkVDVF9OQU1FKTtcbiAgfVxuXG4gIC8vIEByZXR1cm5zIHtHTGVudW19XG4gIG9iamVjdFR5cGUoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0QXR0YWNobWVudFBhcmFtZXRlcihcbiAgICAgIEdMLkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfT0JKRUNUX1RZUEUpO1xuICB9XG5cbiAgLy8gQHJldHVybnMge0dMaW50fVxuICByZWRTaXplKCkge1xuICAgIHJldHVybiB0aGlzLmdldEF0dGFjaG1lbnRQYXJhbWV0ZXIoXG4gICAgICBHTC5GUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX1JFRF9TSVpFKTtcbiAgfVxuXG4gIC8vIEByZXR1cm5zIHtHTGludH1cbiAgc3RlbmNpbFNpemUoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0QXR0YWNobWVudFBhcmFtZXRlcihcbiAgICAgIEdMLkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfU1RFTkNJTF9TSVpFKTtcbiAgfVxuXG4gIC8vIEByZXR1cm5zIHtHTGludH1cbiAgY3ViZU1hcEZhY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0QXR0YWNobWVudFBhcmFtZXRlcihcbiAgICAgIEdMLkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfVEVYVFVSRV9DVUJFX01BUF9GQUNFKTtcbiAgfVxuXG4gIC8vIEByZXR1cm5zIHtHTGludH1cbiAgbGF5ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0QXR0YWNobWVudFBhcmFtZXRlcihcbiAgICAgIEdMLkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfVEVYVFVSRV9MQVlFUik7XG4gIH1cblxuICAvLyBAcmV0dXJucyB7R0xpbnR9XG4gIGxldmVsKCkge1xuICAgIHJldHVybiB0aGlzLmdldEF0dGFjaG1lbnRQYXJhbWV0ZXIoXG4gICAgICBHTC5GUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX1RFWFRVUkVfTEVWRUwpO1xuICB9XG5cbiAgZ2V0UGFyYW1ldGVycygpIHtcbiAgICByZXR1cm4ge1xuICAgICAgcmVkU2l6ZTogdGhpcy5nZXRBdHRhY2htZW50UGFyYW1ldGVyKFxuICAgICAgICBHTC5GUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX1JFRF9TSVpFKSxcbiAgICAgIGdyZWVuU2l6ZTogdGhpcy5nZXRBdHRhY2htZW50UGFyYW1ldGVyKFxuICAgICAgICBHTC5GUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX0dSRUVOX1NJWkUpLFxuICAgICAgYmx1ZVNpemU6IHRoaXMuZ2V0QXR0YWNobWVudFBhcmFtZXRlcihcbiAgICAgICAgR0wuRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9CTFVFX1NJWkUpLFxuICAgICAgYWxwaGFTaXplOiB0aGlzLmdldEF0dGFjaG1lbnRQYXJhbWV0ZXIoXG4gICAgICAgIEdMLkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfQUxQSEFfU0laRSksXG5cbiAgICAgIGRlcHRoU2l6ZTogdGhpcy5nZXRBdHRhY2htZW50UGFyYW1ldGVyKFxuICAgICAgICBHTC5GUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX0RFUFRIX1NJWkUpLFxuICAgICAgc3RlbmNpbFNpemU6IHRoaXMuZ2V0QXR0YWNobWVudFBhcmFtZXRlcihcbiAgICAgICAgR0wuRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9TVEVOQ0lMX1NJWkUpLFxuXG4gICAgICBjb2xvckVuY29kaW5nOiB0aGlzLmdldEF0dGFjaG1lbnRQYXJhbWV0ZXIoXG4gICAgICAgIEdMLkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfQ09MT1JfRU5DT0RJTkcpLFxuICAgICAgY29tcG9uZW50VHlwZTogdGhpcy5nZXRBdHRhY2htZW50UGFyYW1ldGVyKFxuICAgICAgICBHTC5GUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX0NPTVBPTkVOVF9UWVBFKSxcbiAgICAgIG9iamVjdE5hbWU6IHRoaXMuZ2V0QXR0YWNobWVudFBhcmFtZXRlcihcbiAgICAgICAgR0wuRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9PQkpFQ1RfTkFNRSksXG4gICAgICBvYmplY3RUeXBlOiB0aGlzLmdldEF0dGFjaG1lbnRQYXJhbWV0ZXIoXG4gICAgICAgIEdMLkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfT0JKRUNUX1RZUEUpLFxuICAgICAgY3ViZU1hcEZhY2U6IHRoaXMuZ2V0QXR0YWNobWVudFBhcmFtZXRlcihcbiAgICAgICAgR0wuRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9URVhUVVJFX0NVQkVfTUFQX0ZBQ0UpLFxuICAgICAgbGF5ZXI6IHRoaXMuZ2V0QXR0YWNobWVudFBhcmFtZXRlcihcbiAgICAgICAgR0wuRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9URVhUVVJFX0xBWUVSKSxcbiAgICAgIGxldmVsOiB0aGlzLmdldEF0dGFjaG1lbnRQYXJhbWV0ZXIoXG4gICAgICAgIEdMLkZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfVEVYVFVSRV9MRVZFTClcbiAgICB9O1xuICB9XG5cbiAgLy8gKE9wZW5HTCBFUyAzLjAuNCDCpzYuMS4xMywgc2ltaWxhciB0byBnbEdldEZyYW1lYnVmZmVyQXR0YWNobWVudFBhcmFtZXRlcml2KVxuICAvLyBSZXR1cm4gdGhlIHZhbHVlIGZvciB0aGUgcGFzc2VkIHBuYW1lIGdpdmVuIHRoZSB0YXJnZXQgYW5kIGF0dGFjaG1lbnQuXG4gIC8vIFRoZSB0eXBlIHJldHVybmVkIGlzIHRoZSBuYXR1cmFsIHR5cGUgZm9yIHRoZSByZXF1ZXN0ZWQgcG5hbWU6XG4gIC8vIHBuYW1lIHJldHVybmVkIHR5cGVcbiAgLy8gRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9BTFBIQV9TSVpFIEdMaW50XG4gIC8vIEZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfQkxVRV9TSVpFICBHTGludFxuICAvLyBGUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX0NPTE9SX0VOQ09ESU5HIEdMZW51bVxuICAvLyBGUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX0NPTVBPTkVOVF9UWVBFIEdMZW51bVxuICAvLyBGUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX0RFUFRIX1NJWkUgR0xpbnRcbiAgLy8gRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9HUkVFTl9TSVpFIEdMaW50XG4gIC8vIEZSQU1FQlVGRkVSX0FUVEFDSE1FTlRfT0JKRUNUX05BTUUgIFdlYkdMUmVuZGVyYnVmZmVyIG9yIFdlYkdMVGV4dHVyZVxuICAvLyBGUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX09CSkVDVF9UWVBFICBHTGVudW1cbiAgLy8gRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9SRURfU0laRSBHTGludFxuICAvLyBGUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX1NURU5DSUxfU0laRSBHTGludFxuICAvLyBGUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX1RFWFRVUkVfQ1VCRV9NQVBfRkFDRSAgR0xpbnRcbiAgLy8gRlJBTUVCVUZGRVJfQVRUQUNITUVOVF9URVhUVVJFX0xBWUVSICBHTGludFxuICAvLyBGUkFNRUJVRkZFUl9BVFRBQ0hNRU5UX1RFWFRVUkVfTEVWRUwgIEdMaW50XG4gIC8vIElmIHBuYW1lIGlzIG5vdCBpbiB0aGUgdGFibGUgYWJvdmUsIGdlbmVyYXRlcyBhbiBJTlZBTElEX0VOVU0gZXJyb3IuXG4gIC8vIElmIGFuIE9wZW5HTCBlcnJvciBpcyBnZW5lcmF0ZWQsIHJldHVybnMgbnVsbC5cbiAgZ2V0QXR0YWNobWVudFBhcmFtZXRlcih7XG4gICAgcG5hbWUsXG4gICAgdGFyZ2V0LFxuICAgIGF0dGFjaG1lbnRcbiAgfSA9IHt9KSB7XG4gICAgY29uc3Qge2dsfSA9IHRoaXM7XG4gICAgYXNzZXJ0V2ViR0wyQ29udGV4dChnbCk7XG4gICAgY29uc3QgdmFsdWUgPSBnbC5nZXRGcmFtZWJ1ZmZlckF0dGFjaG1lbnRQYXJhbWV0ZXIoXG4gICAgICB0YXJnZXQsIGF0dGFjaG1lbnQsIHBuYW1lXG4gICAgKTtcbiAgICByZXR1cm4gdmFsdWU7XG4gIH1cblxuICAvKiBlc2xpbnQtZGlzYWJsZSBtYXgtbGVuICovXG4gIF9nZXRGcmFtZUJ1ZmZlclN0YXR1cyhzdGF0dXMpIHtcbiAgICBsZXQgZXJyb3I7XG4gICAgc3dpdGNoIChzdGF0dXMpIHtcbiAgICBjYXNlIEdMLkZSQU1FQlVGRkVSX0NPTVBMRVRFOlxuICAgICAgZXJyb3IgPSAnU3VjY2Vzcy4gRnJhbWVidWZmZXIgaXMgY29ycmVjdGx5IHNldCB1cCc7XG4gICAgICBicmVhaztcbiAgICBjYXNlIEdMLkZSQU1FQlVGRkVSX0lOQ09NUExFVEVfQVRUQUNITUVOVDpcbiAgICAgIGVycm9yID0gJ1RoZSBhdHRhY2htZW50IHR5cGVzIGFyZSBtaXNtYXRjaGVkIG9yIG5vdCBhbGwgZnJhbWVidWZmZXIgYXR0YWNobWVudCBwb2ludHMgYXJlIGZyYW1lYnVmZmVyIGF0dGFjaG1lbnQgY29tcGxldGUuJztcbiAgICAgIGJyZWFrO1xuICAgIGNhc2UgR0wuRlJBTUVCVUZGRVJfSU5DT01QTEVURV9NSVNTSU5HX0FUVEFDSE1FTlQ6XG4gICAgICBlcnJvciA9ICdUaGVyZSBpcyBubyBhdHRhY2htZW50Lic7XG4gICAgICBicmVhaztcbiAgICBjYXNlIEdMLkZSQU1FQlVGRkVSX0lOQ09NUExFVEVfRElNRU5TSU9OUzpcbiAgICAgIGVycm9yID0gJ0hlaWdodCBhbmQgd2lkdGggb2YgdGhlIGF0dGFjaG1lbnQgYXJlIG5vdCB0aGUgc2FtZS4nO1xuICAgICAgYnJlYWs7XG4gICAgY2FzZSBHTC5GUkFNRUJVRkZFUl9VTlNVUFBPUlRFRDpcbiAgICAgIGVycm9yID0gJ1RoZSBmb3JtYXQgb2YgdGhlIGF0dGFjaG1lbnQgaXMgbm90IHN1cHBvcnRlZCBvciBpZiBkZXB0aCBhbmQgc3RlbmNpbCBhdHRhY2htZW50cyBhcmUgbm90IHRoZSBzYW1lIHJlbmRlcmJ1ZmZlci4nO1xuICAgICAgYnJlYWs7XG4gICAgLy8gV2hlbiB1c2luZyBhIFdlYkdMIDIgY29udGV4dCwgdGhlIGZvbGxvd2luZyB2YWx1ZXMgY2FuIGJlIHJldHVybmVkXG4gICAgY2FzZSBHTC5GUkFNRUJVRkZFUl9JTkNPTVBMRVRFX01VTFRJU0FNUExFOlxuICAgICAgZXJyb3IgPSAnVGhlIHZhbHVlcyBvZiBHTC5SRU5ERVJCVUZGRVJfU0FNUExFUyBhcmUgZGlmZmVyZW50IGFtb25nIGF0dGFjaGVkIHJlbmRlcmJ1ZmZlcnMsIG9yIGFyZSBub24temVybyBpZiB0aGUgYXR0YWNoZWQgaW1hZ2VzIGFyZSBhIG1peCBvZiByZW5kZXJidWZmZXJzIGFuZCB0ZXh0dXJlcy4nO1xuICAgICAgYnJlYWs7XG4gICAgZGVmYXVsdDpcbiAgICAgIGVycm9yID0gYEZyYW1lYnVmZmVyIGVycm9yICR7c3RhdHVzfWA7XG4gICAgICBicmVhaztcbiAgICB9XG4gICAgcmV0dXJuIGVycm9yO1xuICB9XG4gIC8qIGVzbGludC1lbmFibGUgbWF4LWxlbiAqL1xufVxuIl19