'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var ERR_DELETED = 'Query was deleted before result was available';
var ERR_CANCEL = 'Query was canceled before result was available';

var noop = function noop(x) {
  return x;
};

var QueryManager = function () {

  /**
   * Internal class that helps "asynchronous WebGL query objects" manage
   * pending requests (e.g. for EXT_disjoint_timer_query and WebGL2 queries)
   *
   * Creates and manages promises for the queries.
   * Tracks pending queries enabling polling.
   * Tracks pending queries enabling invalidation.
   * Encapsulates some standard error messages.
   *
   * Remarks:
   * - Maintains a minimal list of pending queries only to minimize GC impact
   * - Exported as a singleton class instance.
   */
  function QueryManager() {
    _classCallCheck(this, QueryManager);

    this.pendingQueries = new Set();
    this.invalidQueryType = null;
    this.invalidErrorMessage = '';
    this.checkInvalid = function () {
      return false;
    };
  }

  // API THAT SHOULD BE EXPOSED TO APPLICATION

  // Checks invalidation callback and then all pending queries for completion
  // Should only be called once per tick


  _createClass(QueryManager, [{
    key: 'poll',
    value: function poll(gl) {
      this.cancelInvalidQueries(gl);

      // Now check availability of results and resolve promises as appropriate
      var _iteratorNormalCompletion = true;
      var _didIteratorError = false;
      var _iteratorError = undefined;

      try {
        for (var _iterator = this.pendingQueries.values()[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
          var query = _step.value;

          var resultAvailable = query.isResultAvailable();
          if (resultAvailable) {
            var result = query.getResult();
            this.resolveQuery(query, result);
          }
        }
      } catch (err) {
        _didIteratorError = true;
        _iteratorError = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion && _iterator.return) {
            _iterator.return();
          }
        } finally {
          if (_didIteratorError) {
            throw _iteratorError;
          }
        }
      }
    }

    // API FOR MANAGED QUERY CLASSES

    // Registers query invalidation method - used to detect disjoint timer queries

  }, {
    key: 'setInvalidator',
    value: function setInvalidator(_ref) {
      var queryType = _ref.queryType,
          errorMessage = _ref.errorMessage,
          checkInvalid = _ref.checkInvalid;

      this.invalidQueryType = queryType;
      this.invalidErrorMessage = errorMessage;
      this.checkInvalid = checkInvalid;
    }

    // Starts a query, sets up a new promise

  }, {
    key: 'beginQuery',
    value: function beginQuery(query) {
      var onComplete = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : noop;
      var onError = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : noop;

      // Make sure disjoint state is cleared, so that this query starts fresh
      // Cancel other queries if needed
      this.cancelInvalidQueries(query.gl);

      // Cancel current promise - noop if already resolved or rejected
      this.cancelQuery(query);

      // Create a new promise with attached resolve and reject methods
      var resolvers = {};
      query.promise = new Promise(function (resolve, reject) {
        resolvers.resolve = resolve;
        resolvers.reject = reject;
      });
      Object.assign(query.promise, resolvers);

      // Add this query to the pending queries
      this.pendingQueries.add(query);
      // Register the callbacks
      return query.promise.then(onComplete).catch(onError);
    }

    // Resolves a query with a result

  }, {
    key: 'resolveQuery',
    value: function resolveQuery(query, result) {
      this.pendingQueries.delete(query);
      query.promise.resolve(result);
    }

    // Rejects the promise

  }, {
    key: 'rejectQuery',
    value: function rejectQuery(query, errorMessage) {
      this.pendingQueries.delete(query);
      if (query.promise) {
        query.promise.reject(new Error(errorMessage));
      }
    }

    // Rejects promise with standard message for Query.delete()

  }, {
    key: 'deleteQuery',
    value: function deleteQuery(query) {
      return this.rejectQuery(query, ERR_DELETED);
    }

    // Rejects promise with standard message for Query.cancel()

  }, {
    key: 'cancelQuery',
    value: function cancelQuery(query) {
      return this.rejectQuery(query, ERR_CANCEL);
    }

    // Rejects promise with registered message for invalidation

  }, {
    key: 'invalidateQuery',
    value: function invalidateQuery(query) {
      if (query instanceof this.invalidQueryType) {
        this.rejectQuery(query, this.invalidErrorMessage);
      }
    }

    // Checks all queries to see if need to be invalidated

  }, {
    key: 'cancelInvalidQueries',
    value: function cancelInvalidQueries(gl) {
      // We assume that we can cancel queries for all context.
      // Should be OK since this is used to check for "disjoint" GPU state
      if (this.checkInvalid(gl)) {
        var _iteratorNormalCompletion2 = true;
        var _didIteratorError2 = false;
        var _iteratorError2 = undefined;

        try {
          for (var _iterator2 = this.pendingQueries.values()[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
            var query = _step2.value;

            this.invalidateQuery(query);
          }
        } catch (err) {
          _didIteratorError2 = true;
          _iteratorError2 = err;
        } finally {
          try {
            if (!_iteratorNormalCompletion2 && _iterator2.return) {
              _iterator2.return();
            }
          } finally {
            if (_didIteratorError2) {
              throw _iteratorError2;
            }
          }
        }
      }
    }
  }]);

  return QueryManager;
}();

exports.default = new QueryManager();
//# sourceMappingURL=data:application/json;base64,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