'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

exports.getUniformDescriptors = getUniformDescriptors;

var _webgl = require('./webgl');

var _webglChecks = require('./webgl-checks');

var _vertexAttributes = require('./vertex-attributes');

var VertexAttributes = _interopRequireWildcard(_vertexAttributes);

var _buffer2 = require('./buffer');

var _buffer3 = _interopRequireDefault(_buffer2);

var _texture = require('./texture');

var _texture2 = _interopRequireDefault(_texture);

var _uniforms = require('./uniforms');

var _shader = require('./shader');

var _utils = require('../utils');

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) newObj[key] = obj[key]; } } newObj.default = obj; return newObj; } }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var Program = function () {
  _createClass(Program, null, [{
    key: 'makeFrom',


    /**
     * Returns a Program wrapped WebGLProgram from a variety of inputs.
     * Allows other functions to transparently accept raw WebGLPrograms etc
     * and manipulate them using the methods in the `Program` class.
     * Checks for ".handle"
     *
     * @param {WebGLRenderingContext} gl - if a new buffer needs to be initialized
     * @param {*} object - candidate that will be coerced to a buffer
     * @returns {Program} - Program object that wraps the buffer parameter
     */
    value: function makeFrom(gl) {
      var object = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      return object instanceof Program ? object :
      // Use .handle if available, else use 'program' directly
      new Program(gl).setData({ handle: object.handle || object });
    }

    /*
     * @classdesc
     * Handles creation of programs, mapping of attributes and uniforms
     *
     * @class
     * @param {WebGLRenderingContext} gl - gl context
     * @param {Object} opts - options
     * @param {String} opts.vs - Vertex shader source
     * @param {String} opts.fs - Fragment shader source
     * @param {String} opts.id= - Id
     */
    /* eslint-disable max-statements */

  }]);

  function Program(gl) {
    var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
        id = _ref.id,
        vs = _ref.vs,
        fs = _ref.fs,
        defaultUniforms = _ref.defaultUniforms,
        handle = _ref.handle;

    _classCallCheck(this, Program);

    (0, _webglChecks.assertWebGLContext)(gl);

    // Create shaders if needed
    this.vs = typeof vs === 'string' ? new _shader.VertexShader(gl, vs) : vs;
    this.fs = typeof vs === 'string' ? new _shader.FragmentShader(gl, fs) : fs;

    (0, _assert2.default)(this.vs instanceof _shader.VertexShader, 'Program: bad vertex shader');
    (0, _assert2.default)(this.fs instanceof _shader.FragmentShader, 'Program: bad fragment shader');

    // If program is not named, name it after shader names
    var programName = this.vs.getName() || this.fs.getName();
    programName = programName ? programName + '-program' : 'program';
    this.id = id || (0, _utils.uid)(programName);

    this.gl = gl;
    this.defaultUniforms = defaultUniforms;
    this.handle = handle;
    if (!this.handle) {
      this.handle = gl.createProgram();
      this._compileAndLink();
    }
    if (!this.handle) {
      throw new Error('Failed to create program');
    }

    // determine attribute locations (i.e. indices)
    this._attributeLocations = this._getAttributeLocations();
    this._attributeCount = this.getAttributeCount();
    this._warn = [];
    this._filledLocations = {};

    // prepare uniform setters
    this._uniformSetters = this._getUniformSetters();
    this._uniformCount = this.getUniformCount();
    this._textureIndexCounter = 0;

    this.userData = {};
    Object.seal(this);
  }
  /* eslint-enable max-statements */

  _createClass(Program, [{
    key: 'delete',
    value: function _delete() {
      var gl = this.gl;

      if (this.handle) {
        gl.deleteProgram(this.handle);
      }
      this.handle = null;
      return this;
    }
  }, {
    key: 'use',
    value: function use() {
      var gl = this.gl;

      gl.useProgram(this.handle);
      return this;
    }

    // DEPRECATED METHODS

  }, {
    key: 'clearBuffers',
    value: function clearBuffers() {
      this._filledLocations = {};
      return this;
    }
  }, {
    key: '_print',
    value: function _print(bufferName) {
      return 'Program ' + this.id + ': Attribute ' + bufferName;
    }

    /**
     * Attach a map of Buffers values to a program
     * Only attributes with names actually present in the linked program
     * will be updated. Other supplied buffers will be ignored.
     *
     * @param {Object} buffers - An object map with attribute names being keys
     *  and values are expected to be instances of Buffer.
     * @returns {Program} Returns itself for chaining.
     */
    /* eslint-disable max-statements */

  }, {
    key: 'setBuffers',
    value: function setBuffers(buffers) {
      var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
          _ref2$clear = _ref2.clear,
          clear = _ref2$clear === undefined ? true : _ref2$clear,
          _ref2$check = _ref2.check,
          check = _ref2$check === undefined ? true : _ref2$check,
          _ref2$drawParams = _ref2.drawParams,
          drawParams = _ref2$drawParams === undefined ? {} : _ref2$drawParams;

      var gl = this.gl;

      if (Array.isArray(buffers)) {
        throw new Error('Program.setBuffers expects map of buffers');
      }

      if (clear) {
        this.clearBuffers();
      }

      // indexing is autodetected - buffer with target gl.ELEMENT_ARRAY_BUFFER
      // index type is saved for drawElement calls
      drawParams.isInstanced = false;
      drawParams.isIndexed = false;
      drawParams.indexType = null;

      var _sortBuffersByLocatio = this._sortBuffersByLocation(buffers),
          locations = _sortBuffersByLocatio.locations,
          elements = _sortBuffersByLocatio.elements;

      // Process locations in order


      for (var location = 0; location < locations.length; ++location) {
        var bufferName = locations[location];
        var buffer = buffers[bufferName];
        // DISABLE MISSING ATTRIBUTE
        if (!buffer) {
          VertexAttributes.disable(gl, location);
        } else {
          var divisor = buffer.layout.instanced ? 1 : 0;
          VertexAttributes.enable(gl, location);
          VertexAttributes.setBuffer({ gl: gl, location: location, buffer: buffer });
          VertexAttributes.setDivisor(gl, location, divisor);
          drawParams.isInstanced = buffer.layout.instanced > 0;
          this._filledLocations[bufferName] = true;
        }
      }

      // SET ELEMENTS ARRAY BUFFER
      if (elements) {
        var _buffer = buffers[elements];
        _buffer.bind();
        drawParams.isIndexed = true;
        drawParams.indexType = _buffer.layout.type;
      }

      if (check) {
        this.checkBuffers();
      }

      return this;
    }
    /* eslint-enable max-statements */

  }, {
    key: 'checkBuffers',
    value: function checkBuffers() {
      for (var attributeName in this._attributeLocations) {
        if (!this._filledLocations[attributeName] && !this._warn[attributeName]) {
          var location = this._attributeLocations[attributeName];
          // throw new Error(`Program ${this.id}: ` +
          //   `Attribute ${location}:${attributeName} not supplied`);
          _utils.log.warn(0, 'Program ' + this.id + ': ' + ('Attribute ' + location + ':' + attributeName + ' not supplied'));
          this._warn[attributeName] = true;
        }
      }
      return this;
    }

    /*
     * @returns {Program} Returns itself for chaining.
     */

  }, {
    key: 'unsetBuffers',
    value: function unsetBuffers() {
      var gl = this.gl;

      var length = this._attributeCount;
      for (var i = 1; i < length; ++i) {
        // VertexAttributes.setDivisor(gl, i, 0);
        VertexAttributes.disable(gl, i);
      }
      gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, null);
      return this;
    }

    /**
     * Apply a set of uniform values to a program
     * Only uniforms with names actually present in the linked program
     * will be updated.
     * other uniforms will be ignored
     *
     * @param {Object} uniformMap - An object with names being keys
     * @returns {Program} - returns itself for chaining.
     */
    /* eslint-disable max-depth */

  }, {
    key: 'setUniforms',
    value: function setUniforms(uniforms) {
      for (var uniformName in uniforms) {
        var uniform = uniforms[uniformName];
        var uniformSetter = this._uniformSetters[uniformName];
        if (uniformSetter) {
          if (uniform instanceof _texture2.default) {
            if (uniformSetter.textureIndex === undefined) {
              uniformSetter.textureIndex = this._textureIndexCounter++;
            }
            // Bind texture to index, and set the uniform sampler to the index
            var texture = uniform;
            var textureIndex = uniformSetter.textureIndex;
            // console.debug('setting texture', textureIndex, texture);

            texture.bind(textureIndex);
            uniformSetter(textureIndex);
          } else {
            // Just set the value
            uniformSetter(uniform);
          }
        }
      }
      return this;
    }
    /* eslint-enable max-depth */

    // RAW WEBGL METHODS

  }, {
    key: 'getAttachedShadersCount',
    value: function getAttachedShadersCount() {
      return this.getProgramParameter(_webgl.GL.ATTACHED_SHADERS);
    }

    // ATTRIBUTES API
    // Note: Locations are numeric indices

  }, {
    key: 'getAttributeCount',
    value: function getAttributeCount() {
      return this.getProgramParameter(_webgl.GL.ACTIVE_ATTRIBUTES);
    }

    /**
     * Returns an object with info about attribute at index "location"/
     * @param {int} location - index of an attribute
     * @returns {WebGLActiveInfo} - info about an active attribute
     *   fields: {name, size, type}
     */

  }, {
    key: 'getAttributeInfo',
    value: function getAttributeInfo(location) {
      return this.gl.getActiveAttrib(this.handle, location);
    }
  }, {
    key: 'getAttributeName',
    value: function getAttributeName(location) {
      return this.getAttributeInfo(location).name;
    }

    /**
     * Returns location (index) of a name
     * @param {String} attributeName - name of an attribute
     *   (matches name in a linked shader)
     * @returns {String[]} - array of actual attribute names from shader linking
     */

  }, {
    key: 'getAttributeLocation',
    value: function getAttributeLocation(attributeName) {
      return this.gl.getAttribLocation(this.handle, attributeName);
    }

    // UNIFORMS API
    // Note: locations are opaque structures

  }, {
    key: 'getUniformCount',
    value: function getUniformCount() {
      return this.getProgramParameter(_webgl.GL.ACTIVE_UNIFORMS);
    }

    /*
     * @returns {WebGLActiveInfo} - object with {name, size, type}
     */

  }, {
    key: 'getUniformInfo',
    value: function getUniformInfo(index) {
      return this.gl.getActiveUniform(this.handle, index);
    }

    /*
     * @returns {WebGLUniformLocation} - opaque object representing location
     * of uniform, used by setter methods
     */

  }, {
    key: 'getUniformLocation',
    value: function getUniformLocation(name) {
      return this.gl.getUniformLocation(this.handle, name);
    }
  }, {
    key: 'getUniformValue',
    value: function getUniformValue(location) {
      return this.gl.getUniform(this.handle, location);
    }

    // PROGRAM API

  }, {
    key: 'isFlaggedForDeletion',
    value: function isFlaggedForDeletion() {
      return this.getProgramParameter(_webgl.GL.DELETE_STATUS);
    }
  }, {
    key: 'getLastLinkStatus',
    value: function getLastLinkStatus() {
      return this.getProgramParameter(_webgl.GL.LINK_STATUS);
    }
  }, {
    key: 'getLastValidationStatus',
    value: function getLastValidationStatus() {
      return this.getProgramParameter(_webgl.GL.VALIDATE_STATUS);
    }

    // WEBGL2 INTERFACE

    // This may be gl.SEPARATE_ATTRIBS or gl.INTERLEAVED_ATTRIBS.

  }, {
    key: 'getTransformFeedbackBufferMode',
    value: function getTransformFeedbackBufferMode() {
      (0, _webglChecks.assertWebGL2Context)(this.gl);
      return this.getProgramParameter(this.gl.TRANSFORM_FEEDBACK_BUFFER_MODE);
    }
  }, {
    key: 'getTransformFeedbackVaryingsCount',
    value: function getTransformFeedbackVaryingsCount() {
      (0, _webglChecks.assertWebGL2Context)(this.gl);
      return this.getProgramParameter(this.gl.TRANSFORM_FEEDBACK_VARYINGS);
    }
  }, {
    key: 'getActiveUniformBlocksCount',
    value: function getActiveUniformBlocksCount() {
      (0, _webglChecks.assertWebGL2Context)(this.gl);
      return this.getProgramParameter(this.gl.ACTIVE_UNIFORM_BLOCKS);
    }

    // Retrieves the assigned color number binding for the user-defined varying
    // out variable name for program. program must have previously been linked.
    // [WebGLHandlesContextLoss]

  }, {
    key: 'getFragDataLocation',
    value: function getFragDataLocation(varyingName) {
      (0, _webglChecks.assertWebGL2Context)(this.gl);
      return this.gl.getFragDataLocation(this.handle, varyingName);
    }

    // Return the value for the passed pname given the passed program.
    // The type returned is the natural type for the requested pname,
    // as given in the following table:
    // pname returned type
    // DELETE_STATUS GLboolean
    // LINK_STATUS GLboolean
    // VALIDATE_STATUS GLboolean
    // ATTACHED_SHADERS  GLint
    // ACTIVE_ATTRIBUTES GLint
    // ACTIVE_UNIFORMS GLint
    // TRANSFORM_FEEDBACK_BUFFER_MODE  GLenum
    // TRANSFORM_FEEDBACK_VARYINGS GLint
    // ACTIVE_UNIFORM_BLOCKS GLint

  }, {
    key: 'getProgramParameter',
    value: function getProgramParameter(pname) {
      return this.gl.getProgramParameter(this.handle, pname);
    }

    // PRIVATE METHODS

  }, {
    key: '_compileAndLink',
    value: function _compileAndLink() {
      var gl = this.gl;

      gl.attachShader(this.handle, this.vs.handle);
      gl.attachShader(this.handle, this.fs.handle);
      gl.linkProgram(this.handle);
      // Program linking error is checked only when debug context is used
      if (gl.debug) {
        gl.validateProgram(this.handle);
        var linked = gl.getProgramParameter(this.handle, gl.LINK_STATUS);
        if (!linked) {
          throw new Error('Error linking ' + gl.getProgramInfoLog(this.handle));
        }
      }
    }
  }, {
    key: '_sortBuffersByLocation',
    value: function _sortBuffersByLocation(buffers) {
      var elements = null;
      var locations = new Array(this._attributeCount);

      for (var bufferName in buffers) {
        var buffer = _buffer3.default.makeFrom(this.gl, buffers[bufferName]);
        var location = this._attributeLocations[bufferName];
        if (location === undefined) {
          if (buffer.target === _webgl.GL.ELEMENT_ARRAY_BUFFER && elements) {
            throw new Error(this._print(bufferName) + ' duplicate gl.ELEMENT_ARRAY_BUFFER');
          } else if (buffer.target === _webgl.GL.ELEMENT_ARRAY_BUFFER) {
            elements = bufferName;
          } else if (!this._warn[bufferName]) {
            _utils.log.warn(2, this._print(bufferName) + ' not used');
            this._warn[bufferName] = true;
          }
        } else {
          if (buffer.target === _webgl.GL.ELEMENT_ARRAY_BUFFER) {
            throw new Error(this._print(bufferName) + ':' + location + ' ' + 'has both location and type gl.ELEMENT_ARRAY_BUFFER');
          }
          locations[location] = bufferName;
        }
      }

      return { locations: locations, elements: elements };
    }

    // Check that all active attributes are enabled

  }, {
    key: '_areAllAttributesEnabled',
    value: function _areAllAttributesEnabled() {
      var gl = this.gl;

      var length = this._attributeCount;
      for (var i = 0; i < length; ++i) {
        if (!VertexAttributes.isEnabled(gl, i)) {
          return false;
        }
      }
      return true;
    }

    // determine attribute locations (maps attribute name to index)

  }, {
    key: '_getAttributeLocations',
    value: function _getAttributeLocations() {
      var attributeLocations = {};
      var length = this.getAttributeCount();
      for (var location = 0; location < length; location++) {
        var name = this.getAttributeName(location);
        attributeLocations[name] = this.getAttributeLocation(name);
      }
      return attributeLocations;
    }

    // create uniform setters
    // Map of uniform names to setter functions

  }, {
    key: '_getUniformSetters',
    value: function _getUniformSetters() {
      var gl = this.gl;

      var uniformSetters = {};
      var length = this.getUniformCount();
      for (var i = 0; i < length; i++) {
        var info = this.getUniformInfo(i);
        var parsedName = (0, _uniforms.parseUniformName)(info.name);
        var location = this.getUniformLocation(parsedName.name);
        uniformSetters[parsedName.name] = (0, _uniforms.getUniformSetter)(gl, location, info, parsedName.isArray);
      }
      return uniformSetters;
    }
  }]);

  return Program;
}();

// create uniform setters
// Map of uniform names to setter functions


exports.default = Program;
function getUniformDescriptors(gl, program) {
  var uniformDescriptors = {};
  var length = program.getUniformCount();
  for (var i = 0; i < length; i++) {
    var info = program.getUniformInfo(i);
    var location = program.getUniformLocation(info.name);
    var descriptor = (0, _uniforms.getUniformSetter)(gl, location, info);
    uniformDescriptors[descriptor.name] = descriptor;
  }
  return uniformDescriptors;
}
//# sourceMappingURL=data:application/json;base64,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