'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

var _webgl = require('./webgl');

var _webglChecks = require('./webgl-checks');

var _buffer = require('./buffer');

var _buffer2 = _interopRequireDefault(_buffer);

var _utils = require('../utils');

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var Texture = function () {
  function Texture(gl, opts) {
    _classCallCheck(this, Texture);

    var _opts$id = opts.id,
        id = _opts$id === undefined ? (0, _utils.uid)('texture') : _opts$id,
        _opts$unpackFlipY = opts.unpackFlipY,
        unpackFlipY = _opts$unpackFlipY === undefined ? true : _opts$unpackFlipY,
        _opts$magFilter = opts.magFilter,
        magFilter = _opts$magFilter === undefined ? _webgl.GL.NEAREST : _opts$magFilter,
        _opts$minFilter = opts.minFilter,
        minFilter = _opts$minFilter === undefined ? _webgl.GL.NEAREST : _opts$minFilter,
        _opts$wrapS = opts.wrapS,
        wrapS = _opts$wrapS === undefined ? _webgl.GL.CLAMP_TO_EDGE : _opts$wrapS,
        _opts$wrapT = opts.wrapT,
        wrapT = _opts$wrapT === undefined ? _webgl.GL.CLAMP_TO_EDGE : _opts$wrapT,
        _opts$target = opts.target,
        target = _opts$target === undefined ? _webgl.GL.TEXTURE_2D : _opts$target,
        handle = opts.handle;


    (0, _webglChecks.assertWebGLContext)(gl);

    this.handle = handle || gl.createTexture();
    this.id = id;
    this.gl = gl;
    this.target = target;
    this.hasFloatTexture = gl.getExtension('OES_texture_float');
    this.width = null;
    this.height = null;
    this.textureUnit = undefined;
    this.userData = {};

    this.setPixelStorageModes(Object.assign({}, opts, { unpackFlipY: unpackFlipY }));
    this.setParameters(Object.assign({}, opts, { magFilter: magFilter, minFilter: minFilter, wrapS: wrapS, wrapT: wrapT }));
  }

  _createClass(Texture, [{
    key: 'delete',
    value: function _delete() {
      if (this.handle) {
        this.gl.deleteTexture(this.handle);
        this.handle = null;
      }
      return this;
    }
  }, {
    key: 'toString',
    value: function toString() {
      return 'Texture(' + this.id + ',' + this.width + 'x' + this.height + ')';
    }
  }, {
    key: 'generateMipmap',
    value: function generateMipmap() {
      this.gl.bindTexture(this.target, this.handle);
      this.gl.generateMipmap(this.target);
      this.gl.bindTexture(this.target, null);
      return this;
    }

    /*
     * @param {*} pixels -
     *  null - create empty texture of specified format
     *  Typed array - init from image data in typed array
     *  Buffer|WebGLBuffer - (WEBGL2) init from image data in WebGLBuffer
     *  HTMLImageElement|Image - Inits with content of image. Auto width/height
     *  HTMLCanvasElement - Inits with contents of canvas. Auto width/height
     *  HTMLVideoElement - Creates video texture. Auto width/height
     *
     * @param {GLint} width -
     * @param {GLint} height -
     * @param {GLint} mipMapLevel -
     * @param {GLenum} format - format of image data.
     * @param {GLenum} type
     *  - format of array (autodetect from type) or
     *  - (WEBGL2) format of buffer
     * @param {Number} offset - (WEBGL2) offset from start of buffer
     * @param {GLint} border - must be 0.
     */
    /* eslint-disable max-len, max-statements, complexity */

  }, {
    key: 'setImageData',
    value: function setImageData(_ref) {
      var _ref$target = _ref.target,
          target = _ref$target === undefined ? this.target : _ref$target,
          _ref$pixels = _ref.pixels,
          pixels = _ref$pixels === undefined ? null : _ref$pixels,
          _ref$data = _ref.data,
          data = _ref$data === undefined ? null : _ref$data,
          width = _ref.width,
          height = _ref.height,
          _ref$mipmapLevel = _ref.mipmapLevel,
          mipmapLevel = _ref$mipmapLevel === undefined ? 0 : _ref$mipmapLevel,
          _ref$format = _ref.format,
          format = _ref$format === undefined ? _webgl.GL.RGBA : _ref$format,
          type = _ref.type,
          _ref$offset = _ref.offset,
          offset = _ref$offset === undefined ? 0 : _ref$offset,
          _ref$border = _ref.border,
          border = _ref$border === undefined ? 0 : _ref$border;
      var gl = this.gl;


      pixels = pixels || data;

      // Support ndarrays
      if (pixels && pixels.data) {
        var ndarray = pixels;
        pixels = ndarray.data;
        width = ndarray.shape[0];
        height = ndarray.shape[1];
      }

      gl.bindTexture(this.target, this.handle);

      if (pixels === null) {

        // Create an minimal texture
        width = width || 1;
        height = height || 1;
        type = type || _webgl.GL.UNSIGNED_BYTE;
        // pixels = new Uint8Array([255, 0, 0, 1]);
        gl.texImage2D(target, mipmapLevel, format, width, height, border, format, type, pixels);
        this.width = width;
        this.height = height;
      } else if (ArrayBuffer.isView(pixels)) {

        // Create from a typed array
        (0, _assert2.default)(width > 0 && height > 0, 'Texture2D: Width and height required');
        type = type || (0, _webgl.glTypeFromArray)(pixels);
        // TODO - WebGL2 check?
        if (type === gl.FLOAT && !this.hasFloatTexture) {
          throw new Error('floating point textures are not supported.');
        }
        gl.texImage2D(target, mipmapLevel, format, width, height, border, format, type, pixels);
        this.width = width;
        this.height = height;
      } else if (pixels instanceof _webgl.WebGLBuffer || pixels instanceof _buffer2.default) {

        // WebGL2 allows us to create texture directly from a WebGL buffer
        (0, _assert2.default)(gl instanceof _webgl.WebGL2RenderingContext, 'Requires WebGL2');
        type = type || _webgl.GL.UNSIGNED_BYTE;
        // This texImage2D signature uses currently bound GL_PIXEL_UNPACK_BUFFER
        var buffer = _buffer2.default.makeFrom(pixels);
        gl.bindBuffer(_webgl.GL.PIXEL_UNPACK_BUFFER, buffer.handle);
        gl.texImage2D(target, mipmapLevel, format, width, height, border, format, type, offset);
        gl.bindBuffer(_webgl.GL.GL_PIXEL_UNPACK_BUFFER, null);
        this.width = width;
        this.height = height;
      } else {

        var imageSize = this._deduceImageSize(pixels);
        // Assume pixels is a browser supported object (ImageData, Canvas, ...)
        (0, _assert2.default)(width === undefined && height === undefined, 'Texture2D.setImageData: Width and height must not be provided');
        type = type || _webgl.GL.UNSIGNED_BYTE;
        gl.texImage2D(target, mipmapLevel, format, format, type, pixels);
        this.width = imageSize.width;
        this.height = imageSize.height;
      }

      gl.bindTexture(this.target, null);

      return this;
    }
    /* eslint-enable max-len, max-statements, complexity */

    /* global ImageData, HTMLImageElement, HTMLCanvasElement, HTMLVideoElement */

  }, {
    key: '_deduceImageSize',
    value: function _deduceImageSize(image) {
      if (typeof ImageData !== 'undefined' && image instanceof ImageData) {
        return { width: image.width, height: image.height };
      } else if (typeof HTMLImageElement !== 'undefined' && image instanceof HTMLImageElement) {
        return { width: image.naturalWidth, height: image.naturalHeight };
      } else if (typeof HTMLCanvasElement !== 'undefined' && image instanceof HTMLCanvasElement) {
        return { width: image.width, height: image.height };
      } else if (typeof HTMLVideoElement !== 'undefined' && image instanceof HTMLVideoElement) {
        return { width: image.videoWidth, height: image.videoHeight };
      }
      throw new Error('Unknown image data format. Failed to deduce image size');
    }

    /**
     * Batch update pixel storage modes
     * @param {GLint} packAlignment - Packing of pixel data in memory (1,2,4,8)
     * @param {GLint} unpackAlignment - Unpacking pixel data from memory(1,2,4,8)
     * @param {GLboolean} unpackFlipY -  Flip source data along its vertical axis
     * @param {GLboolean} unpackPremultiplyAlpha -
     *   Multiplies the alpha channel into the other color channels
     * @param {GLenum} unpackColorspaceConversion -
     *   Default color space conversion or no color space conversion.
     *
     * @param {GLint} packRowLength -
     *  Number of pixels in a row.
     * @param {} packSkipPixels -
     *   Number of pixels skipped before the first pixel is written into memory.
     * @param {} packSkipRows -
     *   Number of rows of pixels skipped before first pixel is written to memory.
     * @param {} unpackRowLength -
     *   Number of pixels in a row.
     * @param {} unpackImageHeight -
     *   Image height used for reading pixel data from memory
     * @param {} unpackSkipPixels -
     *   Number of pixel images skipped before first pixel is read from memory
     * @param {} unpackSkipRows -
     *   Number of rows of pixels skipped before first pixel is read from memory
     * @param {} unpackSkipImages -
     *   Number of pixel images skipped before first pixel is read from memory
     */
    /* eslint-disable complexity, max-statements */

  }, {
    key: 'setPixelStorageModes',
    value: function setPixelStorageModes() {
      var _ref2 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          packAlignment = _ref2.packAlignment,
          unpackAlignment = _ref2.unpackAlignment,
          unpackFlipY = _ref2.unpackFlipY,
          unpackPremultiplyAlpha = _ref2.unpackPremultiplyAlpha,
          unpackColorspaceConversion = _ref2.unpackColorspaceConversion,
          packRowLength = _ref2.packRowLength,
          packSkipPixels = _ref2.packSkipPixels,
          packSkipRows = _ref2.packSkipRows,
          unpackRowLength = _ref2.unpackRowLength,
          unpackImageHeight = _ref2.unpackImageHeight,
          unpackSkipPixels = _ref2.unpackSkipPixels,
          unpackSkipRows = _ref2.unpackSkipRows,
          unpackSkipImages = _ref2.unpackSkipImages;

      var gl = this.gl;


      gl.bindTexture(this.target, this.handle);

      if (packAlignment) {
        gl.pixelStorei(gl.PACK_ALIGNMENT, packAlignment);
      }
      if (unpackAlignment) {
        gl.pixelStorei(gl.UNPACK_ALIGNMENT, unpackAlignment);
      }
      if (unpackFlipY) {
        gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, unpackFlipY);
      }
      if (unpackPremultiplyAlpha) {
        gl.pixelStorei(gl.UNPACK_PREMULTIPLY_ALPHA_WEBGL, unpackPremultiplyAlpha);
      }
      if (unpackColorspaceConversion) {
        gl.pixelStorei(gl.UNPACK_COLORSPACE_CONVERSION_WEBGL, unpackColorspaceConversion);
      }

      // WEBGL2
      if (packRowLength) {
        gl.pixelStorei(gl.PACK_ROW_LENGTH, packRowLength);
      }
      if (packSkipPixels) {
        gl.pixelStorei(gl.PACK_SKIP_PIXELS, packSkipPixels);
      }
      if (packSkipRows) {
        gl.pixelStorei(gl.PACK_SKIP_ROWS, packSkipRows);
      }
      if (unpackRowLength) {
        gl.pixelStorei(gl.UNPACK_ROW_LENGTH, unpackRowLength);
      }
      if (unpackImageHeight) {
        gl.pixelStorei(gl.UNPACK_IMAGE_HEIGHT, unpackImageHeight);
      }
      if (unpackSkipPixels) {
        gl.pixelStorei(gl.UNPACK_SKIP_PIXELS, unpackSkipPixels);
      }
      if (unpackSkipRows) {
        gl.pixelStorei(gl.UNPACK_SKIP_ROWS, unpackSkipRows);
      }
      if (unpackSkipImages) {
        gl.pixelStorei(gl.UNPACK_SKIP_IMAGES, unpackSkipImages);
      }

      gl.bindTexture(this.target, null);
      return this;
    }
    /* eslint-enable complexity, max-statements */

    /**
     * Batch update sampler settings
     *
     * @param {GLenum} magFilter - texture magnification filter.
     * @param {GLenum} minFilter - texture minification filter
     * @param {GLenum} wrapS - texture wrapping function for texture coordinate s.
     * @param {GLenum} wrapT - texture wrapping function for texture coordinate t.
     * WEBGL2 only:
     * @param {GLenum} wrapR - texture wrapping function for texture coordinate r.
     * @param {GLenum} compareFunc - texture comparison function.
     * @param {GLenum} compareMode - texture comparison mode.
     * @param {GLfloat} minLOD - minimum level-of-detail value.
     * @param {GLfloat} maxLOD - maximum level-of-detail value.
     * @param {GLfloat} baseLevel - Texture mipmap level
     * @param {GLfloat} maxLevel - Maximum texture mipmap array level
     */
    /* eslint-disable complexity, max-statements */

  }, {
    key: 'setParameters',
    value: function setParameters(_ref3) {
      var magFilter = _ref3.magFilter,
          minFilter = _ref3.minFilter,
          wrapS = _ref3.wrapS,
          wrapT = _ref3.wrapT,
          wrapR = _ref3.wrapR,
          baseLevel = _ref3.baseLevel,
          maxLevel = _ref3.maxLevel,
          minLOD = _ref3.minLOD,
          maxLOD = _ref3.maxLOD,
          compareFunc = _ref3.compareFunc,
          compareMode = _ref3.compareMode;
      var gl = this.gl;

      gl.bindTexture(this.target, this.handle);

      if (magFilter) {
        gl.texParameteri(this.target, gl.TEXTURE_MAG_FILTER, magFilter);
      }
      if (minFilter) {
        gl.texParameteri(this.target, gl.TEXTURE_MIN_FILTER, minFilter);
      }
      if (wrapS) {
        gl.texParameteri(this.target, gl.TEXTURE_WRAP_S, wrapS);
      }
      if (wrapT) {
        gl.texParameteri(this.target, gl.TEXTURE_WRAP_T, wrapT);
      }
      // WEBGL2
      if (wrapR) {
        gl.texParameteri(this.target, gl.TEXTURE_WRAP_R, wrapR);
      }
      if (baseLevel) {
        gl.texParameteri(this.target, gl.TEXTURE_BASE_LEVEL, baseLevel);
      }
      if (maxLevel) {
        gl.texParameteri(this.target, gl.TEXTURE_MAX_LEVEL, maxLevel);
      }
      if (compareFunc) {
        gl.texParameteri(this.target, gl.TEXTURE_COMPARE_FUNC, compareFunc);
      }
      if (compareMode) {
        gl.texParameteri(this.target, gl.TEXTURE_COMPARE_MODE, compareMode);
      }
      if (minLOD) {
        gl.texParameterf(this.target, gl.TEXTURE_MIN_LOD, minLOD);
      }
      if (maxLOD) {
        gl.texParameterf(this.target, gl.TEXTURE_MAX_LOD, maxLOD);
      }

      gl.bindTexture(this.target, null);
      return this;
    }
    /* eslint-enable complexity, max-statements */

  }, {
    key: 'getParameters',
    value: function getParameters() {
      var gl = this.gl;

      gl.bindTexture(this.target, this.handle);
      var webglParams = {
        magFilter: gl.getTexParameter(this.target, gl.TEXTURE_MAG_FILTER),
        minFilter: gl.getTexParameter(this.target, gl.TEXTURE_MIN_FILTER),
        wrapS: gl.getTexParameter(this.target, gl.TEXTURE_WRAP_S),
        wrapT: gl.getTexParameter(this.target, gl.TEXTURE_WRAP_T)
      };
      gl.bindTexture(this.target, null);
      return webglParams;
    }

    // Deprecated methods

  }, {
    key: 'image2D',
    value: function image2D(_ref4) {
      var pixels = _ref4.pixels,
          _ref4$format = _ref4.format,
          format = _ref4$format === undefined ? _webgl.GL.RGBA : _ref4$format,
          _ref4$type = _ref4.type,
          type = _ref4$type === undefined ? _webgl.GL.UNSIGNED_BYTE : _ref4$type;

      // TODO - WebGL2 check?
      if (type === _webgl.GL.FLOAT && !this.hasFloatTexture) {
        throw new Error('floating point textures are not supported.');
      }

      this.gl.bindTexture(this.target, this.handle);
      this.gl.texImage2D(_webgl.GL.TEXTURE_2D, 0, format, format, type, pixels);
      this.gl.bindTexture(this.target, null);
      return this;
    }
  }, {
    key: 'update',
    value: function update(opts) {
      throw new Error('Texture.update() is deprecated()');
    }
  }]);

  return Texture;
}();

exports.default = Texture;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy93ZWJnbC90ZXh0dXJlLmpzIl0sIm5hbWVzIjpbIlRleHR1cmUiLCJnbCIsIm9wdHMiLCJpZCIsInVucGFja0ZsaXBZIiwibWFnRmlsdGVyIiwiTkVBUkVTVCIsIm1pbkZpbHRlciIsIndyYXBTIiwiQ0xBTVBfVE9fRURHRSIsIndyYXBUIiwidGFyZ2V0IiwiVEVYVFVSRV8yRCIsImhhbmRsZSIsImNyZWF0ZVRleHR1cmUiLCJoYXNGbG9hdFRleHR1cmUiLCJnZXRFeHRlbnNpb24iLCJ3aWR0aCIsImhlaWdodCIsInRleHR1cmVVbml0IiwidW5kZWZpbmVkIiwidXNlckRhdGEiLCJzZXRQaXhlbFN0b3JhZ2VNb2RlcyIsIk9iamVjdCIsImFzc2lnbiIsInNldFBhcmFtZXRlcnMiLCJkZWxldGVUZXh0dXJlIiwiYmluZFRleHR1cmUiLCJnZW5lcmF0ZU1pcG1hcCIsInBpeGVscyIsImRhdGEiLCJtaXBtYXBMZXZlbCIsImZvcm1hdCIsIlJHQkEiLCJ0eXBlIiwib2Zmc2V0IiwiYm9yZGVyIiwibmRhcnJheSIsInNoYXBlIiwiVU5TSUdORURfQllURSIsInRleEltYWdlMkQiLCJBcnJheUJ1ZmZlciIsImlzVmlldyIsIkZMT0FUIiwiRXJyb3IiLCJidWZmZXIiLCJtYWtlRnJvbSIsImJpbmRCdWZmZXIiLCJQSVhFTF9VTlBBQ0tfQlVGRkVSIiwiR0xfUElYRUxfVU5QQUNLX0JVRkZFUiIsImltYWdlU2l6ZSIsIl9kZWR1Y2VJbWFnZVNpemUiLCJpbWFnZSIsIkltYWdlRGF0YSIsIkhUTUxJbWFnZUVsZW1lbnQiLCJuYXR1cmFsV2lkdGgiLCJuYXR1cmFsSGVpZ2h0IiwiSFRNTENhbnZhc0VsZW1lbnQiLCJIVE1MVmlkZW9FbGVtZW50IiwidmlkZW9XaWR0aCIsInZpZGVvSGVpZ2h0IiwicGFja0FsaWdubWVudCIsInVucGFja0FsaWdubWVudCIsInVucGFja1ByZW11bHRpcGx5QWxwaGEiLCJ1bnBhY2tDb2xvcnNwYWNlQ29udmVyc2lvbiIsInBhY2tSb3dMZW5ndGgiLCJwYWNrU2tpcFBpeGVscyIsInBhY2tTa2lwUm93cyIsInVucGFja1Jvd0xlbmd0aCIsInVucGFja0ltYWdlSGVpZ2h0IiwidW5wYWNrU2tpcFBpeGVscyIsInVucGFja1NraXBSb3dzIiwidW5wYWNrU2tpcEltYWdlcyIsInBpeGVsU3RvcmVpIiwiUEFDS19BTElHTk1FTlQiLCJVTlBBQ0tfQUxJR05NRU5UIiwiVU5QQUNLX0ZMSVBfWV9XRUJHTCIsIlVOUEFDS19QUkVNVUxUSVBMWV9BTFBIQV9XRUJHTCIsIlVOUEFDS19DT0xPUlNQQUNFX0NPTlZFUlNJT05fV0VCR0wiLCJQQUNLX1JPV19MRU5HVEgiLCJQQUNLX1NLSVBfUElYRUxTIiwiUEFDS19TS0lQX1JPV1MiLCJVTlBBQ0tfUk9XX0xFTkdUSCIsIlVOUEFDS19JTUFHRV9IRUlHSFQiLCJVTlBBQ0tfU0tJUF9QSVhFTFMiLCJVTlBBQ0tfU0tJUF9ST1dTIiwiVU5QQUNLX1NLSVBfSU1BR0VTIiwid3JhcFIiLCJiYXNlTGV2ZWwiLCJtYXhMZXZlbCIsIm1pbkxPRCIsIm1heExPRCIsImNvbXBhcmVGdW5jIiwiY29tcGFyZU1vZGUiLCJ0ZXhQYXJhbWV0ZXJpIiwiVEVYVFVSRV9NQUdfRklMVEVSIiwiVEVYVFVSRV9NSU5fRklMVEVSIiwiVEVYVFVSRV9XUkFQX1MiLCJURVhUVVJFX1dSQVBfVCIsIlRFWFRVUkVfV1JBUF9SIiwiVEVYVFVSRV9CQVNFX0xFVkVMIiwiVEVYVFVSRV9NQVhfTEVWRUwiLCJURVhUVVJFX0NPTVBBUkVfRlVOQyIsIlRFWFRVUkVfQ09NUEFSRV9NT0RFIiwidGV4UGFyYW1ldGVyZiIsIlRFWFRVUkVfTUlOX0xPRCIsIlRFWFRVUkVfTUFYX0xPRCIsIndlYmdsUGFyYW1zIiwiZ2V0VGV4UGFyYW1ldGVyIl0sIm1hcHBpbmdzIjoiOzs7Ozs7Ozs7QUFBQTs7QUFFQTs7QUFDQTs7OztBQUNBOztBQUNBOzs7Ozs7OztJQUVxQkEsTztBQUVuQixtQkFBWUMsRUFBWixFQUFnQkMsSUFBaEIsRUFBc0I7QUFBQTs7QUFBQSxtQkFVaEJBLElBVmdCLENBRWxCQyxFQUZrQjtBQUFBLFFBRWxCQSxFQUZrQiw0QkFFYixnQkFBSSxTQUFKLENBRmE7QUFBQSw0QkFVaEJELElBVmdCLENBR2xCRSxXQUhrQjtBQUFBLFFBR2xCQSxXQUhrQixxQ0FHSixJQUhJO0FBQUEsMEJBVWhCRixJQVZnQixDQUlsQkcsU0FKa0I7QUFBQSxRQUlsQkEsU0FKa0IsbUNBSU4sVUFBR0MsT0FKRztBQUFBLDBCQVVoQkosSUFWZ0IsQ0FLbEJLLFNBTGtCO0FBQUEsUUFLbEJBLFNBTGtCLG1DQUtOLFVBQUdELE9BTEc7QUFBQSxzQkFVaEJKLElBVmdCLENBTWxCTSxLQU5rQjtBQUFBLFFBTWxCQSxLQU5rQiwrQkFNVixVQUFHQyxhQU5PO0FBQUEsc0JBVWhCUCxJQVZnQixDQU9sQlEsS0FQa0I7QUFBQSxRQU9sQkEsS0FQa0IsK0JBT1YsVUFBR0QsYUFQTztBQUFBLHVCQVVoQlAsSUFWZ0IsQ0FRbEJTLE1BUmtCO0FBQUEsUUFRbEJBLE1BUmtCLGdDQVFULFVBQUdDLFVBUk07QUFBQSxRQVNsQkMsTUFUa0IsR0FVaEJYLElBVmdCLENBU2xCVyxNQVRrQjs7O0FBWXBCLHlDQUFtQlosRUFBbkI7O0FBRUEsU0FBS1ksTUFBTCxHQUFjQSxVQUFVWixHQUFHYSxhQUFILEVBQXhCO0FBQ0EsU0FBS1gsRUFBTCxHQUFVQSxFQUFWO0FBQ0EsU0FBS0YsRUFBTCxHQUFVQSxFQUFWO0FBQ0EsU0FBS1UsTUFBTCxHQUFjQSxNQUFkO0FBQ0EsU0FBS0ksZUFBTCxHQUF1QmQsR0FBR2UsWUFBSCxDQUFnQixtQkFBaEIsQ0FBdkI7QUFDQSxTQUFLQyxLQUFMLEdBQWEsSUFBYjtBQUNBLFNBQUtDLE1BQUwsR0FBYyxJQUFkO0FBQ0EsU0FBS0MsV0FBTCxHQUFtQkMsU0FBbkI7QUFDQSxTQUFLQyxRQUFMLEdBQWdCLEVBQWhCOztBQUVBLFNBQUtDLG9CQUFMLENBQTBCQyxPQUFPQyxNQUFQLENBQWMsRUFBZCxFQUFrQnRCLElBQWxCLEVBQXdCLEVBQUNFLHdCQUFELEVBQXhCLENBQTFCO0FBQ0EsU0FBS3FCLGFBQUwsQ0FBbUJGLE9BQU9DLE1BQVAsQ0FBYyxFQUFkLEVBQWtCdEIsSUFBbEIsRUFBd0IsRUFBQ0csb0JBQUQsRUFBWUUsb0JBQVosRUFBdUJDLFlBQXZCLEVBQThCRSxZQUE5QixFQUF4QixDQUFuQjtBQUNEOzs7OzhCQUVRO0FBQ1AsVUFBSSxLQUFLRyxNQUFULEVBQWlCO0FBQ2YsYUFBS1osRUFBTCxDQUFReUIsYUFBUixDQUFzQixLQUFLYixNQUEzQjtBQUNBLGFBQUtBLE1BQUwsR0FBYyxJQUFkO0FBQ0Q7QUFDRCxhQUFPLElBQVA7QUFDRDs7OytCQUVVO0FBQ1QsMEJBQWtCLEtBQUtWLEVBQXZCLFNBQTZCLEtBQUtjLEtBQWxDLFNBQTJDLEtBQUtDLE1BQWhEO0FBQ0Q7OztxQ0FFZ0I7QUFDZixXQUFLakIsRUFBTCxDQUFRMEIsV0FBUixDQUFvQixLQUFLaEIsTUFBekIsRUFBaUMsS0FBS0UsTUFBdEM7QUFDQSxXQUFLWixFQUFMLENBQVEyQixjQUFSLENBQXVCLEtBQUtqQixNQUE1QjtBQUNBLFdBQUtWLEVBQUwsQ0FBUTBCLFdBQVIsQ0FBb0IsS0FBS2hCLE1BQXpCLEVBQWlDLElBQWpDO0FBQ0EsYUFBTyxJQUFQO0FBQ0Q7O0FBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7QUFtQkE7Ozs7dUNBWUc7QUFBQSw2QkFWREEsTUFVQztBQUFBLFVBVkRBLE1BVUMsK0JBVlEsS0FBS0EsTUFVYjtBQUFBLDZCQVREa0IsTUFTQztBQUFBLFVBVERBLE1BU0MsK0JBVFEsSUFTUjtBQUFBLDJCQVJEQyxJQVFDO0FBQUEsVUFSREEsSUFRQyw2QkFSTSxJQVFOO0FBQUEsVUFQRGIsS0FPQyxRQVBEQSxLQU9DO0FBQUEsVUFOREMsTUFNQyxRQU5EQSxNQU1DO0FBQUEsa0NBTERhLFdBS0M7QUFBQSxVQUxEQSxXQUtDLG9DQUxhLENBS2I7QUFBQSw2QkFKREMsTUFJQztBQUFBLFVBSkRBLE1BSUMsK0JBSlEsVUFBR0MsSUFJWDtBQUFBLFVBSERDLElBR0MsUUFIREEsSUFHQztBQUFBLDZCQUZEQyxNQUVDO0FBQUEsVUFGREEsTUFFQywrQkFGUSxDQUVSO0FBQUEsNkJBRERDLE1BQ0M7QUFBQSxVQUREQSxNQUNDLCtCQURRLENBQ1I7QUFBQSxVQUNNbkMsRUFETixHQUNZLElBRFosQ0FDTUEsRUFETjs7O0FBR0Q0QixlQUFTQSxVQUFVQyxJQUFuQjs7QUFFQTtBQUNBLFVBQUlELFVBQVVBLE9BQU9DLElBQXJCLEVBQTJCO0FBQ3pCLFlBQU1PLFVBQVVSLE1BQWhCO0FBQ0FBLGlCQUFTUSxRQUFRUCxJQUFqQjtBQUNBYixnQkFBUW9CLFFBQVFDLEtBQVIsQ0FBYyxDQUFkLENBQVI7QUFDQXBCLGlCQUFTbUIsUUFBUUMsS0FBUixDQUFjLENBQWQsQ0FBVDtBQUNEOztBQUVEckMsU0FBRzBCLFdBQUgsQ0FBZSxLQUFLaEIsTUFBcEIsRUFBNEIsS0FBS0UsTUFBakM7O0FBRUEsVUFBSWdCLFdBQVcsSUFBZixFQUFxQjs7QUFFbkI7QUFDQVosZ0JBQVFBLFNBQVMsQ0FBakI7QUFDQUMsaUJBQVNBLFVBQVUsQ0FBbkI7QUFDQWdCLGVBQU9BLFFBQVEsVUFBR0ssYUFBbEI7QUFDQTtBQUNBdEMsV0FBR3VDLFVBQUgsQ0FBYzdCLE1BQWQsRUFDRW9CLFdBREYsRUFDZUMsTUFEZixFQUN1QmYsS0FEdkIsRUFDOEJDLE1BRDlCLEVBQ3NDa0IsTUFEdEMsRUFDOENKLE1BRDlDLEVBQ3NERSxJQUR0RCxFQUM0REwsTUFENUQ7QUFFQSxhQUFLWixLQUFMLEdBQWFBLEtBQWI7QUFDQSxhQUFLQyxNQUFMLEdBQWNBLE1BQWQ7QUFFRCxPQVpELE1BWU8sSUFBSXVCLFlBQVlDLE1BQVosQ0FBbUJiLE1BQW5CLENBQUosRUFBZ0M7O0FBRXJDO0FBQ0EsOEJBQU9aLFFBQVEsQ0FBUixJQUFhQyxTQUFTLENBQTdCLEVBQWdDLHNDQUFoQztBQUNBZ0IsZUFBT0EsUUFBUSw0QkFBZ0JMLE1BQWhCLENBQWY7QUFDQTtBQUNBLFlBQUlLLFNBQVNqQyxHQUFHMEMsS0FBWixJQUFxQixDQUFDLEtBQUs1QixlQUEvQixFQUFnRDtBQUM5QyxnQkFBTSxJQUFJNkIsS0FBSixDQUFVLDRDQUFWLENBQU47QUFDRDtBQUNEM0MsV0FBR3VDLFVBQUgsQ0FBYzdCLE1BQWQsRUFDRW9CLFdBREYsRUFDZUMsTUFEZixFQUN1QmYsS0FEdkIsRUFDOEJDLE1BRDlCLEVBQ3NDa0IsTUFEdEMsRUFDOENKLE1BRDlDLEVBQ3NERSxJQUR0RCxFQUM0REwsTUFENUQ7QUFFQSxhQUFLWixLQUFMLEdBQWFBLEtBQWI7QUFDQSxhQUFLQyxNQUFMLEdBQWNBLE1BQWQ7QUFFRCxPQWRNLE1BY0EsSUFBSVcsd0NBQWlDQSxrQ0FBckMsRUFBK0Q7O0FBRXBFO0FBQ0EsOEJBQU81QiwyQ0FBUCxFQUE2QyxpQkFBN0M7QUFDQWlDLGVBQU9BLFFBQVEsVUFBR0ssYUFBbEI7QUFDQTtBQUNBLFlBQU1NLFNBQVMsaUJBQU9DLFFBQVAsQ0FBZ0JqQixNQUFoQixDQUFmO0FBQ0E1QixXQUFHOEMsVUFBSCxDQUFjLFVBQUdDLG1CQUFqQixFQUFzQ0gsT0FBT2hDLE1BQTdDO0FBQ0FaLFdBQUd1QyxVQUFILENBQWM3QixNQUFkLEVBQ0VvQixXQURGLEVBQ2VDLE1BRGYsRUFDdUJmLEtBRHZCLEVBQzhCQyxNQUQ5QixFQUNzQ2tCLE1BRHRDLEVBQzhDSixNQUQ5QyxFQUNzREUsSUFEdEQsRUFDNERDLE1BRDVEO0FBRUFsQyxXQUFHOEMsVUFBSCxDQUFjLFVBQUdFLHNCQUFqQixFQUF5QyxJQUF6QztBQUNBLGFBQUtoQyxLQUFMLEdBQWFBLEtBQWI7QUFDQSxhQUFLQyxNQUFMLEdBQWNBLE1BQWQ7QUFFRCxPQWRNLE1BY0E7O0FBRUwsWUFBTWdDLFlBQVksS0FBS0MsZ0JBQUwsQ0FBc0J0QixNQUF0QixDQUFsQjtBQUNBO0FBQ0EsOEJBQU9aLFVBQVVHLFNBQVYsSUFBdUJGLFdBQVdFLFNBQXpDLEVBQ0UsK0RBREY7QUFFQWMsZUFBT0EsUUFBUSxVQUFHSyxhQUFsQjtBQUNBdEMsV0FBR3VDLFVBQUgsQ0FBYzdCLE1BQWQsRUFBc0JvQixXQUF0QixFQUFtQ0MsTUFBbkMsRUFBMkNBLE1BQTNDLEVBQW1ERSxJQUFuRCxFQUF5REwsTUFBekQ7QUFDQSxhQUFLWixLQUFMLEdBQWFpQyxVQUFVakMsS0FBdkI7QUFDQSxhQUFLQyxNQUFMLEdBQWNnQyxVQUFVaEMsTUFBeEI7QUFDRDs7QUFFRGpCLFNBQUcwQixXQUFILENBQWUsS0FBS2hCLE1BQXBCLEVBQTRCLElBQTVCOztBQUVBLGFBQU8sSUFBUDtBQUNEO0FBQ0Q7O0FBRUE7Ozs7cUNBQ2lCeUMsSyxFQUFPO0FBQ3RCLFVBQUksT0FBT0MsU0FBUCxLQUFxQixXQUFyQixJQUFvQ0QsaUJBQWlCQyxTQUF6RCxFQUFvRTtBQUNsRSxlQUFPLEVBQUNwQyxPQUFPbUMsTUFBTW5DLEtBQWQsRUFBcUJDLFFBQVFrQyxNQUFNbEMsTUFBbkMsRUFBUDtBQUNELE9BRkQsTUFFTyxJQUFJLE9BQU9vQyxnQkFBUCxLQUE0QixXQUE1QixJQUNURixpQkFBaUJFLGdCQURaLEVBQzhCO0FBQ25DLGVBQU8sRUFBQ3JDLE9BQU9tQyxNQUFNRyxZQUFkLEVBQTRCckMsUUFBUWtDLE1BQU1JLGFBQTFDLEVBQVA7QUFDRCxPQUhNLE1BR0EsSUFBSSxPQUFPQyxpQkFBUCxLQUE2QixXQUE3QixJQUNUTCxpQkFBaUJLLGlCQURaLEVBQytCO0FBQ3BDLGVBQU8sRUFBQ3hDLE9BQU9tQyxNQUFNbkMsS0FBZCxFQUFxQkMsUUFBUWtDLE1BQU1sQyxNQUFuQyxFQUFQO0FBQ0QsT0FITSxNQUdBLElBQUksT0FBT3dDLGdCQUFQLEtBQTRCLFdBQTVCLElBQ1ROLGlCQUFpQk0sZ0JBRFosRUFDOEI7QUFDbkMsZUFBTyxFQUFDekMsT0FBT21DLE1BQU1PLFVBQWQsRUFBMEJ6QyxRQUFRa0MsTUFBTVEsV0FBeEMsRUFBUDtBQUNEO0FBQ0QsWUFBTSxJQUFJaEIsS0FBSixDQUFVLHdEQUFWLENBQU47QUFDRDs7QUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0FBMkJBOzs7OzJDQWdCUTtBQUFBLHNGQUFKLEVBQUk7QUFBQSxVQWROaUIsYUFjTSxTQWROQSxhQWNNO0FBQUEsVUFiTkMsZUFhTSxTQWJOQSxlQWFNO0FBQUEsVUFaTjFELFdBWU0sU0FaTkEsV0FZTTtBQUFBLFVBWE4yRCxzQkFXTSxTQVhOQSxzQkFXTTtBQUFBLFVBVk5DLDBCQVVNLFNBVk5BLDBCQVVNO0FBQUEsVUFSTkMsYUFRTSxTQVJOQSxhQVFNO0FBQUEsVUFQTkMsY0FPTSxTQVBOQSxjQU9NO0FBQUEsVUFOTkMsWUFNTSxTQU5OQSxZQU1NO0FBQUEsVUFMTkMsZUFLTSxTQUxOQSxlQUtNO0FBQUEsVUFKTkMsaUJBSU0sU0FKTkEsaUJBSU07QUFBQSxVQUhOQyxnQkFHTSxTQUhOQSxnQkFHTTtBQUFBLFVBRk5DLGNBRU0sU0FGTkEsY0FFTTtBQUFBLFVBRE5DLGdCQUNNLFNBRE5BLGdCQUNNOztBQUFBLFVBQ0N2RSxFQURELEdBQ08sSUFEUCxDQUNDQSxFQUREOzs7QUFHTkEsU0FBRzBCLFdBQUgsQ0FBZSxLQUFLaEIsTUFBcEIsRUFBNEIsS0FBS0UsTUFBakM7O0FBRUEsVUFBSWdELGFBQUosRUFBbUI7QUFDakI1RCxXQUFHd0UsV0FBSCxDQUFleEUsR0FBR3lFLGNBQWxCLEVBQWtDYixhQUFsQztBQUNEO0FBQ0QsVUFBSUMsZUFBSixFQUFxQjtBQUNuQjdELFdBQUd3RSxXQUFILENBQWV4RSxHQUFHMEUsZ0JBQWxCLEVBQW9DYixlQUFwQztBQUNEO0FBQ0QsVUFBSTFELFdBQUosRUFBaUI7QUFDZkgsV0FBR3dFLFdBQUgsQ0FBZXhFLEdBQUcyRSxtQkFBbEIsRUFBdUN4RSxXQUF2QztBQUNEO0FBQ0QsVUFBSTJELHNCQUFKLEVBQTRCO0FBQzFCOUQsV0FBR3dFLFdBQUgsQ0FBZXhFLEdBQUc0RSw4QkFBbEIsRUFBa0RkLHNCQUFsRDtBQUNEO0FBQ0QsVUFBSUMsMEJBQUosRUFBZ0M7QUFDOUIvRCxXQUFHd0UsV0FBSCxDQUFleEUsR0FBRzZFLGtDQUFsQixFQUNFZCwwQkFERjtBQUVEOztBQUVEO0FBQ0EsVUFBSUMsYUFBSixFQUFtQjtBQUNqQmhFLFdBQUd3RSxXQUFILENBQWV4RSxHQUFHOEUsZUFBbEIsRUFBbUNkLGFBQW5DO0FBQ0Q7QUFDRCxVQUFJQyxjQUFKLEVBQW9CO0FBQ2xCakUsV0FBR3dFLFdBQUgsQ0FBZXhFLEdBQUcrRSxnQkFBbEIsRUFBb0NkLGNBQXBDO0FBQ0Q7QUFDRCxVQUFJQyxZQUFKLEVBQWtCO0FBQ2hCbEUsV0FBR3dFLFdBQUgsQ0FBZXhFLEdBQUdnRixjQUFsQixFQUFrQ2QsWUFBbEM7QUFDRDtBQUNELFVBQUlDLGVBQUosRUFBcUI7QUFDbkJuRSxXQUFHd0UsV0FBSCxDQUFleEUsR0FBR2lGLGlCQUFsQixFQUFxQ2QsZUFBckM7QUFDRDtBQUNELFVBQUlDLGlCQUFKLEVBQXVCO0FBQ3JCcEUsV0FBR3dFLFdBQUgsQ0FBZXhFLEdBQUdrRixtQkFBbEIsRUFBdUNkLGlCQUF2QztBQUNEO0FBQ0QsVUFBSUMsZ0JBQUosRUFBc0I7QUFDcEJyRSxXQUFHd0UsV0FBSCxDQUFleEUsR0FBR21GLGtCQUFsQixFQUFzQ2QsZ0JBQXRDO0FBQ0Q7QUFDRCxVQUFJQyxjQUFKLEVBQW9CO0FBQ2xCdEUsV0FBR3dFLFdBQUgsQ0FBZXhFLEdBQUdvRixnQkFBbEIsRUFBb0NkLGNBQXBDO0FBQ0Q7QUFDRCxVQUFJQyxnQkFBSixFQUFzQjtBQUNwQnZFLFdBQUd3RSxXQUFILENBQWV4RSxHQUFHcUYsa0JBQWxCLEVBQXNDZCxnQkFBdEM7QUFDRDs7QUFFRHZFLFNBQUcwQixXQUFILENBQWUsS0FBS2hCLE1BQXBCLEVBQTRCLElBQTVCO0FBQ0EsYUFBTyxJQUFQO0FBQ0Q7QUFDRDs7QUFFQTs7Ozs7Ozs7Ozs7Ozs7OztBQWdCQTs7Ozt5Q0FjRztBQUFBLFVBWkROLFNBWUMsU0FaREEsU0FZQztBQUFBLFVBWERFLFNBV0MsU0FYREEsU0FXQztBQUFBLFVBVkRDLEtBVUMsU0FWREEsS0FVQztBQUFBLFVBVERFLEtBU0MsU0FUREEsS0FTQztBQUFBLFVBUEQ2RSxLQU9DLFNBUERBLEtBT0M7QUFBQSxVQU5EQyxTQU1DLFNBTkRBLFNBTUM7QUFBQSxVQUxEQyxRQUtDLFNBTERBLFFBS0M7QUFBQSxVQUpEQyxNQUlDLFNBSkRBLE1BSUM7QUFBQSxVQUhEQyxNQUdDLFNBSERBLE1BR0M7QUFBQSxVQUZEQyxXQUVDLFNBRkRBLFdBRUM7QUFBQSxVQUREQyxXQUNDLFNBRERBLFdBQ0M7QUFBQSxVQUNNNUYsRUFETixHQUNZLElBRFosQ0FDTUEsRUFETjs7QUFFREEsU0FBRzBCLFdBQUgsQ0FBZSxLQUFLaEIsTUFBcEIsRUFBNEIsS0FBS0UsTUFBakM7O0FBRUEsVUFBSVIsU0FBSixFQUFlO0FBQ2JKLFdBQUc2RixhQUFILENBQWlCLEtBQUtuRixNQUF0QixFQUE4QlYsR0FBRzhGLGtCQUFqQyxFQUFxRDFGLFNBQXJEO0FBQ0Q7QUFDRCxVQUFJRSxTQUFKLEVBQWU7QUFDYk4sV0FBRzZGLGFBQUgsQ0FBaUIsS0FBS25GLE1BQXRCLEVBQThCVixHQUFHK0Ysa0JBQWpDLEVBQXFEekYsU0FBckQ7QUFDRDtBQUNELFVBQUlDLEtBQUosRUFBVztBQUNUUCxXQUFHNkYsYUFBSCxDQUFpQixLQUFLbkYsTUFBdEIsRUFBOEJWLEdBQUdnRyxjQUFqQyxFQUFpRHpGLEtBQWpEO0FBQ0Q7QUFDRCxVQUFJRSxLQUFKLEVBQVc7QUFDVFQsV0FBRzZGLGFBQUgsQ0FBaUIsS0FBS25GLE1BQXRCLEVBQThCVixHQUFHaUcsY0FBakMsRUFBaUR4RixLQUFqRDtBQUNEO0FBQ0Q7QUFDQSxVQUFJNkUsS0FBSixFQUFXO0FBQ1R0RixXQUFHNkYsYUFBSCxDQUFpQixLQUFLbkYsTUFBdEIsRUFBOEJWLEdBQUdrRyxjQUFqQyxFQUFpRFosS0FBakQ7QUFDRDtBQUNELFVBQUlDLFNBQUosRUFBZTtBQUNidkYsV0FBRzZGLGFBQUgsQ0FBaUIsS0FBS25GLE1BQXRCLEVBQThCVixHQUFHbUcsa0JBQWpDLEVBQXFEWixTQUFyRDtBQUNEO0FBQ0QsVUFBSUMsUUFBSixFQUFjO0FBQ1p4RixXQUFHNkYsYUFBSCxDQUFpQixLQUFLbkYsTUFBdEIsRUFBOEJWLEdBQUdvRyxpQkFBakMsRUFBb0RaLFFBQXBEO0FBQ0Q7QUFDRCxVQUFJRyxXQUFKLEVBQWlCO0FBQ2YzRixXQUFHNkYsYUFBSCxDQUFpQixLQUFLbkYsTUFBdEIsRUFBOEJWLEdBQUdxRyxvQkFBakMsRUFBdURWLFdBQXZEO0FBQ0Q7QUFDRCxVQUFJQyxXQUFKLEVBQWlCO0FBQ2Y1RixXQUFHNkYsYUFBSCxDQUFpQixLQUFLbkYsTUFBdEIsRUFBOEJWLEdBQUdzRyxvQkFBakMsRUFBdURWLFdBQXZEO0FBQ0Q7QUFDRCxVQUFJSCxNQUFKLEVBQVk7QUFDVnpGLFdBQUd1RyxhQUFILENBQWlCLEtBQUs3RixNQUF0QixFQUE4QlYsR0FBR3dHLGVBQWpDLEVBQWtEZixNQUFsRDtBQUNEO0FBQ0QsVUFBSUMsTUFBSixFQUFZO0FBQ1YxRixXQUFHdUcsYUFBSCxDQUFpQixLQUFLN0YsTUFBdEIsRUFBOEJWLEdBQUd5RyxlQUFqQyxFQUFrRGYsTUFBbEQ7QUFDRDs7QUFFRDFGLFNBQUcwQixXQUFILENBQWUsS0FBS2hCLE1BQXBCLEVBQTRCLElBQTVCO0FBQ0EsYUFBTyxJQUFQO0FBQ0Q7QUFDRDs7OztvQ0FFZ0I7QUFBQSxVQUNQVixFQURPLEdBQ0QsSUFEQyxDQUNQQSxFQURPOztBQUVkQSxTQUFHMEIsV0FBSCxDQUFlLEtBQUtoQixNQUFwQixFQUE0QixLQUFLRSxNQUFqQztBQUNBLFVBQU04RixjQUFjO0FBQ2xCdEcsbUJBQVdKLEdBQUcyRyxlQUFILENBQW1CLEtBQUtqRyxNQUF4QixFQUFnQ1YsR0FBRzhGLGtCQUFuQyxDQURPO0FBRWxCeEYsbUJBQVdOLEdBQUcyRyxlQUFILENBQW1CLEtBQUtqRyxNQUF4QixFQUFnQ1YsR0FBRytGLGtCQUFuQyxDQUZPO0FBR2xCeEYsZUFBT1AsR0FBRzJHLGVBQUgsQ0FBbUIsS0FBS2pHLE1BQXhCLEVBQWdDVixHQUFHZ0csY0FBbkMsQ0FIVztBQUlsQnZGLGVBQU9ULEdBQUcyRyxlQUFILENBQW1CLEtBQUtqRyxNQUF4QixFQUFnQ1YsR0FBR2lHLGNBQW5DO0FBSlcsT0FBcEI7QUFNQWpHLFNBQUcwQixXQUFILENBQWUsS0FBS2hCLE1BQXBCLEVBQTRCLElBQTVCO0FBQ0EsYUFBT2dHLFdBQVA7QUFDRDs7QUFFRDs7OzttQ0FNRztBQUFBLFVBSEQ5RSxNQUdDLFNBSERBLE1BR0M7QUFBQSwrQkFGREcsTUFFQztBQUFBLFVBRkRBLE1BRUMsZ0NBRlEsVUFBR0MsSUFFWDtBQUFBLDZCQUREQyxJQUNDO0FBQUEsVUFEREEsSUFDQyw4QkFETSxVQUFHSyxhQUNUOztBQUNEO0FBQ0EsVUFBSUwsU0FBUyxVQUFHUyxLQUFaLElBQXFCLENBQUMsS0FBSzVCLGVBQS9CLEVBQWdEO0FBQzlDLGNBQU0sSUFBSTZCLEtBQUosQ0FBVSw0Q0FBVixDQUFOO0FBQ0Q7O0FBRUQsV0FBSzNDLEVBQUwsQ0FBUTBCLFdBQVIsQ0FBb0IsS0FBS2hCLE1BQXpCLEVBQWlDLEtBQUtFLE1BQXRDO0FBQ0EsV0FBS1osRUFBTCxDQUFRdUMsVUFBUixDQUFtQixVQUFHNUIsVUFBdEIsRUFBa0MsQ0FBbEMsRUFBcUNvQixNQUFyQyxFQUE2Q0EsTUFBN0MsRUFBcURFLElBQXJELEVBQTJETCxNQUEzRDtBQUNBLFdBQUs1QixFQUFMLENBQVEwQixXQUFSLENBQW9CLEtBQUtoQixNQUF6QixFQUFpQyxJQUFqQztBQUNBLGFBQU8sSUFBUDtBQUNEOzs7MkJBRU1ULEksRUFBTTtBQUNYLFlBQU0sSUFBSTBDLEtBQUosQ0FBVSxrQ0FBVixDQUFOO0FBQ0Q7Ozs7OztrQkFyWGtCNUMsTyIsImZpbGUiOiJ0ZXh0dXJlLmpzIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHtHTCwgV2ViR0wyUmVuZGVyaW5nQ29udGV4dCwgV2ViR0xCdWZmZXIsIGdsVHlwZUZyb21BcnJheX1cbiAgZnJvbSAnLi93ZWJnbCc7XG5pbXBvcnQge2Fzc2VydFdlYkdMQ29udGV4dH0gZnJvbSAnLi93ZWJnbC1jaGVja3MnO1xuaW1wb3J0IEJ1ZmZlciBmcm9tICcuL2J1ZmZlcic7XG5pbXBvcnQge3VpZH0gZnJvbSAnLi4vdXRpbHMnO1xuaW1wb3J0IGFzc2VydCBmcm9tICdhc3NlcnQnO1xuXG5leHBvcnQgZGVmYXVsdCBjbGFzcyBUZXh0dXJlIHtcblxuICBjb25zdHJ1Y3RvcihnbCwgb3B0cykge1xuICAgIGNvbnN0IHtcbiAgICAgIGlkID0gdWlkKCd0ZXh0dXJlJyksXG4gICAgICB1bnBhY2tGbGlwWSA9IHRydWUsXG4gICAgICBtYWdGaWx0ZXIgPSBHTC5ORUFSRVNULFxuICAgICAgbWluRmlsdGVyID0gR0wuTkVBUkVTVCxcbiAgICAgIHdyYXBTID0gR0wuQ0xBTVBfVE9fRURHRSxcbiAgICAgIHdyYXBUID0gR0wuQ0xBTVBfVE9fRURHRSxcbiAgICAgIHRhcmdldCA9IEdMLlRFWFRVUkVfMkQsXG4gICAgICBoYW5kbGVcbiAgICB9ID0gb3B0cztcblxuICAgIGFzc2VydFdlYkdMQ29udGV4dChnbCk7XG5cbiAgICB0aGlzLmhhbmRsZSA9IGhhbmRsZSB8fCBnbC5jcmVhdGVUZXh0dXJlKCk7XG4gICAgdGhpcy5pZCA9IGlkO1xuICAgIHRoaXMuZ2wgPSBnbDtcbiAgICB0aGlzLnRhcmdldCA9IHRhcmdldDtcbiAgICB0aGlzLmhhc0Zsb2F0VGV4dHVyZSA9IGdsLmdldEV4dGVuc2lvbignT0VTX3RleHR1cmVfZmxvYXQnKTtcbiAgICB0aGlzLndpZHRoID0gbnVsbDtcbiAgICB0aGlzLmhlaWdodCA9IG51bGw7XG4gICAgdGhpcy50ZXh0dXJlVW5pdCA9IHVuZGVmaW5lZDtcbiAgICB0aGlzLnVzZXJEYXRhID0ge307XG5cbiAgICB0aGlzLnNldFBpeGVsU3RvcmFnZU1vZGVzKE9iamVjdC5hc3NpZ24oe30sIG9wdHMsIHt1bnBhY2tGbGlwWX0pKTtcbiAgICB0aGlzLnNldFBhcmFtZXRlcnMoT2JqZWN0LmFzc2lnbih7fSwgb3B0cywge21hZ0ZpbHRlciwgbWluRmlsdGVyLCB3cmFwUywgd3JhcFR9KSk7XG4gIH1cblxuICBkZWxldGUoKSB7XG4gICAgaWYgKHRoaXMuaGFuZGxlKSB7XG4gICAgICB0aGlzLmdsLmRlbGV0ZVRleHR1cmUodGhpcy5oYW5kbGUpO1xuICAgICAgdGhpcy5oYW5kbGUgPSBudWxsO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIHRvU3RyaW5nKCkge1xuICAgIHJldHVybiBgVGV4dHVyZSgke3RoaXMuaWR9LCR7dGhpcy53aWR0aH14JHt0aGlzLmhlaWdodH0pYDtcbiAgfVxuXG4gIGdlbmVyYXRlTWlwbWFwKCkge1xuICAgIHRoaXMuZ2wuYmluZFRleHR1cmUodGhpcy50YXJnZXQsIHRoaXMuaGFuZGxlKTtcbiAgICB0aGlzLmdsLmdlbmVyYXRlTWlwbWFwKHRoaXMudGFyZ2V0KTtcbiAgICB0aGlzLmdsLmJpbmRUZXh0dXJlKHRoaXMudGFyZ2V0LCBudWxsKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qXG4gICAqIEBwYXJhbSB7Kn0gcGl4ZWxzIC1cbiAgICogIG51bGwgLSBjcmVhdGUgZW1wdHkgdGV4dHVyZSBvZiBzcGVjaWZpZWQgZm9ybWF0XG4gICAqICBUeXBlZCBhcnJheSAtIGluaXQgZnJvbSBpbWFnZSBkYXRhIGluIHR5cGVkIGFycmF5XG4gICAqICBCdWZmZXJ8V2ViR0xCdWZmZXIgLSAoV0VCR0wyKSBpbml0IGZyb20gaW1hZ2UgZGF0YSBpbiBXZWJHTEJ1ZmZlclxuICAgKiAgSFRNTEltYWdlRWxlbWVudHxJbWFnZSAtIEluaXRzIHdpdGggY29udGVudCBvZiBpbWFnZS4gQXV0byB3aWR0aC9oZWlnaHRcbiAgICogIEhUTUxDYW52YXNFbGVtZW50IC0gSW5pdHMgd2l0aCBjb250ZW50cyBvZiBjYW52YXMuIEF1dG8gd2lkdGgvaGVpZ2h0XG4gICAqICBIVE1MVmlkZW9FbGVtZW50IC0gQ3JlYXRlcyB2aWRlbyB0ZXh0dXJlLiBBdXRvIHdpZHRoL2hlaWdodFxuICAgKlxuICAgKiBAcGFyYW0ge0dMaW50fSB3aWR0aCAtXG4gICAqIEBwYXJhbSB7R0xpbnR9IGhlaWdodCAtXG4gICAqIEBwYXJhbSB7R0xpbnR9IG1pcE1hcExldmVsIC1cbiAgICogQHBhcmFtIHtHTGVudW19IGZvcm1hdCAtIGZvcm1hdCBvZiBpbWFnZSBkYXRhLlxuICAgKiBAcGFyYW0ge0dMZW51bX0gdHlwZVxuICAgKiAgLSBmb3JtYXQgb2YgYXJyYXkgKGF1dG9kZXRlY3QgZnJvbSB0eXBlKSBvclxuICAgKiAgLSAoV0VCR0wyKSBmb3JtYXQgb2YgYnVmZmVyXG4gICAqIEBwYXJhbSB7TnVtYmVyfSBvZmZzZXQgLSAoV0VCR0wyKSBvZmZzZXQgZnJvbSBzdGFydCBvZiBidWZmZXJcbiAgICogQHBhcmFtIHtHTGludH0gYm9yZGVyIC0gbXVzdCBiZSAwLlxuICAgKi9cbiAgLyogZXNsaW50LWRpc2FibGUgbWF4LWxlbiwgbWF4LXN0YXRlbWVudHMsIGNvbXBsZXhpdHkgKi9cbiAgc2V0SW1hZ2VEYXRhKHtcbiAgICB0YXJnZXQgPSB0aGlzLnRhcmdldCxcbiAgICBwaXhlbHMgPSBudWxsLFxuICAgIGRhdGEgPSBudWxsLFxuICAgIHdpZHRoLFxuICAgIGhlaWdodCxcbiAgICBtaXBtYXBMZXZlbCA9IDAsXG4gICAgZm9ybWF0ID0gR0wuUkdCQSxcbiAgICB0eXBlLFxuICAgIG9mZnNldCA9IDAsXG4gICAgYm9yZGVyID0gMFxuICB9KSB7XG4gICAgY29uc3Qge2dsfSA9IHRoaXM7XG5cbiAgICBwaXhlbHMgPSBwaXhlbHMgfHwgZGF0YTtcblxuICAgIC8vIFN1cHBvcnQgbmRhcnJheXNcbiAgICBpZiAocGl4ZWxzICYmIHBpeGVscy5kYXRhKSB7XG4gICAgICBjb25zdCBuZGFycmF5ID0gcGl4ZWxzO1xuICAgICAgcGl4ZWxzID0gbmRhcnJheS5kYXRhO1xuICAgICAgd2lkdGggPSBuZGFycmF5LnNoYXBlWzBdO1xuICAgICAgaGVpZ2h0ID0gbmRhcnJheS5zaGFwZVsxXTtcbiAgICB9XG5cbiAgICBnbC5iaW5kVGV4dHVyZSh0aGlzLnRhcmdldCwgdGhpcy5oYW5kbGUpO1xuXG4gICAgaWYgKHBpeGVscyA9PT0gbnVsbCkge1xuXG4gICAgICAvLyBDcmVhdGUgYW4gbWluaW1hbCB0ZXh0dXJlXG4gICAgICB3aWR0aCA9IHdpZHRoIHx8IDE7XG4gICAgICBoZWlnaHQgPSBoZWlnaHQgfHwgMTtcbiAgICAgIHR5cGUgPSB0eXBlIHx8IEdMLlVOU0lHTkVEX0JZVEU7XG4gICAgICAvLyBwaXhlbHMgPSBuZXcgVWludDhBcnJheShbMjU1LCAwLCAwLCAxXSk7XG4gICAgICBnbC50ZXhJbWFnZTJEKHRhcmdldCxcbiAgICAgICAgbWlwbWFwTGV2ZWwsIGZvcm1hdCwgd2lkdGgsIGhlaWdodCwgYm9yZGVyLCBmb3JtYXQsIHR5cGUsIHBpeGVscyk7XG4gICAgICB0aGlzLndpZHRoID0gd2lkdGg7XG4gICAgICB0aGlzLmhlaWdodCA9IGhlaWdodDtcblxuICAgIH0gZWxzZSBpZiAoQXJyYXlCdWZmZXIuaXNWaWV3KHBpeGVscykpIHtcblxuICAgICAgLy8gQ3JlYXRlIGZyb20gYSB0eXBlZCBhcnJheVxuICAgICAgYXNzZXJ0KHdpZHRoID4gMCAmJiBoZWlnaHQgPiAwLCAnVGV4dHVyZTJEOiBXaWR0aCBhbmQgaGVpZ2h0IHJlcXVpcmVkJyk7XG4gICAgICB0eXBlID0gdHlwZSB8fCBnbFR5cGVGcm9tQXJyYXkocGl4ZWxzKTtcbiAgICAgIC8vIFRPRE8gLSBXZWJHTDIgY2hlY2s/XG4gICAgICBpZiAodHlwZSA9PT0gZ2wuRkxPQVQgJiYgIXRoaXMuaGFzRmxvYXRUZXh0dXJlKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignZmxvYXRpbmcgcG9pbnQgdGV4dHVyZXMgYXJlIG5vdCBzdXBwb3J0ZWQuJyk7XG4gICAgICB9XG4gICAgICBnbC50ZXhJbWFnZTJEKHRhcmdldCxcbiAgICAgICAgbWlwbWFwTGV2ZWwsIGZvcm1hdCwgd2lkdGgsIGhlaWdodCwgYm9yZGVyLCBmb3JtYXQsIHR5cGUsIHBpeGVscyk7XG4gICAgICB0aGlzLndpZHRoID0gd2lkdGg7XG4gICAgICB0aGlzLmhlaWdodCA9IGhlaWdodDtcblxuICAgIH0gZWxzZSBpZiAocGl4ZWxzIGluc3RhbmNlb2YgV2ViR0xCdWZmZXIgfHwgcGl4ZWxzIGluc3RhbmNlb2YgQnVmZmVyKSB7XG5cbiAgICAgIC8vIFdlYkdMMiBhbGxvd3MgdXMgdG8gY3JlYXRlIHRleHR1cmUgZGlyZWN0bHkgZnJvbSBhIFdlYkdMIGJ1ZmZlclxuICAgICAgYXNzZXJ0KGdsIGluc3RhbmNlb2YgV2ViR0wyUmVuZGVyaW5nQ29udGV4dCwgJ1JlcXVpcmVzIFdlYkdMMicpO1xuICAgICAgdHlwZSA9IHR5cGUgfHwgR0wuVU5TSUdORURfQllURTtcbiAgICAgIC8vIFRoaXMgdGV4SW1hZ2UyRCBzaWduYXR1cmUgdXNlcyBjdXJyZW50bHkgYm91bmQgR0xfUElYRUxfVU5QQUNLX0JVRkZFUlxuICAgICAgY29uc3QgYnVmZmVyID0gQnVmZmVyLm1ha2VGcm9tKHBpeGVscyk7XG4gICAgICBnbC5iaW5kQnVmZmVyKEdMLlBJWEVMX1VOUEFDS19CVUZGRVIsIGJ1ZmZlci5oYW5kbGUpO1xuICAgICAgZ2wudGV4SW1hZ2UyRCh0YXJnZXQsXG4gICAgICAgIG1pcG1hcExldmVsLCBmb3JtYXQsIHdpZHRoLCBoZWlnaHQsIGJvcmRlciwgZm9ybWF0LCB0eXBlLCBvZmZzZXQpO1xuICAgICAgZ2wuYmluZEJ1ZmZlcihHTC5HTF9QSVhFTF9VTlBBQ0tfQlVGRkVSLCBudWxsKTtcbiAgICAgIHRoaXMud2lkdGggPSB3aWR0aDtcbiAgICAgIHRoaXMuaGVpZ2h0ID0gaGVpZ2h0O1xuXG4gICAgfSBlbHNlIHtcblxuICAgICAgY29uc3QgaW1hZ2VTaXplID0gdGhpcy5fZGVkdWNlSW1hZ2VTaXplKHBpeGVscyk7XG4gICAgICAvLyBBc3N1bWUgcGl4ZWxzIGlzIGEgYnJvd3NlciBzdXBwb3J0ZWQgb2JqZWN0IChJbWFnZURhdGEsIENhbnZhcywgLi4uKVxuICAgICAgYXNzZXJ0KHdpZHRoID09PSB1bmRlZmluZWQgJiYgaGVpZ2h0ID09PSB1bmRlZmluZWQsXG4gICAgICAgICdUZXh0dXJlMkQuc2V0SW1hZ2VEYXRhOiBXaWR0aCBhbmQgaGVpZ2h0IG11c3Qgbm90IGJlIHByb3ZpZGVkJyk7XG4gICAgICB0eXBlID0gdHlwZSB8fCBHTC5VTlNJR05FRF9CWVRFO1xuICAgICAgZ2wudGV4SW1hZ2UyRCh0YXJnZXQsIG1pcG1hcExldmVsLCBmb3JtYXQsIGZvcm1hdCwgdHlwZSwgcGl4ZWxzKTtcbiAgICAgIHRoaXMud2lkdGggPSBpbWFnZVNpemUud2lkdGg7XG4gICAgICB0aGlzLmhlaWdodCA9IGltYWdlU2l6ZS5oZWlnaHQ7XG4gICAgfVxuXG4gICAgZ2wuYmluZFRleHR1cmUodGhpcy50YXJnZXQsIG51bGwpO1xuXG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cbiAgLyogZXNsaW50LWVuYWJsZSBtYXgtbGVuLCBtYXgtc3RhdGVtZW50cywgY29tcGxleGl0eSAqL1xuXG4gIC8qIGdsb2JhbCBJbWFnZURhdGEsIEhUTUxJbWFnZUVsZW1lbnQsIEhUTUxDYW52YXNFbGVtZW50LCBIVE1MVmlkZW9FbGVtZW50ICovXG4gIF9kZWR1Y2VJbWFnZVNpemUoaW1hZ2UpIHtcbiAgICBpZiAodHlwZW9mIEltYWdlRGF0YSAhPT0gJ3VuZGVmaW5lZCcgJiYgaW1hZ2UgaW5zdGFuY2VvZiBJbWFnZURhdGEpIHtcbiAgICAgIHJldHVybiB7d2lkdGg6IGltYWdlLndpZHRoLCBoZWlnaHQ6IGltYWdlLmhlaWdodH07XG4gICAgfSBlbHNlIGlmICh0eXBlb2YgSFRNTEltYWdlRWxlbWVudCAhPT0gJ3VuZGVmaW5lZCcgJiZcbiAgICAgIGltYWdlIGluc3RhbmNlb2YgSFRNTEltYWdlRWxlbWVudCkge1xuICAgICAgcmV0dXJuIHt3aWR0aDogaW1hZ2UubmF0dXJhbFdpZHRoLCBoZWlnaHQ6IGltYWdlLm5hdHVyYWxIZWlnaHR9O1xuICAgIH0gZWxzZSBpZiAodHlwZW9mIEhUTUxDYW52YXNFbGVtZW50ICE9PSAndW5kZWZpbmVkJyAmJlxuICAgICAgaW1hZ2UgaW5zdGFuY2VvZiBIVE1MQ2FudmFzRWxlbWVudCkge1xuICAgICAgcmV0dXJuIHt3aWR0aDogaW1hZ2Uud2lkdGgsIGhlaWdodDogaW1hZ2UuaGVpZ2h0fTtcbiAgICB9IGVsc2UgaWYgKHR5cGVvZiBIVE1MVmlkZW9FbGVtZW50ICE9PSAndW5kZWZpbmVkJyAmJlxuICAgICAgaW1hZ2UgaW5zdGFuY2VvZiBIVE1MVmlkZW9FbGVtZW50KSB7XG4gICAgICByZXR1cm4ge3dpZHRoOiBpbWFnZS52aWRlb1dpZHRoLCBoZWlnaHQ6IGltYWdlLnZpZGVvSGVpZ2h0fTtcbiAgICB9XG4gICAgdGhyb3cgbmV3IEVycm9yKCdVbmtub3duIGltYWdlIGRhdGEgZm9ybWF0LiBGYWlsZWQgdG8gZGVkdWNlIGltYWdlIHNpemUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBCYXRjaCB1cGRhdGUgcGl4ZWwgc3RvcmFnZSBtb2Rlc1xuICAgKiBAcGFyYW0ge0dMaW50fSBwYWNrQWxpZ25tZW50IC0gUGFja2luZyBvZiBwaXhlbCBkYXRhIGluIG1lbW9yeSAoMSwyLDQsOClcbiAgICogQHBhcmFtIHtHTGludH0gdW5wYWNrQWxpZ25tZW50IC0gVW5wYWNraW5nIHBpeGVsIGRhdGEgZnJvbSBtZW1vcnkoMSwyLDQsOClcbiAgICogQHBhcmFtIHtHTGJvb2xlYW59IHVucGFja0ZsaXBZIC0gIEZsaXAgc291cmNlIGRhdGEgYWxvbmcgaXRzIHZlcnRpY2FsIGF4aXNcbiAgICogQHBhcmFtIHtHTGJvb2xlYW59IHVucGFja1ByZW11bHRpcGx5QWxwaGEgLVxuICAgKiAgIE11bHRpcGxpZXMgdGhlIGFscGhhIGNoYW5uZWwgaW50byB0aGUgb3RoZXIgY29sb3IgY2hhbm5lbHNcbiAgICogQHBhcmFtIHtHTGVudW19IHVucGFja0NvbG9yc3BhY2VDb252ZXJzaW9uIC1cbiAgICogICBEZWZhdWx0IGNvbG9yIHNwYWNlIGNvbnZlcnNpb24gb3Igbm8gY29sb3Igc3BhY2UgY29udmVyc2lvbi5cbiAgICpcbiAgICogQHBhcmFtIHtHTGludH0gcGFja1Jvd0xlbmd0aCAtXG4gICAqICBOdW1iZXIgb2YgcGl4ZWxzIGluIGEgcm93LlxuICAgKiBAcGFyYW0ge30gcGFja1NraXBQaXhlbHMgLVxuICAgKiAgIE51bWJlciBvZiBwaXhlbHMgc2tpcHBlZCBiZWZvcmUgdGhlIGZpcnN0IHBpeGVsIGlzIHdyaXR0ZW4gaW50byBtZW1vcnkuXG4gICAqIEBwYXJhbSB7fSBwYWNrU2tpcFJvd3MgLVxuICAgKiAgIE51bWJlciBvZiByb3dzIG9mIHBpeGVscyBza2lwcGVkIGJlZm9yZSBmaXJzdCBwaXhlbCBpcyB3cml0dGVuIHRvIG1lbW9yeS5cbiAgICogQHBhcmFtIHt9IHVucGFja1Jvd0xlbmd0aCAtXG4gICAqICAgTnVtYmVyIG9mIHBpeGVscyBpbiBhIHJvdy5cbiAgICogQHBhcmFtIHt9IHVucGFja0ltYWdlSGVpZ2h0IC1cbiAgICogICBJbWFnZSBoZWlnaHQgdXNlZCBmb3IgcmVhZGluZyBwaXhlbCBkYXRhIGZyb20gbWVtb3J5XG4gICAqIEBwYXJhbSB7fSB1bnBhY2tTa2lwUGl4ZWxzIC1cbiAgICogICBOdW1iZXIgb2YgcGl4ZWwgaW1hZ2VzIHNraXBwZWQgYmVmb3JlIGZpcnN0IHBpeGVsIGlzIHJlYWQgZnJvbSBtZW1vcnlcbiAgICogQHBhcmFtIHt9IHVucGFja1NraXBSb3dzIC1cbiAgICogICBOdW1iZXIgb2Ygcm93cyBvZiBwaXhlbHMgc2tpcHBlZCBiZWZvcmUgZmlyc3QgcGl4ZWwgaXMgcmVhZCBmcm9tIG1lbW9yeVxuICAgKiBAcGFyYW0ge30gdW5wYWNrU2tpcEltYWdlcyAtXG4gICAqICAgTnVtYmVyIG9mIHBpeGVsIGltYWdlcyBza2lwcGVkIGJlZm9yZSBmaXJzdCBwaXhlbCBpcyByZWFkIGZyb20gbWVtb3J5XG4gICAqL1xuICAvKiBlc2xpbnQtZGlzYWJsZSBjb21wbGV4aXR5LCBtYXgtc3RhdGVtZW50cyAqL1xuICBzZXRQaXhlbFN0b3JhZ2VNb2Rlcyh7XG4gICAgcGFja0FsaWdubWVudCxcbiAgICB1bnBhY2tBbGlnbm1lbnQsXG4gICAgdW5wYWNrRmxpcFksXG4gICAgdW5wYWNrUHJlbXVsdGlwbHlBbHBoYSxcbiAgICB1bnBhY2tDb2xvcnNwYWNlQ29udmVyc2lvbixcbiAgICAvLyBXRUJHTDJcbiAgICBwYWNrUm93TGVuZ3RoLFxuICAgIHBhY2tTa2lwUGl4ZWxzLFxuICAgIHBhY2tTa2lwUm93cyxcbiAgICB1bnBhY2tSb3dMZW5ndGgsXG4gICAgdW5wYWNrSW1hZ2VIZWlnaHQsXG4gICAgdW5wYWNrU2tpcFBpeGVscyxcbiAgICB1bnBhY2tTa2lwUm93cyxcbiAgICB1bnBhY2tTa2lwSW1hZ2VzXG4gIH0gPSB7fSkge1xuICAgIGNvbnN0IHtnbH0gPSB0aGlzO1xuXG4gICAgZ2wuYmluZFRleHR1cmUodGhpcy50YXJnZXQsIHRoaXMuaGFuZGxlKTtcblxuICAgIGlmIChwYWNrQWxpZ25tZW50KSB7XG4gICAgICBnbC5waXhlbFN0b3JlaShnbC5QQUNLX0FMSUdOTUVOVCwgcGFja0FsaWdubWVudCk7XG4gICAgfVxuICAgIGlmICh1bnBhY2tBbGlnbm1lbnQpIHtcbiAgICAgIGdsLnBpeGVsU3RvcmVpKGdsLlVOUEFDS19BTElHTk1FTlQsIHVucGFja0FsaWdubWVudCk7XG4gICAgfVxuICAgIGlmICh1bnBhY2tGbGlwWSkge1xuICAgICAgZ2wucGl4ZWxTdG9yZWkoZ2wuVU5QQUNLX0ZMSVBfWV9XRUJHTCwgdW5wYWNrRmxpcFkpO1xuICAgIH1cbiAgICBpZiAodW5wYWNrUHJlbXVsdGlwbHlBbHBoYSkge1xuICAgICAgZ2wucGl4ZWxTdG9yZWkoZ2wuVU5QQUNLX1BSRU1VTFRJUExZX0FMUEhBX1dFQkdMLCB1bnBhY2tQcmVtdWx0aXBseUFscGhhKTtcbiAgICB9XG4gICAgaWYgKHVucGFja0NvbG9yc3BhY2VDb252ZXJzaW9uKSB7XG4gICAgICBnbC5waXhlbFN0b3JlaShnbC5VTlBBQ0tfQ09MT1JTUEFDRV9DT05WRVJTSU9OX1dFQkdMLFxuICAgICAgICB1bnBhY2tDb2xvcnNwYWNlQ29udmVyc2lvbik7XG4gICAgfVxuXG4gICAgLy8gV0VCR0wyXG4gICAgaWYgKHBhY2tSb3dMZW5ndGgpIHtcbiAgICAgIGdsLnBpeGVsU3RvcmVpKGdsLlBBQ0tfUk9XX0xFTkdUSCwgcGFja1Jvd0xlbmd0aCk7XG4gICAgfVxuICAgIGlmIChwYWNrU2tpcFBpeGVscykge1xuICAgICAgZ2wucGl4ZWxTdG9yZWkoZ2wuUEFDS19TS0lQX1BJWEVMUywgcGFja1NraXBQaXhlbHMpO1xuICAgIH1cbiAgICBpZiAocGFja1NraXBSb3dzKSB7XG4gICAgICBnbC5waXhlbFN0b3JlaShnbC5QQUNLX1NLSVBfUk9XUywgcGFja1NraXBSb3dzKTtcbiAgICB9XG4gICAgaWYgKHVucGFja1Jvd0xlbmd0aCkge1xuICAgICAgZ2wucGl4ZWxTdG9yZWkoZ2wuVU5QQUNLX1JPV19MRU5HVEgsIHVucGFja1Jvd0xlbmd0aCk7XG4gICAgfVxuICAgIGlmICh1bnBhY2tJbWFnZUhlaWdodCkge1xuICAgICAgZ2wucGl4ZWxTdG9yZWkoZ2wuVU5QQUNLX0lNQUdFX0hFSUdIVCwgdW5wYWNrSW1hZ2VIZWlnaHQpO1xuICAgIH1cbiAgICBpZiAodW5wYWNrU2tpcFBpeGVscykge1xuICAgICAgZ2wucGl4ZWxTdG9yZWkoZ2wuVU5QQUNLX1NLSVBfUElYRUxTLCB1bnBhY2tTa2lwUGl4ZWxzKTtcbiAgICB9XG4gICAgaWYgKHVucGFja1NraXBSb3dzKSB7XG4gICAgICBnbC5waXhlbFN0b3JlaShnbC5VTlBBQ0tfU0tJUF9ST1dTLCB1bnBhY2tTa2lwUm93cyk7XG4gICAgfVxuICAgIGlmICh1bnBhY2tTa2lwSW1hZ2VzKSB7XG4gICAgICBnbC5waXhlbFN0b3JlaShnbC5VTlBBQ0tfU0tJUF9JTUFHRVMsIHVucGFja1NraXBJbWFnZXMpO1xuICAgIH1cblxuICAgIGdsLmJpbmRUZXh0dXJlKHRoaXMudGFyZ2V0LCBudWxsKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuICAvKiBlc2xpbnQtZW5hYmxlIGNvbXBsZXhpdHksIG1heC1zdGF0ZW1lbnRzICovXG5cbiAgLyoqXG4gICAqIEJhdGNoIHVwZGF0ZSBzYW1wbGVyIHNldHRpbmdzXG4gICAqXG4gICAqIEBwYXJhbSB7R0xlbnVtfSBtYWdGaWx0ZXIgLSB0ZXh0dXJlIG1hZ25pZmljYXRpb24gZmlsdGVyLlxuICAgKiBAcGFyYW0ge0dMZW51bX0gbWluRmlsdGVyIC0gdGV4dHVyZSBtaW5pZmljYXRpb24gZmlsdGVyXG4gICAqIEBwYXJhbSB7R0xlbnVtfSB3cmFwUyAtIHRleHR1cmUgd3JhcHBpbmcgZnVuY3Rpb24gZm9yIHRleHR1cmUgY29vcmRpbmF0ZSBzLlxuICAgKiBAcGFyYW0ge0dMZW51bX0gd3JhcFQgLSB0ZXh0dXJlIHdyYXBwaW5nIGZ1bmN0aW9uIGZvciB0ZXh0dXJlIGNvb3JkaW5hdGUgdC5cbiAgICogV0VCR0wyIG9ubHk6XG4gICAqIEBwYXJhbSB7R0xlbnVtfSB3cmFwUiAtIHRleHR1cmUgd3JhcHBpbmcgZnVuY3Rpb24gZm9yIHRleHR1cmUgY29vcmRpbmF0ZSByLlxuICAgKiBAcGFyYW0ge0dMZW51bX0gY29tcGFyZUZ1bmMgLSB0ZXh0dXJlIGNvbXBhcmlzb24gZnVuY3Rpb24uXG4gICAqIEBwYXJhbSB7R0xlbnVtfSBjb21wYXJlTW9kZSAtIHRleHR1cmUgY29tcGFyaXNvbiBtb2RlLlxuICAgKiBAcGFyYW0ge0dMZmxvYXR9IG1pbkxPRCAtIG1pbmltdW0gbGV2ZWwtb2YtZGV0YWlsIHZhbHVlLlxuICAgKiBAcGFyYW0ge0dMZmxvYXR9IG1heExPRCAtIG1heGltdW0gbGV2ZWwtb2YtZGV0YWlsIHZhbHVlLlxuICAgKiBAcGFyYW0ge0dMZmxvYXR9IGJhc2VMZXZlbCAtIFRleHR1cmUgbWlwbWFwIGxldmVsXG4gICAqIEBwYXJhbSB7R0xmbG9hdH0gbWF4TGV2ZWwgLSBNYXhpbXVtIHRleHR1cmUgbWlwbWFwIGFycmF5IGxldmVsXG4gICAqL1xuICAvKiBlc2xpbnQtZGlzYWJsZSBjb21wbGV4aXR5LCBtYXgtc3RhdGVtZW50cyAqL1xuICBzZXRQYXJhbWV0ZXJzKHtcbiAgICBtYWdGaWx0ZXIsXG4gICAgbWluRmlsdGVyLFxuICAgIHdyYXBTLFxuICAgIHdyYXBULFxuICAgIC8vIFdFQkdMMlxuICAgIHdyYXBSLFxuICAgIGJhc2VMZXZlbCxcbiAgICBtYXhMZXZlbCxcbiAgICBtaW5MT0QsXG4gICAgbWF4TE9ELFxuICAgIGNvbXBhcmVGdW5jLFxuICAgIGNvbXBhcmVNb2RlXG4gIH0pIHtcbiAgICBjb25zdCB7Z2x9ID0gdGhpcztcbiAgICBnbC5iaW5kVGV4dHVyZSh0aGlzLnRhcmdldCwgdGhpcy5oYW5kbGUpO1xuXG4gICAgaWYgKG1hZ0ZpbHRlcikge1xuICAgICAgZ2wudGV4UGFyYW1ldGVyaSh0aGlzLnRhcmdldCwgZ2wuVEVYVFVSRV9NQUdfRklMVEVSLCBtYWdGaWx0ZXIpO1xuICAgIH1cbiAgICBpZiAobWluRmlsdGVyKSB7XG4gICAgICBnbC50ZXhQYXJhbWV0ZXJpKHRoaXMudGFyZ2V0LCBnbC5URVhUVVJFX01JTl9GSUxURVIsIG1pbkZpbHRlcik7XG4gICAgfVxuICAgIGlmICh3cmFwUykge1xuICAgICAgZ2wudGV4UGFyYW1ldGVyaSh0aGlzLnRhcmdldCwgZ2wuVEVYVFVSRV9XUkFQX1MsIHdyYXBTKTtcbiAgICB9XG4gICAgaWYgKHdyYXBUKSB7XG4gICAgICBnbC50ZXhQYXJhbWV0ZXJpKHRoaXMudGFyZ2V0LCBnbC5URVhUVVJFX1dSQVBfVCwgd3JhcFQpO1xuICAgIH1cbiAgICAvLyBXRUJHTDJcbiAgICBpZiAod3JhcFIpIHtcbiAgICAgIGdsLnRleFBhcmFtZXRlcmkodGhpcy50YXJnZXQsIGdsLlRFWFRVUkVfV1JBUF9SLCB3cmFwUik7XG4gICAgfVxuICAgIGlmIChiYXNlTGV2ZWwpIHtcbiAgICAgIGdsLnRleFBhcmFtZXRlcmkodGhpcy50YXJnZXQsIGdsLlRFWFRVUkVfQkFTRV9MRVZFTCwgYmFzZUxldmVsKTtcbiAgICB9XG4gICAgaWYgKG1heExldmVsKSB7XG4gICAgICBnbC50ZXhQYXJhbWV0ZXJpKHRoaXMudGFyZ2V0LCBnbC5URVhUVVJFX01BWF9MRVZFTCwgbWF4TGV2ZWwpO1xuICAgIH1cbiAgICBpZiAoY29tcGFyZUZ1bmMpIHtcbiAgICAgIGdsLnRleFBhcmFtZXRlcmkodGhpcy50YXJnZXQsIGdsLlRFWFRVUkVfQ09NUEFSRV9GVU5DLCBjb21wYXJlRnVuYyk7XG4gICAgfVxuICAgIGlmIChjb21wYXJlTW9kZSkge1xuICAgICAgZ2wudGV4UGFyYW1ldGVyaSh0aGlzLnRhcmdldCwgZ2wuVEVYVFVSRV9DT01QQVJFX01PREUsIGNvbXBhcmVNb2RlKTtcbiAgICB9XG4gICAgaWYgKG1pbkxPRCkge1xuICAgICAgZ2wudGV4UGFyYW1ldGVyZih0aGlzLnRhcmdldCwgZ2wuVEVYVFVSRV9NSU5fTE9ELCBtaW5MT0QpO1xuICAgIH1cbiAgICBpZiAobWF4TE9EKSB7XG4gICAgICBnbC50ZXhQYXJhbWV0ZXJmKHRoaXMudGFyZ2V0LCBnbC5URVhUVVJFX01BWF9MT0QsIG1heExPRCk7XG4gICAgfVxuXG4gICAgZ2wuYmluZFRleHR1cmUodGhpcy50YXJnZXQsIG51bGwpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG4gIC8qIGVzbGludC1lbmFibGUgY29tcGxleGl0eSwgbWF4LXN0YXRlbWVudHMgKi9cblxuICBnZXRQYXJhbWV0ZXJzKCkge1xuICAgIGNvbnN0IHtnbH0gPSB0aGlzO1xuICAgIGdsLmJpbmRUZXh0dXJlKHRoaXMudGFyZ2V0LCB0aGlzLmhhbmRsZSk7XG4gICAgY29uc3Qgd2ViZ2xQYXJhbXMgPSB7XG4gICAgICBtYWdGaWx0ZXI6IGdsLmdldFRleFBhcmFtZXRlcih0aGlzLnRhcmdldCwgZ2wuVEVYVFVSRV9NQUdfRklMVEVSKSxcbiAgICAgIG1pbkZpbHRlcjogZ2wuZ2V0VGV4UGFyYW1ldGVyKHRoaXMudGFyZ2V0LCBnbC5URVhUVVJFX01JTl9GSUxURVIpLFxuICAgICAgd3JhcFM6IGdsLmdldFRleFBhcmFtZXRlcih0aGlzLnRhcmdldCwgZ2wuVEVYVFVSRV9XUkFQX1MpLFxuICAgICAgd3JhcFQ6IGdsLmdldFRleFBhcmFtZXRlcih0aGlzLnRhcmdldCwgZ2wuVEVYVFVSRV9XUkFQX1QpXG4gICAgfTtcbiAgICBnbC5iaW5kVGV4dHVyZSh0aGlzLnRhcmdldCwgbnVsbCk7XG4gICAgcmV0dXJuIHdlYmdsUGFyYW1zO1xuICB9XG5cbiAgLy8gRGVwcmVjYXRlZCBtZXRob2RzXG5cbiAgaW1hZ2UyRCh7XG4gICAgcGl4ZWxzLFxuICAgIGZvcm1hdCA9IEdMLlJHQkEsXG4gICAgdHlwZSA9IEdMLlVOU0lHTkVEX0JZVEVcbiAgfSkge1xuICAgIC8vIFRPRE8gLSBXZWJHTDIgY2hlY2s/XG4gICAgaWYgKHR5cGUgPT09IEdMLkZMT0FUICYmICF0aGlzLmhhc0Zsb2F0VGV4dHVyZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdmbG9hdGluZyBwb2ludCB0ZXh0dXJlcyBhcmUgbm90IHN1cHBvcnRlZC4nKTtcbiAgICB9XG5cbiAgICB0aGlzLmdsLmJpbmRUZXh0dXJlKHRoaXMudGFyZ2V0LCB0aGlzLmhhbmRsZSk7XG4gICAgdGhpcy5nbC50ZXhJbWFnZTJEKEdMLlRFWFRVUkVfMkQsIDAsIGZvcm1hdCwgZm9ybWF0LCB0eXBlLCBwaXhlbHMpO1xuICAgIHRoaXMuZ2wuYmluZFRleHR1cmUodGhpcy50YXJnZXQsIG51bGwpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgdXBkYXRlKG9wdHMpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ1RleHR1cmUudXBkYXRlKCkgaXMgZGVwcmVjYXRlZCgpJyk7XG4gIH1cbn1cbiJdfQ==