'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }(); // WebGL2 VertexArray Objects Helper


var _webglChecks = require('../webgl/webgl-checks');

var _queryManager = require('./helpers/query-manager');

var _queryManager2 = _interopRequireDefault(_queryManager);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

/* eslint-disable no-multi-spaces */
var GL_QUERY_COUNTER_BITS_EXT = 0x8864;
var GL_QUERY_RESULT_EXT = 0x8866;
var GL_QUERY_RESULT_AVAILABLE_EXT = 0x8867;
var GL_TIME_ELAPSED_EXT = 0x88BF;
var GL_TIMESTAMP_EXT = 0x8E28;
var GL_GPU_DISJOINT_EXT = 0x8FBB;
/* eslint-enable no-multi-spaces */

var noop = function noop(x) {
  return x;
};

var ERR_GPU_DISJOINT = 'Disjoint GPU operation invalidated timer queries';
var ERR_TIMER_QUERY_NOT_SUPPORTED = 'Timer queries require "EXT_disjoint_timer_query" extension';

var TimerQuery = function () {
  _createClass(TimerQuery, null, [{
    key: 'isSupported',

    /**
     * Returns true if TimerQuery is supported by the WebGL implementation
     * (depends on the EXT_disjoint_timer_query extension)/
     * Can also check whether timestamp queries are available.
     *
     * @param {WebGLRenderingContext} gl - gl context
     * @param {Object} opts= - options
     * @param {Object} opts.requireTimestamps=false -
     *   If true, checks if timestamps are supported
     * @return {Boolean} - TimerQueries are supported with specified configuration
     */
    value: function isSupported(gl) {
      var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
          _ref$requireTimestamp = _ref.requireTimestamps,
          requireTimestamps = _ref$requireTimestamp === undefined ? false : _ref$requireTimestamp;

      (0, _webglChecks.assertWebGLContext)(gl);
      var ext = gl.getExtension('EXT_disjoint_timer_query');
      var queryCounterBits = ext ? ext.getQueryEXT(GL_TIMESTAMP_EXT, GL_QUERY_COUNTER_BITS_EXT) : 0;
      var timestampsSupported = queryCounterBits > 0;
      return Boolean(ext) && (!requireTimestamps || timestampsSupported);
    }

    /**
     * @classdesc
     * Provides a way to measure the duration of a set of GL commands,
     * without stalling the rendering pipeline.
     *
     * Exposes a `promise` member that tracks the state of the query
     * when `poll` is used to update queries.
     *
     * @example
        const timerQuery = new TimerQuery({
          onComplete: timestamp => console.log(timestamp)
          onError: error => console.warn(error)
        });
         timerQuery.begin();
         // Issue GPU calls
         timerQuery.end();
         // Poll for resolved queries
        requestAnimationFrame(() => TimerQuery.poll(gl))
     *
     * Remarks:
     * - On Chrome, go to chrome:flags and enable "WebGL Draft Extensions"
     *
     * - For full functionality, TimerQuery depends on a `poll` function being
     *   called regularly. When this is done, completed queries will be
     *   automatically detected and any callbacks called.
     *
     * - TimerQuery instance creation will always succeed, even when the required
     *   extension is not supported. Instead any issued queries will fail
     *   immediately. This allows applications to unconditionally use TimerQueries
     *   without adding logic to check whether they are supported; the
     *   difference being that the `onComplete` callback never gets called,
     *   (the `onError` callback, if supplied, will be called instead).
     *
     * - Even when supported, timer queries can fail whenever a change in the
     *   GPU occurs that will make the values returned by this extension unusable
     *   for performance metrics. Power conservation might cause the GPU to go to
     *   sleep at the lower levers. TimerQuery will detect this condition and
     *   fail any outstanding queries (which calls the `onError` function,
     *   if supplied).
     *
     * @class
     * @param {WebGLRenderingContext} gl - gl context
     * @param {Object} opts - options
     * @param {Function} opts.onComplete - called with a timestamp.
     *   Specifying this parameter causes a timestamp query to be initiated
     * @param {Function} opts.onComplete - called with a timestamp.
     *   Specifying this parameter causes a timestamp query to be initiated
     */

  }]);

  function TimerQuery(gl) {
    var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
        _ref2$onComplete = _ref2.onComplete,
        onComplete = _ref2$onComplete === undefined ? noop : _ref2$onComplete,
        _ref2$onError = _ref2.onError,
        onError = _ref2$onError === undefined ? noop : _ref2$onError;

    _classCallCheck(this, TimerQuery);

    (0, _webglChecks.assertWebGLContext)(gl);

    this.gl = gl;
    this.ext = this.gl.getExtension('EXT_disjoint_timer_query');
    this.handle = this.ext ? this.ext.createQueryEXT() : null;
    this.target = null;
    this.userData = {};

    this.onComplete = onComplete;
    this.onError = onError;

    // query manager needs a promise field
    this.promise = null;

    Object.seal(this);
  }

  /**
   * Destroys the WebGL object
   * Rejects any pending query
   *
   * @return {TimerQuery} - returns itself, to enable chaining of calls.
   */


  _createClass(TimerQuery, [{
    key: 'delete',
    value: function _delete() {
      if (this.handle) {
        _queryManager2.default.deleteQuery(this);
        this.ext.deleteQueryEXT(this.handle);
        (0, _webglChecks.glCheckError)(this.gl);
        this.handle = null;
      }
      return this;
    }

    /**
     * Measures GPU time delta between this call and a matching `end` call in the
     * GPU instruction stream.
     *
     * Remarks:
     * - Due to OpenGL API limitations, after calling `begin()` on one TimerQuery
     *   instance, `end()` must be called on that same instance before
     *   calling `begin()` on another query. While there can be multiple
     *   outstanding queries representing disjoint `begin()`/`end()` intervals.
     *   It is not possible to interleave or overlap `begin` and `end` calls.
     *
     * - Triggering a new query when a TimerQuery is already tracking an
     *   unresolved query causes that query to be cancelled.
     *
     * @return {TimerQuery} - returns itself, to enable chaining of calls.
     */

  }, {
    key: 'begin',
    value: function begin() {
      _queryManager2.default.beginQuery(this, this.onComplete, this.onError);
      if (this.ext) {
        this.target = GL_TIME_ELAPSED_EXT;
        this.ext.beginQueryEXT(this.target, this.handle);
      } else {
        _queryManager2.default.rejectQuery(this, ERR_TIMER_QUERY_NOT_SUPPORTED);
      }
      return this;
    }

    /**
     * Inserts a query end marker into the GPU instruction stream.
     * Can be called multiple times.
     *
     * @return {TimerQuery} - returns itself, to enable chaining of calls.
     */

  }, {
    key: 'end',
    value: function end() {
      // Note: calling end does not affect the pending promise
      if (this.target) {
        this.ext.endQueryEXT(this.target);
        this.target = null;
      }
      return this;
    }

    /**
     * Generates a GPU time stamp when the GPU instruction stream reaches
     * this instruction.
     * To measure time deltas, two queries are needed.
     *
     * Remarks:
     * - timestamp() queries may not be available even when the timer query
     *   extension is. See TimeQuery.isSupported() flags.
     *
     * - Triggering a new query when a TimerQuery is already tracking an
     *   unresolved query causes that query to be cancelled.
     *
     * @return {TimerQuery} - returns itself, to enable chaining of calls.
     */

  }, {
    key: 'timestamp',
    value: function timestamp() {
      _queryManager2.default.beginQuery(this, this.onComplete, this.onError);
      if (this.ext) {
        this.ext.queryCounterEXT(this.handle, GL_TIMESTAMP_EXT);
      } else {
        _queryManager2.default.rejectQuery(this, ERR_TIMER_QUERY_NOT_SUPPORTED);
      }
      return this;
    }

    /**
     * Cancels a pending query
     * Note - Cancel's the promise and calls end on the current query if needed.
     *
     * @return {TimerQuery} - returns itself, to enable chaining of calls.
     */

  }, {
    key: 'cancel',
    value: function cancel() {
      this.end();
      _queryManager2.default.cancelQuery(this);
      return this;
    }

    /**
     * @return {Boolean} - true if query result is available
     */

  }, {
    key: 'isResultAvailable',
    value: function isResultAvailable() {
      return this.ext && this.ext.getQueryObjectEXT(this.handle, GL_QUERY_RESULT_AVAILABLE_EXT);
    }

    /**
     * Returns the query result, converted to milliseconds to match
     * JavaScript conventions.
     *
     * @return {Number} - measured time or timestamp, in milliseconds
     */

  }, {
    key: 'getResult',
    value: function getResult() {
      return this.ext ? this.ext.getQueryObjectEXT(this.handle, GL_QUERY_RESULT_EXT) / 1e6 : 0;
    }
  }]);

  return TimerQuery;
}();

// NOTE: This call lets the queryManager know how to detect disjoint GPU state
// It will check dsjoint state on polls and before adding a new query
// and reject any outstanding TimerQueries with our supplied error message.


exports.default = TimerQuery;
_queryManager2.default.setInvalidator({
  queryType: TimerQuery,
  errorMessage: ERR_GPU_DISJOINT,
  // Note: Querying the disjoint state resets it
  checkInvalid: function checkInvalid(gl) {
    return gl.getParameter(GL_GPU_DISJOINT_EXT);
  }
});
//# sourceMappingURL=data:application/json;base64,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