'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _UNIFORM_BASE_DESCRIP;

exports.parseUniformName = parseUniformName;
exports.getUniformSetter = getUniformSetter;
exports.checkUniformValues = checkUniformValues;
exports.getUniformsTable = getUniformsTable;

var _webgl = require('./webgl');

var _texture = require('./texture');

var _texture2 = _interopRequireDefault(_texture);

var _utils = require('../utils');

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

// TODO - use tables to reduce complexity of method below
/* eslint-disable max-len */
var UNIFORM_BASE_DESCRIPTORS = (_UNIFORM_BASE_DESCRIP = {}, _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.FLOAT, { function: 'uniform1f', type: Float32Array }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.INT, { function: 'uniform1i', type: Uint16Array }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.BOOL, { function: 'uniform1i', type: Uint16Array }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.FLOAT_VEC2, { function: 'uniform2fv', type: Float32Array, elements: 2 }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.FLOAT_VEC3, { function: 'uniform3fv', type: Float32Array, elements: 3 }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.FLOAT_VEC4, { function: 'uniform4fv', type: Float32Array, elements: 4 }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.INT_VEC2, { function: 'uniform2iv', type: Uint16Array, elements: 2 }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.INT_VEC3, { function: 'uniform3iv', type: Uint16Array, elements: 3 }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.INT_VEC4, { function: 'uniform4iv', type: Uint16Array, elements: 4 }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.BOOL_VEC2, { function: 'uniform2iv', type: Uint16Array, elements: 2 }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.BOOL_VEC3, { function: 'uniform3fv', type: Uint16Array, elements: 3 }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.BOOL_VEC4, { function: 'uniform4iv', type: Uint16Array, elements: 4 }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.FLOAT_MAT2, { function: 'uniformMatrix2fv', type: Float32Array, matrix: true, elements: 4 }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.FLOAT_MAT3, { mfunction: 'uniformMatrix3fv', type: Float32Array, matrix: true, elements: 9 }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.FLOAT_MAT4, { function: 'uniformMatrix4fv', type: Float32Array, matrix: true, elements: 16 }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.SAMPLER_2D, { function: 'uniform1i', type: Uint16Array, texture: true }), _defineProperty(_UNIFORM_BASE_DESCRIP, _webgl.GL.SAMPLER_CUBE, { function: 'uniform1i', type: Uint16Array, texture: true }), _UNIFORM_BASE_DESCRIP);
/* eslint-enable max-len */

function parseUniformName(name) {
  // name = name[name.length - 1] === ']' ?
  // name.substr(0, name.length - 3) : name;

  // if array name then clean the array brackets
  var UNIFORM_NAME_REGEXP = /([^\[]*)(\[[0-9]+\])?/;
  var matches = name.match(UNIFORM_NAME_REGEXP);
  if (!matches || matches.length < 2) {
    throw new Error('Failed to parse GLSL uniform name ' + name);
  }

  return {
    name: matches[1],
    length: matches[2] || 1,
    isArray: Boolean(matches[2])
  };
}

// Returns a Magic Uniform Setter
/* eslint-disable complexity */
function getUniformSetter(gl, location, info) {
  var descriptor = UNIFORM_BASE_DESCRIPTORS[info.type];
  if (!descriptor) {
    throw new Error('Unknown GLSL uniform type ' + info.type);
  }

  var glFunction = gl[descriptor.function].bind(gl);
  var TypedArray = descriptor.type;

  // How many data elements does app need to provide
  var flatArrayLength = info.size * (descriptor.elements || 1);

  // console.log('getSetter', location, info, flatArrayLength);

  // Set a uniform array
  var setter = void 0;
  if (flatArrayLength > 1) {
    setter = function setter(val) {
      if (!(val instanceof TypedArray)) {
        var typedArray = new TypedArray(flatArrayLength);
        typedArray.set(val);
        val = typedArray;
      }
      (0, _assert2.default)(val.length === flatArrayLength);
      if (descriptor.matrix) {
        // Second param: whether to transpose the matrix. Must be false.
        glFunction(location, false, val);
      } else {
        glFunction(location, val);
      }
    };
  } else {
    setter = function setter(val) {
      return glFunction(location, val);
    };
  }

  // Set a primitive-valued uniform
  return setter;
}

// Basic checks of uniform values without knowledge of program
// To facilitate early detection of e.g. undefined values in JavaScript
function checkUniformValues(uniforms, source) {
  for (var uniformName in uniforms) {
    var value = uniforms[uniformName];
    if (!checkUniformValue(value)) {
      // Add space to source
      source = source ? source + ' ' : '';
      /* eslint-disable no-console */
      /* global console */
      // Value could be unprintable so write the object on console
      console.error(source + ' Bad uniform ' + uniformName, value);
      /* eslint-enable no-console */
      throw new Error(source + ' Bad uniform ' + uniformName);
    }
  }
  return true;
}

function checkUniformValue(value) {
  var ok = true;

  // Test for texture (for sampler uniforms)
  // WebGL2: if (value instanceof Texture || value instanceof Sampler) {
  if (value instanceof _texture2.default) {
    ok = true;
    // Check that every element in array is a number, and at least 1 element
  } else if (Array.isArray(value)) {
    var _iteratorNormalCompletion = true;
    var _didIteratorError = false;
    var _iteratorError = undefined;

    try {
      for (var _iterator = value[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
        var element = _step.value;

        if (!isFinite(element)) {
          ok = false;
        }
      }
    } catch (err) {
      _didIteratorError = true;
      _iteratorError = err;
    } finally {
      try {
        if (!_iteratorNormalCompletion && _iterator.return) {
          _iterator.return();
        }
      } finally {
        if (_didIteratorError) {
          throw _iteratorError;
        }
      }
    }

    ok = ok && value.length > 0;
    // Typed arrays can only contain numbers, but check length
  } else if (ArrayBuffer.isView(value)) {
    ok = value.length > 0;
    // Check that single value is a number
  } else if (!isFinite(value)) {
    ok = false;
  }

  return ok;
}

// Prepares a table suitable for console.table
/* eslint-disable max-statements */
function getUniformsTable() {
  var _ref = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
      _ref$header = _ref.header,
      header = _ref$header === undefined ? 'Uniforms' : _ref$header,
      program = _ref.program,
      uniforms = _ref.uniforms;

  (0, _assert2.default)(program);

  var uniformLocations = program._uniformSetters;
  var table = _defineProperty({}, header, {});

  // Add program's provided uniforms
  for (var uniformName in uniformLocations) {
    var uniform = uniforms[uniformName];
    if (uniform !== undefined) {
      table[uniformName] = {
        Type: uniform,
        Value: (0, _utils.formatValue)(uniform)
      };
    }
  }

  // Add program's unprovided uniforms
  for (var _uniformName in uniformLocations) {
    var _uniform = uniforms[_uniformName];
    if (_uniform === undefined) {
      table[_uniformName] = {
        Type: 'NOT PROVIDED',
        Value: 'N/A'
      };
    }
  }

  var unusedTable = {};
  var unusedCount = 0;

  // List any unused uniforms
  for (var _uniformName2 in uniforms) {
    var _uniform2 = uniforms[_uniformName2];
    if (!table[_uniformName2]) {
      unusedCount++;
      unusedTable[_uniformName2] = {
        Type: 'NOT USED: ' + _uniform2,
        Value: (0, _utils.formatValue)(_uniform2)
      };
    }
  }

  return { table: table, unusedTable: unusedTable, unusedCount: unusedCount };
}

/*
  if (vector) {
    switch (type) {
    case GL.FLOAT:
      glFunction = gl.uniform1f;
      break;
    case GL.FLOAT_VEC2:
      glFunction = gl.uniform2fv;
      TypedArray = isArray ? Float32Array : new Float32Array(2);
      break;
    case GL.FLOAT_VEC3:
      glFunction = gl.uniform3fv;
      TypedArray = isArray ? Float32Array : new Float32Array(3);
      break;
    case GL.FLOAT_VEC4:
      glFunction = gl.uniform4fv;
      TypedArray = isArray ? Float32Array : new Float32Array(4);
      break;
    case GL.INT:
    case GL.BOOL:
    case GL.SAMPLER_2D:
    case GL.SAMPLER_CUBE:
      glFunction = gl.uniform1i;
      break;
    case GL.INT_VEC2:
    case GL.BOOL_VEC2:
      glFunction = gl.uniform2iv;
      TypedArray = isArray ? Uint16Array : new Uint16Array(2);
      break;
    case GL.INT_VEC3:
    case GL.BOOL_VEC3:
      glFunction = gl.uniform3iv;
      TypedArray = isArray ? Uint16Array : new Uint16Array(3);
      break;
    case GL.INT_VEC4:
    case GL.BOOL_VEC4:
      glFunction = gl.uniform4iv;
      TypedArray = isArray ? Uint16Array : new Uint16Array(4);
      break;
    case GL.FLOAT_MAT2:
      matrix = true;
      glFunction = gl.uniformMatrix2fv;
      break;
    case GL.FLOAT_MAT3:
      matrix = true;
      glFunction = gl.uniformMatrix3fv;
      break;
    case GL.FLOAT_MAT4:
      matrix = true;
      glFunction = gl.uniformMatrix4fv;
      break;
    default:
      break;
    }
  }
*/
//# sourceMappingURL=data:application/json;base64,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