'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GL_BUFFER_USAGE = exports.GL_TARGETS = exports.GL_DRAW_MODES = exports.GL_INDEX_TYPES = undefined;
exports.isWebGLContext = isWebGLContext;
exports.isWebGL2Context = isWebGL2Context;
exports.assertWebGLContext = assertWebGLContext;
exports.assertWebGL2Context = assertWebGL2Context;
exports.isIndexType = isIndexType;
exports.assertIndexType = assertIndexType;
exports.isDrawMode = isDrawMode;
exports.assertDrawMode = assertDrawMode;
exports.assertArrayTypeMatch = assertArrayTypeMatch;
exports.glGetError = glGetError;
exports.glCheckError = glCheckError;

var _webglTypes = require('./webgl-types');

var _webglConstants = require('./webgl-constants');

var _webglConstants2 = _interopRequireDefault(_webglConstants);

var _webglArrays = require('./webgl-arrays');

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// Helper definitions for validation of webgl parameters
/* eslint-disable no-inline-comments, max-len */
var ERR_CONTEXT = 'Invalid WebGLRenderingContext';
var ERR_WEBGL2 = 'Requires WebGL2';

function isWebGLContext(gl) {
  return gl instanceof _webglTypes.WebGLRenderingContext || gl && gl.ARRAY_BUFFER === 0x8892;
}

function isWebGL2Context(gl) {
  return gl instanceof _webglTypes.WebGL2RenderingContext || gl && gl.TEXTURE_BINDING_3D === 0x806A;
}

function assertWebGLContext(gl) {
  // Need to handle debug context
  (0, _assert2.default)(isWebGLContext(gl), ERR_CONTEXT);
}

function assertWebGL2Context(gl) {
  // Need to handle debug context
  (0, _assert2.default)(isWebGL2Context(gl), ERR_WEBGL2);
}

// INDEX TYPES

// For drawElements, size of indices
var GL_INDEX_TYPES = exports.GL_INDEX_TYPES = [_webglConstants2.default.UNSIGNED_BYTE, _webglConstants2.default.UNSIGNED_SHORT, _webglConstants2.default.UNSIGNED_INT];

function isIndexType(type) {
  return GL_INDEX_TYPES.indexOf(type) !== -1;
}

function assertIndexType(glType, source) {
  (0, _assert2.default)(isIndexType(glType), 'Bad index type ' + (0, _webglConstants.glKey)(glType) + ' ' + source);
}

// DRAW MODES

var GL_DRAW_MODES = exports.GL_DRAW_MODES = [_webglConstants2.default.POINTS, _webglConstants2.default.LINE_STRIP, _webglConstants2.default.LINE_LOOP, _webglConstants2.default.LINES, _webglConstants2.default.TRIANGLE_STRIP, _webglConstants2.default.TRIANGLE_FAN, _webglConstants2.default.TRIANGLES];

function isDrawMode(glMode) {
  return GL_DRAW_MODES.indexOf(glMode) !== -1;
}

function assertDrawMode(glType, source) {
  (0, _assert2.default)(isDrawMode(glType), 'Bad draw mode ' + (0, _webglConstants.glKey)(glType) + ' ' + source);
}

// TARGET TYPES

var GL_TARGETS = exports.GL_TARGETS = [_webglConstants2.default.ARRAY_BUFFER, // vertex attributes (e.g. vertex/texture coords or color)
_webglConstants2.default.ELEMENT_ARRAY_BUFFER, // Buffer used for element indices.
// For WebGL 2 contexts
_webglConstants2.default.COPY_READ_BUFFER, // Buffer for copying from one buffer object to another
_webglConstants2.default.COPY_WRITE_BUFFER, // Buffer for copying from one buffer object to another
_webglConstants2.default.TRANSFORM_FEEDBACK_BUFFER, // Buffer for transform feedback operations
_webglConstants2.default.UNIFORM_BUFFER, // Buffer used for storing uniform blocks
_webglConstants2.default.PIXEL_PACK_BUFFER, // Buffer used for pixel transfer operations
_webglConstants2.default.PIXEL_UNPACK_BUFFER // Buffer used for pixel transfer operations
];

// USAGE TYPES

var GL_BUFFER_USAGE = exports.GL_BUFFER_USAGE = [_webglConstants2.default.STATIC_DRAW, // Buffer used often and not change often. Contents are written to the buffer, but not read.
_webglConstants2.default.DYNAMIC_DRAW, // Buffer used often and change often. Contents are written to the buffer, but not read.
_webglConstants2.default.STREAM_DRAW, // Buffer not used often. Contents are written to the buffer, but not read.
// For WebGL 2 contexts
_webglConstants2.default.STATIC_READ, // Buffer used often and not change often. Contents are read from the buffer, but not written.
_webglConstants2.default.DYNAMIC_READ, // Buffer used often and change often. Contents are read from the buffer, but not written.
_webglConstants2.default.STREAM_READ, // Contents of the buffer are likely to not be used often. Contents are read from the buffer, but not written.
_webglConstants2.default.STATIC_COPY, // Buffer used often and not change often. Contents are neither written or read by the user.
_webglConstants2.default.DYNAMIC_COPY, // Buffer used often and change often. Contents are neither written or read by the user.
_webglConstants2.default.STREAM_COPY // Buffer used often and not change often. Contents are neither written or read by the user.
];

function assertArrayTypeMatch(array, type, source) {
  (0, _assert2.default)(type === (0, _webglArrays.glTypeFromArray)(array), (array.constructor.name || 'Array') + ' ' + ('does not match element type ' + (0, _webglConstants.glKey)(type) + ' ' + source));
}

// GL errors

// Returns an Error representing the Latest webGl error or null
function glGetError(gl) {
  // Loop to ensure all errors are cleared
  var errorStack = [];
  var glError = gl.getError();
  while (glError !== gl.NO_ERROR) {
    errorStack.push(glGetErrorMessage(gl, glError));
    glError = gl.getError();
  }
  return errorStack.length ? new Error(errorStack.join('\n')) : null;
}

function glCheckError(gl) {
  if (gl.debug) {
    var error = glGetError(gl);
    if (error) {
      throw error;
    }
  }
}

function glGetErrorMessage(gl, glError) {
  switch (glError) {
    case _webglConstants2.default.CONTEXT_LOST_WEBGL:
      //  If the WebGL context is lost, this error is returned on the
      // first call to getError. Afterwards and until the context has been
      // restored, it returns gl.NO_ERROR.
      return 'WebGL context lost';
    case _webglConstants2.default.INVALID_ENUM:
      // An unacceptable value has been specified for an enumerated argument.
      return 'WebGL invalid enumerated argument';
    case _webglConstants2.default.INVALID_VALUE:
      // A numeric argument is out of range.
      return 'WebGL invalid value';
    case _webglConstants2.default.INVALID_OPERATION:
      // The specified command is not allowed for the current state.
      return 'WebGL invalid operation';
    case _webglConstants2.default.INVALID_FRAMEBUFFER_OPERATION:
      // The currently bound framebuffer is not framebuffer complete
      // when trying to render to or to read from it.
      return 'WebGL invalid framebuffer operation';
    case _webglConstants2.default.OUT_OF_MEMORY:
      // Not enough memory is left to execute the command.
      return 'WebGL out of memory';
    default:
      return 'WebGL unknown error ' + glError;
  }
}
//# sourceMappingURL=data:application/json;base64,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