'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getShaderName = getShaderName;
exports.formatGLSLCompilerError = formatGLSLCompilerError;

var _webglConstants = require('./webgl-constants');

var _webglConstants2 = _interopRequireDefault(_webglConstants);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// import shaderName from 'glsl-shader-name';
// replace with regexp
function getShaderName(shader) {
  return 'unnamed';
}

/**
 * Formats a GLSL shader compiler error and generates a string
 * showing the source code around the error.
 *
 * From https://github.com/wwwtyro/gl-format-compiler-error (public domain)
 *
 * @param {String} errLog - error log from gl.getShaderInfoLog
 * @param {String} src - original shader source code
 * @param {Number} shaderType - shader type (GL constant)
 * @return {String} - Formatted strings has the error marked inline with src.
 */
/* eslint-disable no-continue, max-statements */
function formatGLSLCompilerError(errLog, src, shaderType) {
  var errorStrings = errLog.split(/\r?\n/);
  var errors = {};

  // Parse the error - note: browser and driver dependent
  for (var i = 0; i < errorStrings.length; i++) {
    var errorString = errorStrings[i];
    if (errorString.length <= 1) {
      continue;
    }
    var lineNo = parseInt(errorString.split(':')[2], 10);
    if (isNaN(lineNo)) {
      return 'Could not parse GLSL compiler error: ' + errLog;
    }
    errors[lineNo] = errorString;
  }

  // Format the error inline with the code
  var message = '';
  var lines = addLineNumbers(src).split(/\r?\n/);

  for (var _i = 0; _i < lines.length; _i++) {
    var line = lines[_i];
    if (!errors[_i + 3] && !errors[_i + 2] && !errors[_i + 1]) {
      continue;
    }
    message += line + '\n';
    if (errors[_i + 1]) {
      var e = errors[_i + 1];
      e = e.substr(e.split(':', 3).join(':').length + 1).trim();
      message += '^^^ ' + e + '\n\n';
    }
  }

  var name = getShaderName(src) || 'unknown name';
  var type = getShaderTypeName(shaderType);
  return 'GLSL error in ' + type + ' shader ' + name + '\n' + message;
}

/**
 * Prepends line numbers to each line of a string.
 * The line numbers will be left-padded with spaces to ensure an
 * aligned layout when rendered using monospace fonts.
 *
 * Adapted from https://github.com/Jam3/add-line-numbers, MIT license
 *
 * @param {String} string - multi-line string to add line numbers to
 * @param {Number} start=1 - number of spaces to add
 * @param {String} delim =': ' - injected between line number and original line
 * @return {String} string - The original string with line numbers added
 */
function addLineNumbers(string) {
  var start = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 1;
  var delim = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : ': ';

  var lines = string.split(/\r?\n/);
  var maxDigits = String(lines.length + start - 1).length;
  return lines.map(function (line, i) {
    var lineNumber = i + start;
    var digits = String(lineNumber).length;
    var prefix = padLeft(lineNumber, maxDigits - digits);
    return prefix + delim + line;
  }).join('\n');
}

/**
 * Pads a string with a number of spaces (space characters) to the left
 * @param {String} string - string to pad
 * @param {Number} digits - number of spaces to add
 * @return {String} string - The padded string
 */
function padLeft(string, digits) {
  var result = '';
  for (var i = 0; i < digits; ++i) {
    result += ' ';
  }
  return '' + result + string;
}

function getShaderTypeName(type) {
  switch (type) {
    case _webglConstants2.default.FRAGMENT_SHADER:
      return 'fragment';
    case _webglConstants2.default.VERTEX_SHADER:
      return 'vertex';
    default:
      return 'unknown type';
  }
}
//# sourceMappingURL=data:application/json;base64,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