"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.blit = blit;
exports.copyToDataUrl = copyToDataUrl;
exports.copyToImage = copyToImage;
exports.copyToTexture = copyToTexture;
exports.readPixelsToArray = readPixelsToArray;
exports.readPixelsToBuffer = readPixelsToBuffer;

var _buffer = _interopRequireDefault(require("./buffer"));

var _framebuffer = _interopRequireDefault(require("./framebuffer"));

var _texture = _interopRequireDefault(require("./texture"));

var _gltools = require("@luma.gl/gltools");

var _typedArrayUtils = require("../webgl-utils/typed-array-utils");

var _formatUtils = require("../webgl-utils/format-utils");

var _textureUtils = require("../webgl-utils/texture-utils");

var _assert = require("../utils/assert");

function readPixelsToArray(source) {
  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  var _options$sourceX = options.sourceX,
      sourceX = _options$sourceX === void 0 ? 0 : _options$sourceX,
      _options$sourceY = options.sourceY,
      sourceY = _options$sourceY === void 0 ? 0 : _options$sourceY,
      _options$sourceFormat = options.sourceFormat,
      sourceFormat = _options$sourceFormat === void 0 ? 6408 : _options$sourceFormat;
  var _options$sourceAttach = options.sourceAttachment,
      sourceAttachment = _options$sourceAttach === void 0 ? 36064 : _options$sourceAttach,
      _options$target = options.target,
      target = _options$target === void 0 ? null : _options$target,
      sourceWidth = options.sourceWidth,
      sourceHeight = options.sourceHeight,
      sourceType = options.sourceType;

  var _getFramebuffer = getFramebuffer(source),
      framebuffer = _getFramebuffer.framebuffer,
      deleteFramebuffer = _getFramebuffer.deleteFramebuffer;

  (0, _assert.assert)(framebuffer);
  var gl = framebuffer.gl,
      handle = framebuffer.handle,
      attachments = framebuffer.attachments;
  sourceWidth = sourceWidth || framebuffer.width;
  sourceHeight = sourceHeight || framebuffer.height;

  if (sourceAttachment === 36064 && handle === null) {
    sourceAttachment = 1028;
  }

  (0, _assert.assert)(attachments[sourceAttachment]);
  sourceType = sourceType || attachments[sourceAttachment].type;
  target = getPixelArray(target, sourceType, sourceFormat, sourceWidth, sourceHeight);
  sourceType = sourceType || (0, _typedArrayUtils.getGLTypeFromTypedArray)(target);
  var prevHandle = gl.bindFramebuffer(36160, handle);
  gl.readPixels(sourceX, sourceY, sourceWidth, sourceHeight, sourceFormat, sourceType, target);
  gl.bindFramebuffer(36160, prevHandle || null);

  if (deleteFramebuffer) {
    framebuffer.delete();
  }

  return target;
}

function readPixelsToBuffer(source, _ref) {
  var _ref$sourceX = _ref.sourceX,
      sourceX = _ref$sourceX === void 0 ? 0 : _ref$sourceX,
      _ref$sourceY = _ref.sourceY,
      sourceY = _ref$sourceY === void 0 ? 0 : _ref$sourceY,
      _ref$sourceFormat = _ref.sourceFormat,
      sourceFormat = _ref$sourceFormat === void 0 ? 6408 : _ref$sourceFormat,
      _ref$target = _ref.target,
      target = _ref$target === void 0 ? null : _ref$target,
      _ref$targetByteOffset = _ref.targetByteOffset,
      targetByteOffset = _ref$targetByteOffset === void 0 ? 0 : _ref$targetByteOffset,
      sourceWidth = _ref.sourceWidth,
      sourceHeight = _ref.sourceHeight,
      sourceType = _ref.sourceType;

  var _getFramebuffer2 = getFramebuffer(source),
      framebuffer = _getFramebuffer2.framebuffer,
      deleteFramebuffer = _getFramebuffer2.deleteFramebuffer;

  (0, _assert.assert)(framebuffer);
  sourceWidth = sourceWidth || framebuffer.width;
  sourceHeight = sourceHeight || framebuffer.height;
  var gl2 = (0, _gltools.assertWebGL2Context)(framebuffer.gl);
  sourceType = sourceType || (target ? target.type : 5121);

  if (!target) {
    var components = (0, _formatUtils.glFormatToComponents)(sourceFormat);
    var byteCount = (0, _formatUtils.glTypeToBytes)(sourceType);
    var byteLength = targetByteOffset + sourceWidth * sourceHeight * components * byteCount;
    target = new _buffer.default(gl2, {
      byteLength: byteLength,
      accessor: {
        type: sourceType,
        size: components
      }
    });
  }

  target.bind({
    target: 35051
  });
  (0, _gltools.withParameters)(gl2, {
    framebuffer: framebuffer
  }, function () {
    gl2.readPixels(sourceX, sourceY, sourceWidth, sourceHeight, sourceFormat, sourceType, targetByteOffset);
  });
  target.unbind({
    target: 35051
  });

  if (deleteFramebuffer) {
    framebuffer.delete();
  }

  return target;
}

function copyToDataUrl(source) {
  var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _ref2$sourceAttachmen = _ref2.sourceAttachment,
      sourceAttachment = _ref2$sourceAttachmen === void 0 ? 36064 : _ref2$sourceAttachmen,
      _ref2$targetMaxHeight = _ref2.targetMaxHeight,
      targetMaxHeight = _ref2$targetMaxHeight === void 0 ? Number.MAX_SAFE_INTEGER : _ref2$targetMaxHeight;

  var data = readPixelsToArray(source, {
    sourceAttachment: sourceAttachment
  });
  var width = source.width,
      height = source.height;

  while (height > targetMaxHeight) {
    var _scalePixels = (0, _typedArrayUtils.scalePixels)({
      data: data,
      width: width,
      height: height
    });

    data = _scalePixels.data;
    width = _scalePixels.width;
    height = _scalePixels.height;
  }

  (0, _typedArrayUtils.flipRows)({
    data: data,
    width: width,
    height: height
  });
  var canvas = document.createElement('canvas');
  canvas.width = width;
  canvas.height = height;
  var context = canvas.getContext('2d');
  var imageData = context.createImageData(width, height);
  imageData.data.set(data);
  context.putImageData(imageData, 0, 0);
  return canvas.toDataURL();
}

function copyToImage(source) {
  var _ref3 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _ref3$sourceAttachmen = _ref3.sourceAttachment,
      sourceAttachment = _ref3$sourceAttachmen === void 0 ? 36064 : _ref3$sourceAttachmen,
      _ref3$targetImage = _ref3.targetImage,
      targetImage = _ref3$targetImage === void 0 ? null : _ref3$targetImage;

  var dataUrl = copyToDataUrl(source, {
    sourceAttachment: sourceAttachment
  });
  targetImage = targetImage || new Image();
  targetImage.src = dataUrl;
  return targetImage;
}

function copyToTexture(source, target) {
  var options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
  var _options$sourceX2 = options.sourceX,
      sourceX = _options$sourceX2 === void 0 ? 0 : _options$sourceX2,
      _options$sourceY2 = options.sourceY,
      sourceY = _options$sourceY2 === void 0 ? 0 : _options$sourceY2,
      _options$targetMipmap = options.targetMipmaplevel,
      targetMipmaplevel = _options$targetMipmap === void 0 ? 0 : _options$targetMipmap,
      _options$targetIntern = options.targetInternalFormat,
      targetInternalFormat = _options$targetIntern === void 0 ? 6408 : _options$targetIntern;
  var targetX = options.targetX,
      targetY = options.targetY,
      targetZ = options.targetZ,
      width = options.width,
      height = options.height;

  var _getFramebuffer3 = getFramebuffer(source),
      framebuffer = _getFramebuffer3.framebuffer,
      deleteFramebuffer = _getFramebuffer3.deleteFramebuffer;

  (0, _assert.assert)(framebuffer);
  var gl = framebuffer.gl,
      handle = framebuffer.handle;
  var isSubCopy = typeof targetX !== 'undefined' || typeof targetY !== 'undefined' || typeof targetZ !== 'undefined';
  targetX = targetX || 0;
  targetY = targetY || 0;
  targetZ = targetZ || 0;
  var prevHandle = gl.bindFramebuffer(36160, handle);
  (0, _assert.assert)(target);
  var texture = null;

  if (target instanceof _texture.default) {
    texture = target;
    width = Number.isFinite(width) ? width : texture.width;
    height = Number.isFinite(height) ? height : texture.height;
    texture.bind(0);
    target = texture.target;
  }

  if (!isSubCopy) {
    gl.copyTexImage2D(target, targetMipmaplevel, targetInternalFormat, sourceX, sourceY, width, height, 0);
  } else {
    switch (target) {
      case 3553:
      case 34067:
        gl.copyTexSubImage2D(target, targetMipmaplevel, targetX, targetY, sourceX, sourceY, width, height);
        break;

      case 35866:
      case 32879:
        var gl2 = (0, _gltools.assertWebGL2Context)(gl);
        gl2.copyTexSubImage3D(target, targetMipmaplevel, targetX, targetY, targetZ, sourceX, sourceY, width, height);
        break;

      default:
    }
  }

  if (texture) {
    texture.unbind();
  }

  gl.bindFramebuffer(36160, prevHandle || null);

  if (deleteFramebuffer) {
    framebuffer.delete();
  }

  return texture;
}

function blit(source, target) {
  var options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
  var _options$sourceX3 = options.sourceX0,
      sourceX0 = _options$sourceX3 === void 0 ? 0 : _options$sourceX3,
      _options$sourceY3 = options.sourceY0,
      sourceY0 = _options$sourceY3 === void 0 ? 0 : _options$sourceY3,
      _options$targetX = options.targetX0,
      targetX0 = _options$targetX === void 0 ? 0 : _options$targetX,
      _options$targetY = options.targetY0,
      targetY0 = _options$targetY === void 0 ? 0 : _options$targetY,
      _options$color = options.color,
      color = _options$color === void 0 ? true : _options$color,
      _options$depth = options.depth,
      depth = _options$depth === void 0 ? false : _options$depth,
      _options$stencil = options.stencil,
      stencil = _options$stencil === void 0 ? false : _options$stencil,
      _options$filter = options.filter,
      filter = _options$filter === void 0 ? 9728 : _options$filter;
  var sourceX1 = options.sourceX1,
      sourceY1 = options.sourceY1,
      targetX1 = options.targetX1,
      targetY1 = options.targetY1,
      _options$sourceAttach2 = options.sourceAttachment,
      sourceAttachment = _options$sourceAttach2 === void 0 ? 36064 : _options$sourceAttach2,
      _options$mask = options.mask,
      mask = _options$mask === void 0 ? 0 : _options$mask;

  var _getFramebuffer4 = getFramebuffer(source),
      srcFramebuffer = _getFramebuffer4.framebuffer,
      deleteSrcFramebuffer = _getFramebuffer4.deleteFramebuffer;

  var _getFramebuffer5 = getFramebuffer(target),
      dstFramebuffer = _getFramebuffer5.framebuffer,
      deleteDstFramebuffer = _getFramebuffer5.deleteFramebuffer;

  (0, _assert.assert)(srcFramebuffer);
  (0, _assert.assert)(dstFramebuffer);
  var gl = dstFramebuffer.gl,
      handle = dstFramebuffer.handle,
      width = dstFramebuffer.width,
      height = dstFramebuffer.height,
      readBuffer = dstFramebuffer.readBuffer;
  var gl2 = (0, _gltools.assertWebGL2Context)(gl);

  if (!srcFramebuffer.handle && sourceAttachment === 36064) {
    sourceAttachment = 1028;
  }

  if (color) {
    mask |= 16384;
  }

  if (depth) {
    mask |= 256;
  }

  if (stencil) {
    mask |= 1024;
  }

  if (deleteSrcFramebuffer || deleteDstFramebuffer) {
    if (mask & (256 | 1024)) {
      mask = 16384;

      _gltools.log.warn('Blitting from or into a Texture object, forcing mask to GL.COLOR_BUFFER_BIT')();
    }
  }

  (0, _assert.assert)(mask);
  sourceX1 = sourceX1 === undefined ? srcFramebuffer.width : sourceX1;
  sourceY1 = sourceY1 === undefined ? srcFramebuffer.height : sourceY1;
  targetX1 = targetX1 === undefined ? width : targetX1;
  targetY1 = targetY1 === undefined ? height : targetY1;
  var prevDrawHandle = gl.bindFramebuffer(36009, handle);
  var prevReadHandle = gl.bindFramebuffer(36008, srcFramebuffer.handle);
  gl2.readBuffer(sourceAttachment);
  gl2.blitFramebuffer(sourceX0, sourceY0, sourceX1, sourceY1, targetX0, targetY0, targetX1, targetY1, mask, filter);
  gl2.readBuffer(readBuffer);
  gl2.bindFramebuffer(36008, prevReadHandle || null);
  gl2.bindFramebuffer(36009, prevDrawHandle || null);

  if (deleteSrcFramebuffer) {
    srcFramebuffer.delete();
  }

  if (deleteDstFramebuffer) {
    dstFramebuffer.delete();
  }

  return dstFramebuffer;
}

function getFramebuffer(source) {
  if (!(source instanceof _framebuffer.default)) {
    return {
      framebuffer: (0, _textureUtils.toFramebuffer)(source),
      deleteFramebuffer: true
    };
  }

  return {
    framebuffer: source,
    deleteFramebuffer: false
  };
}

function getPixelArray(pixelArray, type, format, width, height) {
  if (pixelArray) {
    return pixelArray;
  }

  type = type || 5121;
  var ArrayType = (0, _typedArrayUtils.getTypedArrayFromGLType)(type, {
    clamped: false
  });
  var components = (0, _formatUtils.glFormatToComponents)(format);
  return new ArrayType(width * height * components);
}
//# sourceMappingURL=copy-and-blit.js.map