import Resource from './resource';
import Accessor from './accessor';
import { getGLTypeFromTypedArray, getTypedArrayFromGLType } from '../webgl-utils/typed-array-utils';
import { assertWebGL2Context, log } from '@luma.gl/gltools';
import { assert } from '../utils/assert';
import { checkProps } from '../utils/check-props';
const DEBUG_DATA_LENGTH = 10;
const DEPRECATED_PROPS = {
  offset: 'accessor.offset',
  stride: 'accessor.stride',
  type: 'accessor.type',
  size: 'accessor.size',
  divisor: 'accessor.divisor',
  normalized: 'accessor.normalized',
  integer: 'accessor.integer',
  instanced: 'accessor.divisor',
  isInstanced: 'accessor.divisor'
};
const PROP_CHECKS_INITIALIZE = {
  removedProps: {},
  replacedProps: {
    bytes: 'byteLength'
  },
  deprecatedProps: DEPRECATED_PROPS
};
const PROP_CHECKS_SET_PROPS = {
  removedProps: DEPRECATED_PROPS
};
export default class Buffer extends Resource {
  get [Symbol.toStringTag]() {
    return 'Buffer';
  }

  constructor(gl) {
    let props = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    super(gl, props);
    this.stubRemovedMethods('Buffer', 'v6.0', ['layout', 'setLayout', 'getIndexedParameter']);
    this.target = props.target || (this.gl.webgl2 ? 36662 : 34962);
    this.initialize(props);
    Object.seal(this);
  }

  getElementCount() {
    let accessor = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.accessor;
    return Math.round(this.byteLength / Accessor.getBytesPerElement(accessor));
  }

  getVertexCount() {
    let accessor = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.accessor;
    return Math.round(this.byteLength / Accessor.getBytesPerVertex(accessor));
  }

  initialize() {
    let props = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

    if (ArrayBuffer.isView(props)) {
      props = {
        data: props
      };
    }

    if (Number.isFinite(props)) {
      props = {
        byteLength: props
      };
    }

    props = checkProps('Buffer', props, PROP_CHECKS_INITIALIZE);
    this.usage = props.usage || 35044;
    this.debugData = null;
    this.setAccessor(Object.assign({}, props, props.accessor));

    if (props.data) {
      this._setData(props.data, props.offset, props.byteLength);
    } else {
      this._setByteLength(props.byteLength || 0);
    }

    return this;
  }

  setProps(props) {
    props = checkProps('Buffer', props, PROP_CHECKS_SET_PROPS);

    if ('accessor' in props) {
      this.setAccessor(props.accessor);
    }

    return this;
  }

  setAccessor(accessor) {
    accessor = Object.assign({}, accessor);
    delete accessor.buffer;
    this.accessor = new Accessor(accessor);
    return this;
  }

  reallocate(byteLength) {
    if (byteLength > this.byteLength) {
      this._setByteLength(byteLength);

      return true;
    }

    this.bytesUsed = byteLength;
    return false;
  }

  setData(props) {
    return this.initialize(props);
  }

  subData(props) {
    if (ArrayBuffer.isView(props)) {
      props = {
        data: props
      };
    }

    const {
      data,
      offset = 0,
      srcOffset = 0
    } = props;
    const byteLength = props.byteLength || props.length;
    assert(data);
    const target = this.gl.webgl2 ? 36663 : this.target;
    this.gl.bindBuffer(target, this.handle);

    if (srcOffset !== 0 || byteLength !== undefined) {
      assertWebGL2Context(this.gl);
      this.gl.bufferSubData(this.target, offset, data, srcOffset, byteLength);
    } else {
      this.gl.bufferSubData(target, offset, data);
    }

    this.gl.bindBuffer(target, null);
    this.debugData = null;

    this._inferType(data);

    return this;
  }

  copyData(_ref) {
    let {
      sourceBuffer,
      readOffset = 0,
      writeOffset = 0,
      size
    } = _ref;
    const {
      gl
    } = this;
    assertWebGL2Context(gl);
    gl.bindBuffer(36662, sourceBuffer.handle);
    gl.bindBuffer(36663, this.handle);
    gl.copyBufferSubData(36662, 36663, readOffset, writeOffset, size);
    gl.bindBuffer(36662, null);
    gl.bindBuffer(36663, null);
    this.debugData = null;
    return this;
  }

  getData() {
    let {
      dstData = null,
      srcByteOffset = 0,
      dstOffset = 0,
      length = 0
    } = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    assertWebGL2Context(this.gl);
    const ArrayType = getTypedArrayFromGLType(this.accessor.type || 5126, {
      clamped: false
    });

    const sourceAvailableElementCount = this._getAvailableElementCount(srcByteOffset);

    const dstElementOffset = dstOffset;
    let dstAvailableElementCount;
    let dstElementCount;

    if (dstData) {
      dstElementCount = dstData.length;
      dstAvailableElementCount = dstElementCount - dstElementOffset;
    } else {
      dstAvailableElementCount = Math.min(sourceAvailableElementCount, length || sourceAvailableElementCount);
      dstElementCount = dstElementOffset + dstAvailableElementCount;
    }

    const copyElementCount = Math.min(sourceAvailableElementCount, dstAvailableElementCount);
    length = length || copyElementCount;
    assert(length <= copyElementCount);
    dstData = dstData || new ArrayType(dstElementCount);
    this.gl.bindBuffer(36662, this.handle);
    this.gl.getBufferSubData(36662, srcByteOffset, dstData, dstOffset, length);
    this.gl.bindBuffer(36662, null);
    return dstData;
  }

  bind() {
    let {
      target = this.target,
      index = this.accessor && this.accessor.index,
      offset = 0,
      size
    } = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

    if (target === 35345 || target === 35982) {
      if (size !== undefined) {
        this.gl.bindBufferRange(target, index, this.handle, offset, size);
      } else {
        assert(offset === 0);
        this.gl.bindBufferBase(target, index, this.handle);
      }
    } else {
      this.gl.bindBuffer(target, this.handle);
    }

    return this;
  }

  unbind() {
    let {
      target = this.target,
      index = this.accessor && this.accessor.index
    } = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    const isIndexedBuffer = target === 35345 || target === 35982;

    if (isIndexedBuffer) {
      this.gl.bindBufferBase(target, index, null);
    } else {
      this.gl.bindBuffer(target, null);
    }

    return this;
  }

  getDebugData() {
    if (!this.debugData) {
      this.debugData = this.getData({
        length: Math.min(DEBUG_DATA_LENGTH, this.byteLength)
      });
      return {
        data: this.debugData,
        changed: true
      };
    }

    return {
      data: this.debugData,
      changed: false
    };
  }

  invalidateDebugData() {
    this.debugData = null;
  }

  _setData(data) {
    let offset = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
    let byteLength = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : data.byteLength + offset;
    assert(ArrayBuffer.isView(data));

    this._trackDeallocatedMemory();

    const target = this._getTarget();

    this.gl.bindBuffer(target, this.handle);
    this.gl.bufferData(target, byteLength, this.usage);
    this.gl.bufferSubData(target, offset, data);
    this.gl.bindBuffer(target, null);
    this.debugData = data.slice(0, DEBUG_DATA_LENGTH);
    this.bytesUsed = byteLength;

    this._trackAllocatedMemory(byteLength);

    const type = getGLTypeFromTypedArray(data);
    assert(type);
    this.setAccessor(new Accessor(this.accessor, {
      type
    }));
    return this;
  }

  _setByteLength(byteLength) {
    let usage = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : this.usage;
    assert(byteLength >= 0);

    this._trackDeallocatedMemory();

    let data = byteLength;

    if (byteLength === 0) {
      data = new Float32Array(0);
    }

    const target = this._getTarget();

    this.gl.bindBuffer(target, this.handle);
    this.gl.bufferData(target, data, usage);
    this.gl.bindBuffer(target, null);
    this.usage = usage;
    this.debugData = null;
    this.bytesUsed = byteLength;

    this._trackAllocatedMemory(byteLength);

    return this;
  }

  _getTarget() {
    return this.gl.webgl2 ? 36663 : this.target;
  }

  _getAvailableElementCount(srcByteOffset) {
    const ArrayType = getTypedArrayFromGLType(this.accessor.type || 5126, {
      clamped: false
    });
    const sourceElementOffset = srcByteOffset / ArrayType.BYTES_PER_ELEMENT;
    return this.getElementCount() - sourceElementOffset;
  }

  _inferType(data) {
    if (!this.accessor.type) {
      this.setAccessor(new Accessor(this.accessor, {
        type: getGLTypeFromTypedArray(data)
      }));
    }
  }

  _createHandle() {
    return this.gl.createBuffer();
  }

  _deleteHandle() {
    this.gl.deleteBuffer(this.handle);

    this._trackDeallocatedMemory();
  }

  _getParameter(pname) {
    this.gl.bindBuffer(this.target, this.handle);
    const value = this.gl.getBufferParameter(this.target, pname);
    this.gl.bindBuffer(this.target, null);
    return value;
  }

  get type() {
    log.deprecated('Buffer.type', 'Buffer.accessor.type')();
    return this.accessor.type;
  }

  get bytes() {
    log.deprecated('Buffer.bytes', 'Buffer.byteLength')();
    return this.byteLength;
  }

  setByteLength(byteLength) {
    log.deprecated('setByteLength', 'reallocate')();
    return this.reallocate(byteLength);
  }

  updateAccessor(opts) {
    log.deprecated('updateAccessor(...)', 'setAccessor(new Accessor(buffer.accessor, ...)')();
    this.accessor = new Accessor(this.accessor, opts);
    return this;
  }

}
//# sourceMappingURL=buffer.js.map