import Buffer from './buffer';
import Framebuffer from './framebuffer';
import Texture from './texture';
import { assertWebGL2Context, withParameters, log } from '@luma.gl/gltools';
import { flipRows, scalePixels } from '../webgl-utils/typed-array-utils';
import { getTypedArrayFromGLType, getGLTypeFromTypedArray } from '../webgl-utils/typed-array-utils';
import { glFormatToComponents, glTypeToBytes } from '../webgl-utils/format-utils';
import { toFramebuffer } from '../webgl-utils/texture-utils';
import { assert } from '../utils/assert';
export function readPixelsToArray(source) {
  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  const {
    sourceX = 0,
    sourceY = 0,
    sourceFormat = 6408
  } = options;
  let {
    sourceAttachment = 36064,
    target = null,
    sourceWidth,
    sourceHeight,
    sourceType
  } = options;
  const {
    framebuffer,
    deleteFramebuffer
  } = getFramebuffer(source);
  assert(framebuffer);
  const {
    gl,
    handle,
    attachments
  } = framebuffer;
  sourceWidth = sourceWidth || framebuffer.width;
  sourceHeight = sourceHeight || framebuffer.height;

  if (sourceAttachment === 36064 && handle === null) {
    sourceAttachment = 1028;
  }

  assert(attachments[sourceAttachment]);
  sourceType = sourceType || attachments[sourceAttachment].type;
  target = getPixelArray(target, sourceType, sourceFormat, sourceWidth, sourceHeight);
  sourceType = sourceType || getGLTypeFromTypedArray(target);
  const prevHandle = gl.bindFramebuffer(36160, handle);
  gl.readPixels(sourceX, sourceY, sourceWidth, sourceHeight, sourceFormat, sourceType, target);
  gl.bindFramebuffer(36160, prevHandle || null);

  if (deleteFramebuffer) {
    framebuffer.delete();
  }

  return target;
}
export function readPixelsToBuffer(source, _ref) {
  let {
    sourceX = 0,
    sourceY = 0,
    sourceFormat = 6408,
    target = null,
    targetByteOffset = 0,
    sourceWidth,
    sourceHeight,
    sourceType
  } = _ref;
  const {
    framebuffer,
    deleteFramebuffer
  } = getFramebuffer(source);
  assert(framebuffer);
  sourceWidth = sourceWidth || framebuffer.width;
  sourceHeight = sourceHeight || framebuffer.height;
  const gl2 = assertWebGL2Context(framebuffer.gl);
  sourceType = sourceType || (target ? target.type : 5121);

  if (!target) {
    const components = glFormatToComponents(sourceFormat);
    const byteCount = glTypeToBytes(sourceType);
    const byteLength = targetByteOffset + sourceWidth * sourceHeight * components * byteCount;
    target = new Buffer(gl2, {
      byteLength,
      accessor: {
        type: sourceType,
        size: components
      }
    });
  }

  target.bind({
    target: 35051
  });
  withParameters(gl2, {
    framebuffer
  }, () => {
    gl2.readPixels(sourceX, sourceY, sourceWidth, sourceHeight, sourceFormat, sourceType, targetByteOffset);
  });
  target.unbind({
    target: 35051
  });

  if (deleteFramebuffer) {
    framebuffer.delete();
  }

  return target;
}
export function copyToDataUrl(source) {
  let {
    sourceAttachment = 36064,
    targetMaxHeight = Number.MAX_SAFE_INTEGER
  } = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  let data = readPixelsToArray(source, {
    sourceAttachment
  });
  let {
    width,
    height
  } = source;

  while (height > targetMaxHeight) {
    ({
      data,
      width,
      height
    } = scalePixels({
      data,
      width,
      height
    }));
  }

  flipRows({
    data,
    width,
    height
  });
  const canvas = document.createElement('canvas');
  canvas.width = width;
  canvas.height = height;
  const context = canvas.getContext('2d');
  const imageData = context.createImageData(width, height);
  imageData.data.set(data);
  context.putImageData(imageData, 0, 0);
  return canvas.toDataURL();
}
export function copyToImage(source) {
  let {
    sourceAttachment = 36064,
    targetImage = null
  } = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  const dataUrl = copyToDataUrl(source, {
    sourceAttachment
  });
  targetImage = targetImage || new Image();
  targetImage.src = dataUrl;
  return targetImage;
}
export function copyToTexture(source, target) {
  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
  const {
    sourceX = 0,
    sourceY = 0,
    targetMipmaplevel = 0,
    targetInternalFormat = 6408
  } = options;
  let {
    targetX,
    targetY,
    targetZ,
    width,
    height
  } = options;
  const {
    framebuffer,
    deleteFramebuffer
  } = getFramebuffer(source);
  assert(framebuffer);
  const {
    gl,
    handle
  } = framebuffer;
  const isSubCopy = typeof targetX !== 'undefined' || typeof targetY !== 'undefined' || typeof targetZ !== 'undefined';
  targetX = targetX || 0;
  targetY = targetY || 0;
  targetZ = targetZ || 0;
  const prevHandle = gl.bindFramebuffer(36160, handle);
  assert(target);
  let texture = null;

  if (target instanceof Texture) {
    texture = target;
    width = Number.isFinite(width) ? width : texture.width;
    height = Number.isFinite(height) ? height : texture.height;
    texture.bind(0);
    target = texture.target;
  }

  if (!isSubCopy) {
    gl.copyTexImage2D(target, targetMipmaplevel, targetInternalFormat, sourceX, sourceY, width, height, 0);
  } else {
    switch (target) {
      case 3553:
      case 34067:
        gl.copyTexSubImage2D(target, targetMipmaplevel, targetX, targetY, sourceX, sourceY, width, height);
        break;

      case 35866:
      case 32879:
        const gl2 = assertWebGL2Context(gl);
        gl2.copyTexSubImage3D(target, targetMipmaplevel, targetX, targetY, targetZ, sourceX, sourceY, width, height);
        break;

      default:
    }
  }

  if (texture) {
    texture.unbind();
  }

  gl.bindFramebuffer(36160, prevHandle || null);

  if (deleteFramebuffer) {
    framebuffer.delete();
  }

  return texture;
}
export function blit(source, target) {
  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
  const {
    sourceX0 = 0,
    sourceY0 = 0,
    targetX0 = 0,
    targetY0 = 0,
    color = true,
    depth = false,
    stencil = false,
    filter = 9728
  } = options;
  let {
    sourceX1,
    sourceY1,
    targetX1,
    targetY1,
    sourceAttachment = 36064,
    mask = 0
  } = options;
  const {
    framebuffer: srcFramebuffer,
    deleteFramebuffer: deleteSrcFramebuffer
  } = getFramebuffer(source);
  const {
    framebuffer: dstFramebuffer,
    deleteFramebuffer: deleteDstFramebuffer
  } = getFramebuffer(target);
  assert(srcFramebuffer);
  assert(dstFramebuffer);
  const {
    gl,
    handle,
    width,
    height,
    readBuffer
  } = dstFramebuffer;
  const gl2 = assertWebGL2Context(gl);

  if (!srcFramebuffer.handle && sourceAttachment === 36064) {
    sourceAttachment = 1028;
  }

  if (color) {
    mask |= 16384;
  }

  if (depth) {
    mask |= 256;
  }

  if (stencil) {
    mask |= 1024;
  }

  if (deleteSrcFramebuffer || deleteDstFramebuffer) {
    if (mask & (256 | 1024)) {
      mask = 16384;
      log.warn('Blitting from or into a Texture object, forcing mask to GL.COLOR_BUFFER_BIT')();
    }
  }

  assert(mask);
  sourceX1 = sourceX1 === undefined ? srcFramebuffer.width : sourceX1;
  sourceY1 = sourceY1 === undefined ? srcFramebuffer.height : sourceY1;
  targetX1 = targetX1 === undefined ? width : targetX1;
  targetY1 = targetY1 === undefined ? height : targetY1;
  const prevDrawHandle = gl.bindFramebuffer(36009, handle);
  const prevReadHandle = gl.bindFramebuffer(36008, srcFramebuffer.handle);
  gl2.readBuffer(sourceAttachment);
  gl2.blitFramebuffer(sourceX0, sourceY0, sourceX1, sourceY1, targetX0, targetY0, targetX1, targetY1, mask, filter);
  gl2.readBuffer(readBuffer);
  gl2.bindFramebuffer(36008, prevReadHandle || null);
  gl2.bindFramebuffer(36009, prevDrawHandle || null);

  if (deleteSrcFramebuffer) {
    srcFramebuffer.delete();
  }

  if (deleteDstFramebuffer) {
    dstFramebuffer.delete();
  }

  return dstFramebuffer;
}

function getFramebuffer(source) {
  if (!(source instanceof Framebuffer)) {
    return {
      framebuffer: toFramebuffer(source),
      deleteFramebuffer: true
    };
  }

  return {
    framebuffer: source,
    deleteFramebuffer: false
  };
}

function getPixelArray(pixelArray, type, format, width, height) {
  if (pixelArray) {
    return pixelArray;
  }

  type = type || 5121;
  const ArrayType = getTypedArrayFromGLType(type, {
    clamped: false
  });
  const components = glFormatToComponents(format);
  return new ArrayType(width * height * components);
}
//# sourceMappingURL=copy-and-blit.js.map