import { getWebGL2Context, assertWebGL2Context, log } from '@luma.gl/gltools';
import Resource from './resource';
import Texture2D from './texture-2d';
import Renderbuffer from './renderbuffer';
import { clear, clearBuffer } from './clear';
import { copyToDataUrl } from './copy-and-blit.js';
import { getFeatures } from '../features';
import { getKey } from '../webgl-utils/constants-to-keys';
import { assert } from '../utils/assert';
const ERR_MULTIPLE_RENDERTARGETS = 'Multiple render targets not supported';
export default class Framebuffer extends Resource {
  get [Symbol.toStringTag]() {
    return 'Framebuffer';
  }

  static isSupported(gl) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      colorBufferFloat,
      colorBufferHalfFloat
    } = options;
    let supported = true;

    if (colorBufferFloat) {
      supported = Boolean(gl.getExtension('EXT_color_buffer_float') || gl.getExtension('WEBGL_color_buffer_float') || gl.getExtension('OES_texture_float'));
    }

    if (colorBufferHalfFloat) {
      supported = supported && Boolean(gl.getExtension('EXT_color_buffer_float') || gl.getExtension('EXT_color_buffer_half_float'));
    }

    return supported;
  }

  static getDefaultFramebuffer(gl) {
    gl.luma = gl.luma || {};
    gl.luma.defaultFramebuffer = gl.luma.defaultFramebuffer || new Framebuffer(gl, {
      id: 'default-framebuffer',
      handle: null,
      attachments: {}
    });
    return gl.luma.defaultFramebuffer;
  }

  get MAX_COLOR_ATTACHMENTS() {
    const gl2 = assertWebGL2Context(this.gl);
    return gl2.getParameter(gl2.MAX_COLOR_ATTACHMENTS);
  }

  get MAX_DRAW_BUFFERS() {
    const gl2 = assertWebGL2Context(this.gl);
    return gl2.getParameter(gl2.MAX_DRAW_BUFFERS);
  }

  constructor(gl) {
    let opts = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    super(gl, opts);
    this.width = null;
    this.height = null;
    this.attachments = {};
    this.readBuffer = 36064;
    this.drawBuffers = [36064];
    this.ownResources = [];
    this.initialize(opts);
    Object.seal(this);
  }

  get color() {
    return this.attachments[36064] || null;
  }

  get texture() {
    return this.attachments[36064] || null;
  }

  get depth() {
    return this.attachments[36096] || this.attachments[33306] || null;
  }

  get stencil() {
    return this.attachments[36128] || this.attachments[33306] || null;
  }

  initialize(_ref) {
    let {
      width = 1,
      height = 1,
      attachments = null,
      color = true,
      depth = true,
      stencil = false,
      check = true,
      readBuffer = undefined,
      drawBuffers = undefined
    } = _ref;
    assert(width >= 0 && height >= 0, 'Width and height need to be integers');
    this.width = width;
    this.height = height;

    if (attachments) {
      for (const attachment in attachments) {
        const target = attachments[attachment];
        const object = Array.isArray(target) ? target[0] : target;
        object.resize({
          width,
          height
        });
      }
    } else {
      attachments = this._createDefaultAttachments(color, depth, stencil, width, height);
    }

    this.update({
      clearAttachments: true,
      attachments,
      readBuffer,
      drawBuffers
    });

    if (attachments && check) {
      this.checkStatus();
    }
  }

  delete() {
    for (const resource of this.ownResources) {
      resource.delete();
    }

    super.delete();
    return this;
  }

  update(_ref2) {
    let {
      attachments = {},
      readBuffer,
      drawBuffers,
      clearAttachments = false,
      resizeAttachments = true
    } = _ref2;
    this.attach(attachments, {
      clearAttachments,
      resizeAttachments
    });
    const {
      gl
    } = this;
    const prevHandle = gl.bindFramebuffer(36160, this.handle);

    if (readBuffer) {
      this._setReadBuffer(readBuffer);
    }

    if (drawBuffers) {
      this._setDrawBuffers(drawBuffers);
    }

    gl.bindFramebuffer(36160, prevHandle || null);
    return this;
  }

  resize() {
    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    let {
      width,
      height
    } = options;

    if (this.handle === null) {
      assert(width === undefined && height === undefined);
      this.width = this.gl.drawingBufferWidth;
      this.height = this.gl.drawingBufferHeight;
      return this;
    }

    if (width === undefined) {
      width = this.gl.drawingBufferWidth;
    }

    if (height === undefined) {
      height = this.gl.drawingBufferHeight;
    }

    if (width !== this.width && height !== this.height) {
      log.log(2, "Resizing framebuffer ".concat(this.id, " to ").concat(width, "x").concat(height))();
    }

    for (const attachmentPoint in this.attachments) {
      this.attachments[attachmentPoint].resize({
        width,
        height
      });
    }

    this.width = width;
    this.height = height;
    return this;
  }

  attach(attachments) {
    let {
      clearAttachments = false,
      resizeAttachments = true
    } = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const newAttachments = {};

    if (clearAttachments) {
      Object.keys(this.attachments).forEach(key => {
        newAttachments[key] = null;
      });
    }

    Object.assign(newAttachments, attachments);
    const prevHandle = this.gl.bindFramebuffer(36160, this.handle);

    for (const key in newAttachments) {
      assert(key !== undefined, 'Misspelled framebuffer binding point?');
      const attachment = Number(key);
      const descriptor = newAttachments[attachment];
      let object = descriptor;

      if (!object) {
        this._unattach(attachment);
      } else if (object instanceof Renderbuffer) {
        this._attachRenderbuffer({
          attachment,
          renderbuffer: object
        });
      } else if (Array.isArray(descriptor)) {
        const [texture, layer = 0, level = 0] = descriptor;
        object = texture;

        this._attachTexture({
          attachment,
          texture,
          layer,
          level
        });
      } else {
        this._attachTexture({
          attachment,
          texture: object,
          layer: 0,
          level: 0
        });
      }

      if (resizeAttachments && object) {
        object.resize({
          width: this.width,
          height: this.height
        });
      }
    }

    this.gl.bindFramebuffer(36160, prevHandle || null);
    Object.assign(this.attachments, attachments);
    Object.keys(this.attachments).filter(key => !this.attachments[key]).forEach(key => {
      delete this.attachments[key];
    });
  }

  checkStatus() {
    const {
      gl
    } = this;
    const status = this.getStatus();

    if (status !== 36053) {
      throw new Error(_getFrameBufferStatus(status));
    }

    return this;
  }

  getStatus() {
    const {
      gl
    } = this;
    const prevHandle = gl.bindFramebuffer(36160, this.handle);
    const status = gl.checkFramebufferStatus(36160);
    gl.bindFramebuffer(36160, prevHandle || null);
    return status;
  }

  clear() {
    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    const {
      color,
      depth,
      stencil,
      drawBuffers = []
    } = options;
    const prevHandle = this.gl.bindFramebuffer(36160, this.handle);

    if (color || depth || stencil) {
      clear(this.gl, {
        color,
        depth,
        stencil
      });
    }

    drawBuffers.forEach((value, drawBuffer) => {
      clearBuffer(this.gl, {
        drawBuffer,
        value
      });
    });
    this.gl.bindFramebuffer(36160, prevHandle || null);
    return this;
  }

  readPixels() {
    let opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    log.error('Framebuffer.readPixels() is no logner supported, use readPixelsToArray(framebuffer)')();
    return null;
  }

  readPixelsToBuffer() {
    let opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    log.error('Framebuffer.readPixelsToBuffer()is no logner supported, use readPixelsToBuffer(framebuffer)')();
    return null;
  }

  copyToDataUrl() {
    let opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    log.error('Framebuffer.copyToDataUrl() is no logner supported, use copyToDataUrl(framebuffer)')();
    return null;
  }

  copyToImage() {
    let opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    log.error('Framebuffer.copyToImage() is no logner supported, use copyToImage(framebuffer)')();
    return null;
  }

  copyToTexture() {
    let opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    log.error('Framebuffer.copyToTexture({...}) is no logner supported, use copyToTexture(source, target, opts})')();
    return null;
  }

  blit() {
    let opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    log.error('Framebuffer.blit({...}) is no logner supported, use blit(source, target, opts)')();
    return null;
  }

  invalidate(_ref3) {
    let {
      attachments = [],
      x = 0,
      y = 0,
      width,
      height
    } = _ref3;
    const gl2 = assertWebGL2Context(this.gl);
    const prevHandle = gl2.bindFramebuffer(36008, this.handle);
    const invalidateAll = x === 0 && y === 0 && width === undefined && height === undefined;

    if (invalidateAll) {
      gl2.invalidateFramebuffer(36008, attachments);
    } else {
      gl2.invalidateFramebuffer(36008, attachments, x, y, width, height);
    }

    gl2.bindFramebuffer(36008, prevHandle);
    return this;
  }

  getAttachmentParameter(attachment, pname, keys) {
    let value = this._getAttachmentParameterFallback(pname);

    if (value === null) {
      this.gl.bindFramebuffer(36160, this.handle);
      value = this.gl.getFramebufferAttachmentParameter(36160, attachment, pname);
      this.gl.bindFramebuffer(36160, null);
    }

    if (keys && value > 1000) {
      value = getKey(this.gl, value);
    }

    return value;
  }

  getAttachmentParameters() {
    let attachment = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 36064;
    let keys = arguments.length > 1 ? arguments[1] : undefined;
    let parameters = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : this.constructor.ATTACHMENT_PARAMETERS || [];
    const values = {};

    for (const pname of parameters) {
      const key = keys ? getKey(this.gl, pname) : pname;
      values[key] = this.getAttachmentParameter(attachment, pname, keys);
    }

    return values;
  }

  getParameters() {
    let keys = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : true;
    const attachments = Object.keys(this.attachments);
    const parameters = {};

    for (const attachmentName of attachments) {
      const attachment = Number(attachmentName);
      const key = keys ? getKey(this.gl, attachment) : attachment;
      parameters[key] = this.getAttachmentParameters(attachment, keys);
    }

    return parameters;
  }

  show() {
    if (typeof window !== 'undefined') {
      window.open(copyToDataUrl(this), 'luma-debug-texture');
    }

    return this;
  }

  log() {
    let logLevel = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0;
    let message = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : '';

    if (logLevel > log.level || typeof window === 'undefined') {
      return this;
    }

    message = message || "Framebuffer ".concat(this.id);
    const image = copyToDataUrl(this, {
      targetMaxHeight: 100
    });
    log.image({
      logLevel,
      message,
      image
    }, message)();
    return this;
  }

  bind() {
    let {
      target = 36160
    } = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    this.gl.bindFramebuffer(target, this.handle);
    return this;
  }

  unbind() {
    let {
      target = 36160
    } = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    this.gl.bindFramebuffer(target, null);
    return this;
  }

  _createDefaultAttachments(color, depth, stencil, width, height) {
    let defaultAttachments = null;

    if (color) {
      defaultAttachments = defaultAttachments || {};
      defaultAttachments[36064] = new Texture2D(this.gl, {
        id: "".concat(this.id, "-color0"),
        pixels: null,
        format: 6408,
        type: 5121,
        width,
        height,
        mipmaps: false,
        parameters: {
          [10241]: 9729,
          [10240]: 9729,
          [10242]: 33071,
          [10243]: 33071
        }
      });
      this.ownResources.push(defaultAttachments[36064]);
    }

    if (depth && stencil) {
      defaultAttachments = defaultAttachments || {};
      defaultAttachments[33306] = new Renderbuffer(this.gl, {
        id: "".concat(this.id, "-depth-stencil"),
        format: 35056,
        width,
        height: 111
      });
      this.ownResources.push(defaultAttachments[33306]);
    } else if (depth) {
      defaultAttachments = defaultAttachments || {};
      defaultAttachments[36096] = new Renderbuffer(this.gl, {
        id: "".concat(this.id, "-depth"),
        format: 33189,
        width,
        height
      });
      this.ownResources.push(defaultAttachments[36096]);
    } else if (stencil) {
      assert(false);
    }

    return defaultAttachments;
  }

  _unattach(attachment) {
    const oldAttachment = this.attachments[attachment];

    if (!oldAttachment) {
      return;
    }

    if (oldAttachment instanceof Renderbuffer) {
      this.gl.framebufferRenderbuffer(36160, attachment, 36161, null);
    } else {
      this.gl.framebufferTexture2D(36160, attachment, 3553, null, 0);
    }

    delete this.attachments[attachment];
  }

  _attachRenderbuffer(_ref4) {
    let {
      attachment = 36064,
      renderbuffer
    } = _ref4;
    const {
      gl
    } = this;
    gl.framebufferRenderbuffer(36160, attachment, 36161, renderbuffer.handle);
    this.attachments[attachment] = renderbuffer;
  }

  _attachTexture(_ref5) {
    let {
      attachment = 36064,
      texture,
      layer,
      level
    } = _ref5;
    const {
      gl
    } = this;
    gl.bindTexture(texture.target, texture.handle);

    switch (texture.target) {
      case 35866:
      case 32879:
        const gl2 = assertWebGL2Context(gl);
        gl2.framebufferTextureLayer(36160, attachment, texture.target, level, layer);
        break;

      case 34067:
        const face = mapIndexToCubeMapFace(layer);
        gl.framebufferTexture2D(36160, attachment, face, texture.handle, level);
        break;

      case 3553:
        gl.framebufferTexture2D(36160, attachment, 3553, texture.handle, level);
        break;

      default:
        assert(false, 'Illegal texture type');
    }

    gl.bindTexture(texture.target, null);
    this.attachments[attachment] = texture;
  }

  _setReadBuffer(readBuffer) {
    const gl2 = getWebGL2Context(this.gl);

    if (gl2) {
      gl2.readBuffer(readBuffer);
    } else {
      assert(readBuffer === 36064 || readBuffer === 1029, ERR_MULTIPLE_RENDERTARGETS);
    }

    this.readBuffer = readBuffer;
  }

  _setDrawBuffers(drawBuffers) {
    const {
      gl
    } = this;
    const gl2 = assertWebGL2Context(gl);

    if (gl2) {
      gl2.drawBuffers(drawBuffers);
    } else {
      const ext = gl.getExtension('WEBGL_draw_buffers');

      if (ext) {
        ext.drawBuffersWEBGL(drawBuffers);
      } else {
        assert(drawBuffers.length === 1 && (drawBuffers[0] === 36064 || drawBuffers[0] === 1029), ERR_MULTIPLE_RENDERTARGETS);
      }
    }

    this.drawBuffers = drawBuffers;
  }

  _getAttachmentParameterFallback(pname) {
    const caps = getFeatures(this.gl);

    switch (pname) {
      case 36052:
        return !caps.WEBGL2 ? 0 : null;

      case 33298:
      case 33299:
      case 33300:
      case 33301:
      case 33302:
      case 33303:
        return !caps.WEBGL2 ? 8 : null;

      case 33297:
        return !caps.WEBGL2 ? 5125 : null;

      case 33296:
        return !caps.WEBGL2 && !caps.EXT_sRGB ? 9729 : null;

      default:
        return null;
    }
  }

  _createHandle() {
    return this.gl.createFramebuffer();
  }

  _deleteHandle() {
    this.gl.deleteFramebuffer(this.handle);
  }

  _bindHandle(handle) {
    return this.gl.bindFramebuffer(36160, handle);
  }

}

function mapIndexToCubeMapFace(layer) {
  return layer < 34069 ? layer + 34069 : layer;
}

function _getFrameBufferStatus(status) {
  const STATUS = Framebuffer.STATUS || {};
  return STATUS[status] || "Framebuffer error ".concat(status);
}

export const FRAMEBUFFER_ATTACHMENT_PARAMETERS = [36049, 36048, 33296, 33298, 33299, 33300, 33301, 33302, 33303];
Framebuffer.ATTACHMENT_PARAMETERS = FRAMEBUFFER_ATTACHMENT_PARAMETERS;
//# sourceMappingURL=framebuffer.js.map